// valid_certification.js
// React version of valid_certification.php logic
import React, { useState, useEffect } from 'react';
import axios from 'axios';

const CERT_COLORS = {
  medical: { primary: '#c90000', secondary: '#000000', accent: '#28a745' },
  hipaa: { primary: '#0056b3', secondary: '#495057', accent: '#17a2b8' },
  legal: { primary: '#8B4513', secondary: '#2c3e50', accent: '#dc3545' }
};

function safeDateFormat(dateStr, format = { year: 'numeric', month: 'long', day: 'numeric' }) {
  if (!dateStr) return '';
  let d = new Date(dateStr);
  if (isNaN(d)) return dateStr;
  return d.toLocaleDateString(undefined, format);
}

function getCertColor(type) {
  return CERT_COLORS[type] || CERT_COLORS['medical'];
}

export default function ValidCertification() {
  // Read cerf_id from query string on mount
  function getQueryParam(name) {
    return new URLSearchParams(window.location.search).get(name) || '';
  }
  const [cerfId, setCerfId] = useState(getQueryParam('cerf_id'));
  const [result, setResult] = useState(null);
  const [info, setInfo] = useState(null);
  const [lang, setLang] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  // Auto-trigger verification if cerf_id is present in query string
  useEffect(() => {
    if (!cerfId) return;
    setLoading(true);
    setError('');
    axios.get('/api/certificates')
      .then(res => {
        const cert = res.data.find(
          c => String(c.cerf_id) === String(cerfId) && c.is_active === 1
        );
        if (cert) {
          setResult({ certificate: cert });
        } else {
          setResult(null);
          setError('No active certificate found for this ID.');
        }
      })
      .catch(() => {
        setResult(null);
        setError('No active certificate found for this ID.');
      })
      .finally(() => setLoading(false));
  }, [cerfId]);

  // Expiration logic
  let isExpired = false;
  let expirationMessage = '';
  let daysUntilExpiry = 0;
  let certType = result?.certificate?.certificate_type || '';
  let colors = getCertColor(certType);
  if (result?.certificate?.date_created) {
    const issueDate = new Date(result.certificate.date_created);
    let expiryYears = certType === 'hipaa' ? 1 : 2;
    let expiryDate = new Date(issueDate);
    expiryDate.setFullYear(issueDate.getFullYear() + expiryYears);
    const now = new Date();
    if (now > expiryDate) {
      isExpired = true;
      const days = Math.floor((now - expiryDate) / (1000 * 60 * 60 * 24));
      expirationMessage = `This certificate expired ${days} days ago on ${safeDateFormat(expiryDate)}`;
    } else {
      daysUntilExpiry = Math.floor((expiryDate - now) / (1000 * 60 * 60 * 24));
      if (daysUntilExpiry <= 90) {
        expirationMessage = `This certificate will expire in ${daysUntilExpiry} days on ${safeDateFormat(expiryDate)}`;
      }
    }
  }

  return (
    <div className="container mt-4">
      <nav className="navbar navbar-expand-lg" style={{background: 'linear-gradient(135deg, #2c3e50 0%, #34495e 100%)', boxShadow: '0 2px 10px rgba(0,0,0,0.1)'}}>
        <div className="container-fluid px-3">
          <a className="navbar-brand d-flex align-items-center" href="https://makeitservices.org/">
            <i className="fas fa-arrow-left me-2 text-light"></i>
            <span className="text-light fw-semibold">Certificate Verification</span>
          </a>
          <form className="d-flex ms-auto mt-3 mt-lg-0" role="search" onSubmit={e => {e.preventDefault(); setCerfId(e.target.cerf_id.value);}}>
            <div className="input-group">
              <input className="form-control" type="search" placeholder="Enter Certificate ID" name="cerf_id" aria-label="Search" required />
              <button className="btn btn-outline-light search-btn" type="submit">
                <i className="fas fa-search"></i>
                <span className="d-none d-sm-inline ms-2">Search</span>
              </button>
            </div>
          </form>
        </div>
      </nav>
      <div className="container-fluid p-0">
        {error && (
          <div className="alert alert-danger rounded-0 border-0 text-center">
            <i className="fas fa-exclamation-circle fa-2x mb-2"></i>
            <h4>{error}</h4>
            <p className="mb-0">Please check the certificate ID and try again.</p>
          </div>
        )}
        {cerfId && result && isExpired && (
          <div className="alert alert-danger mx-2 mx-md-3 mb-3 border-0 shadow" style={{borderLeft: '5px solid #dc3545'}}>
            <div className="d-flex align-items-start">
              <i className="fas fa-exclamation-triangle fa-2x text-danger me-3 mt-1 animated-pulse"></i>
              <div className="flex-grow-1">
                <h5 className="alert-heading mb-2 fw-bold">⚠️ Certificate Expired</h5>
                <p className="mb-2 small">{expirationMessage}</p>
                <hr className="my-2" />
                <div className="small">
                  <strong>Action Required:</strong> This person needs to recycle or refresh their {certType} certification course.<br />
                  <a href="mailto:training@makeitservices.org" className="text-decoration-none">Contact training@makeitservices.org</a> for renewal.
                </div>
              </div>
            </div>
          </div>
        )}
        {cerfId && result && !isExpired && daysUntilExpiry <= 90 && daysUntilExpiry > 0 && (
          <div className="alert alert-warning mx-2 mx-md-3 mb-3 border-0 shadow" style={{borderLeft: '5px solid #ffc107'}}>
            <div className="d-flex align-items-start">
              <i className="fas fa-clock fa-2x text-warning me-3 mt-1 animated-pulse"></i>
              <div className="flex-grow-1">
                <h5 className="alert-heading mb-2 fw-bold">📅 Certificate Expiring Soon</h5>
                <p className="mb-2 small">{expirationMessage}</p>
                <hr className="my-2" />
                <div className="small">
                  <strong>Renewal Recommended:</strong> Please consider renewing this {certType} certification soon.<br />
                  <a href="mailto:training@makeitservices.org" className="text-decoration-none">Contact training@makeitservices.org</a> for renewal.
                </div>
              </div>
            </div>
          </div>
        )}
        {/* Certificate Card - Only show if NOT expired */}
        {cerfId && result && !isExpired && (
          <section className="certificate-card bg-white mx-2 mx-md-3 shadow border-0 rounded-3 overflow-hidden">
            <header className="certificate-header text-center py-4" style={{background: `linear-gradient(135deg, ${colors.primary} 0%, ${colors.secondary} 100%)`}}>
              <img className="profile-image rounded-circle mb-3 border border-white border-4" src={result.user?.profile_image || '/default-avatar.png'} alt="Profile" style={{width: '120px', height: '120px', objectFit: 'cover'}} />
              <div className="d-flex justify-content-center gap-2 flex-wrap mb-3">
                <span className="badge bg-white text-dark px-3 py-2 shadow-sm"><i className="fas fa-user me-1"></i>User: {result.certificate?.user_id}</span>
                <span className="badge bg-light text-dark px-3 py-2 shadow-sm"><i className="fas fa-certificate me-1"></i>Cert: {result.certificate?.cerf_id}</span>
              </div>
              <h3 className="text-white mb-0 fw-bold"><i className="fas fa-graduation-cap me-2"></i>MAKE-IT LEARNING CENTER</h3>
            </header>
            <div className="certificate-body p-4">
              {/* Certificate Type Specific Content */}
              {certType === 'medical' && (
                <div className="text-center mb-4">
                  <h2 className="h4 mb-2" style={{color: colors.primary}}>Medical Interpreter</h2>
                  <h1 className="h3 mb-3 fw-bold" style={{color: colors.primary}}>{result.certificate?.fullname}</h1>
                  <p className="text-muted">has successfully completed the 60-hour Medical Interpreter Training course and demonstrated proficiency in professional medical interpretation</p>
                  <p className="text-dark fw-semibold">{result.certificate?.language_from} ↔ {result.certificate?.language_to}</p>
                  <div className="d-flex justify-content-center gap-2 flex-wrap mt-3">
                    <span className="badge" style={{backgroundColor: colors.primary}}>LPT</span>
                    <span className="badge" style={{backgroundColor: colors.primary}}>MTC</span>
                    <span className="badge" style={{backgroundColor: colors.primary}}>MOC</span>
                    <span className="badge" style={{backgroundColor: colors.accent}}>HIPAA</span>
                  </div>
                </div>
              )}
              {certType === 'hipaa' && (
                <div className="text-center mb-4">
                  <h2 className="h4 mb-2" style={{color: colors.primary}}>HIPAA Privacy & Security</h2>
                  <h1 className="h3 mb-3 fw-bold" style={{color: colors.primary}}>{result.certificate?.fullname}</h1>
                  <p className="text-muted">This certificate confirms that the individual has successfully completed HIPAA compliance training and understands the requirements for protecting patient health information.</p>
                  <div className="d-flex justify-content-center gap-2 flex-wrap mt-3">
                    <span className="badge" style={{backgroundColor: colors.primary}}>Privacy Rule</span>
                    <span className="badge" style={{backgroundColor: colors.accent}}>Security Rule</span>
                    <span className="badge" style={{backgroundColor: colors.accent}}>PHI Protection</span>
                  </div>
                </div>
              )}
              {certType === 'legal' && (
                <div className="text-center mb-4">
                  <h2 className="h4 mb-2" style={{color: colors.primary}}>Legal Interpreter</h2>
                  <h1 className="h3 mb-3 fw-bold" style={{color: colors.primary}}>{result.certificate?.fullname}</h1>
                  <p className="text-muted">This certificate confirms that the individual has successfully completed the 40-hour Legal Interpreter certification program and is qualified to provide legal interpretation services.</p>
                  <p className="text-dark fw-semibold">{result.certificate?.language_from} ↔ {result.certificate?.language_to}</p>
                  <div className="d-flex justify-content-center gap-2 flex-wrap mt-3">
                    <span className="badge" style={{backgroundColor: colors.primary}}>Legal Terms</span>
                    <span className="badge" style={{backgroundColor: colors.accent}}>Court Procedures</span>
                    <span className="badge" style={{backgroundColor: colors.accent}}>Ethics</span>
                  </div>
                </div>
              )}
              {/* Award Date */}
              <div className="text-center py-3 border-top">
                <p className="text-success fw-bold mb-2">{result.certificate?.certificate_text}</p>
                <p className="text-muted mb-0"><i className="fas fa-calendar-alt me-2"></i>Awarded on {safeDateFormat(result.certificate?.date_validity)}</p>
              </div>
              {/* Contact Information */}
              <div className="text-center pt-3 border-top">
                <div className="row g-2 justify-content-center">
                  <div className="col-12 mb-2">
                    <a href="https://makeitservices.org/" className="btn btn-modern w-100">
                      <i className="fas fa-globe me-2"></i>Website
                    </a>
                  </div>
                  <div className="col-12">
                    <a href="mailto:training@makeitservices.org" className="btn btn-modern w-100">
                      <i className="fas fa-envelope me-2"></i>Contact
                    </a>
                  </div>
                </div>
              </div>
            </div>
          </section>
        )}
      </div>
      {/* Responsive styles for mobile */}
      <style>{`
@media (max-width: 576px) {
  .container.mt-4 {
    margin-top: 8px !important;
    padding-left: 2px !important;
    padding-right: 2px !important;
  }
  .navbar {
    padding: 10px 0 !important;
    border-radius: 0 !important;
    box-shadow: none !important;
  }
  .input-group .search-btn {
    min-width: 48px !important;
    height: 40px !important;
    font-size: 1.1em !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    padding: 0 12px !important;
    border-radius: 8px !important;
    background: #34495e !important;
    color: #fff !important;
    border: none !important;
    box-shadow: 0 1px 4px rgba(0,0,0,0.04);
  }
  .input-group .search-btn .fa-search {
    font-size: 1.2em !important;
  }
  .input-group .search-btn span {
    display: inline !important;
    font-size: 1em !important;
    margin-left: 6px !important;
  }
  .input-group {
    flex-wrap: nowrap !important;
  }
  .certificate-card {
    margin-left: 0 !important;
    margin-right: 0 !important;
    border-radius: 0 !important;
    box-shadow: none !important;
    padding: 0 !important;
  }
  .certificate-header {
    padding: 18px 6px !important;
    background: linear-gradient(135deg, #232f3e 0%, #34495e 100%) !important;
    border-bottom: 2px solid #eee !important;
  }
  .profile-image {
    width: 80px !important;
    height: 80px !important;
    margin-bottom: 10px !important;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    border: 3px solid #fff !important;
  }
  .alert {
    font-size: 1em !important;
    padding: 14px !important;
    border-radius: 10px !important;
    margin-top: 12px !important;
    box-shadow: 0 2px 8px rgba(220,53,69,0.08);
  }
  .certificate-body {
    padding: 12px !important;
    background: #f9f9f9 !important;
    border-radius: 0 0 12px 12px !important;
  }
  .h3, .h4 {
    font-size: 1.18rem !important;
    margin-bottom: 10px !important;
    font-family: 'Playfair Display', serif !important;
  }
  .badge {
    font-size: 1em !important;
    padding: 7px 14px !important;
    margin-bottom: 8px !important;
    border-radius: 8px !important;
    box-shadow: 0 1px 4px rgba(0,0,0,0.04);
    font-weight: 600 !important;
    letter-spacing: 0.3px !important;
  }
  .row.g-2 {
    gap: 0.5rem !important;
  }
  .certificate-header h3 {
    font-size: 1.08rem !important;
    margin-bottom: 6px !important;
    font-family: 'Playfair Display', serif !important;
    letter-spacing: 0.7px !important;
  }
  .certificate-header .badge {
    margin-bottom: 4px !important;
  }
  .certificate-body p {
    font-size: 1em !important;
    margin-bottom: 8px !important;
    color: #333 !important;
  }
  .btn-modern {
    background: #fff !important;
    color: #0056b3 !important;
    border: 2px solid #0056b3 !important;
    font-weight: 700 !important;
    font-size: 1.08em !important;
    border-radius: 10px !important;
    box-shadow: 0 2px 8px rgba(0,86,179,0.08);
    transition: background 0.2s, color 0.2s, border 0.2s;
    padding: 10px 0 !important;
    margin-bottom: 8px !important;
    letter-spacing: 0.2px !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
  }
  .btn-modern:hover, .btn-modern:focus {
    background: #0056b3 !important;
    color: #fff !important;
    border-color: #0056b3 !important;
  }
}

/* General tweaks for desktop and mobile */
.btn-modern {
  background: #fff;
  color: #0056b3;
  border: 2px solid #0056b3;
  font-weight: 700;
  font-size: 1.08em;
  border-radius: 10px;
  box-shadow: 0 2px 8px rgba(0,86,179,0.08);
  transition: background 0.2s, color 0.2s, border 0.2s;
  padding: 10px 0;
  margin-bottom: 8px;
  letter-spacing: 0.2px;
  display: flex;
  align-items: center;
  justify-content: center;
}
.btn-modern:hover, .btn-modern:focus {
  background: #0056b3;
  color: #fff;
  border-color: #0056b3;
}
`}</style>
    </div>
  );
}
