import React, { useEffect, useState } from 'react';
import { BASE_API } from '../utils/config';

export default function Teachers() {
  const [teachers, setTeachers] = useState([]);
  const [userImages, setUserImages] = useState({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    setLoading(true);
    fetch(`${BASE_API}/api/teachers`)
      .then(res => res.json())
      .then(async data => {
        // Filter teachers for professional or both, and visible === 1 (from teacher table)
        const filtered = (data.teachers || []).filter(t => (t.school_type === 'both' || t.school_type === 'professional') && t.visible === 1);
        setTeachers(filtered);
        // Fetch all users in one call
        const usersRes = await fetch(`${BASE_API}/api/users`);
        const users = await usersRes.json();
        // Build a map of user_id to user object for quick lookup
        const userMap = {};
        users.forEach(u => {
          userMap[u.id_user] = u;
        });
        // Build image map for visible teachers
        const imgMap = {};
        filtered.forEach(t => {
          const u = userMap[t.user_id];
          imgMap[t.user_id] = u && u.img ? u.img : null;
        });
        setUserImages(imgMap);
        setLoading(false);
      })
      .catch(err => {
        setError('Failed to fetch teachers');
        setLoading(false);
      });
  }, []);

  return (
    <div className="container my-5" style={{maxWidth: 900}}>
      <h1 className="fw-bold mb-4 text-center" style={{ letterSpacing: 1 }}>Teachers</h1>
      {loading && <div className="text-center">Loading...</div>}
      {error && <div className="text-danger text-center">{error}</div>}
      <div className="row g-4">
        {teachers.map(t => {
          // Compose profile image URL if img is set in users table
          let imgUrl = null;
          if (userImages[t.user_id]) {
            // If already a full URL, use as is
            if (userImages[t.user_id].startsWith('http')) {
              imgUrl = userImages[t.user_id];
            } else {
              // Otherwise, serve from backend/public/profile
              imgUrl = `${BASE_API}/profile/${userImages[t.user_id]}`;
            }
          }
          return (
            <div className="col-md-6 col-lg-4" key={t.id}>
              <div className="card h-100 shadow-sm">
                <div className="card-body text-center">
                  <img
                    src={imgUrl || '/assets/img/default-avatar.png'}
                    alt="Profile"
                    style={{ width: 90, height: 90, borderRadius: '50%', objectFit: 'cover', marginBottom: 16, border: '2px solid #eee' }}
                  />
                  <h5 className="card-title mb-1">{t.prenom} {t.nom}</h5>
                  <div className="text-muted" style={{fontSize: 14}}>{t.email}</div>
                  <div className="mt-2"><b>Role:</b> {t.subject}</div>
                  <div className="mt-1" style={{fontSize: 13, color: '#555'}}>{t.bio}</div>
                </div>
                <div className="card-footer text-center" style={{fontSize: 13, background: '#f8f9fa'}}>
                  {t.school_type === 'both' ? 'Both Schools' : 'Professional School'}
                </div>
              </div>
            </div>
          );
        })}
        {teachers.length === 0 && !loading && (
          <div className="text-center text-muted">No professional teachers found.</div>
        )}
      </div>
    </div>
  );
}
