import React, { useEffect, useState } from 'react';
import axios from 'axios';
import CourseCard from './CourseCard';
import './CourseCardPopup.css';

export default function Courses() {
  const [courses, setCourses] = useState([]);
  const [enrolledCounts, setEnrolledCounts] = useState({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    setLoading(true);
    Promise.all([
      axios.get('/api/top_courses'),
      axios.get('/api/enrolled_course_count')
    ])
    .then(([coursesRes, enrolledRes]) => {
      const coursesData = coursesRes.data.courses || [];
      setCourses(coursesData);
      
      console.log('=== FRONTEND DEBUG ===');
      console.log('Courses Data:', coursesData);
      console.log('Enrolled Response:', enrolledRes.data);
      
      // Convert enrolled_counts array to object keyed by course_id
      const countsMap = {};
      if (enrolledRes.data.enrolled_counts && Array.isArray(enrolledRes.data.enrolled_counts)) {
        enrolledRes.data.enrolled_counts.forEach(item => {
          countsMap[item.course_id] = item.num_users;
          console.log(`Mapped course_id ${item.course_id} -> ${item.num_users} users`);
        });
      }
      
      // Debug: Show what we got
      console.log('Courses:', coursesData);
      console.log('Enrolled Counts Map:', countsMap);
      console.log('Raw Enrolled Response:', enrolledRes.data.enrolled_counts);
      console.log('=== END DEBUG ===');
      
      setEnrolledCounts(countsMap);
      setLoading(false);
    })
    .catch(err => {
      setError('Failed to load courses.');
      setLoading(false);
    });
  }, []);

  return (
    <div className="container my-5" style={{maxWidth: 900}}>
      <h1 className="fw-bold mb-4 text-center" style={{ letterSpacing: 1 }}>Courses</h1>
      {loading && <p className="text-center">Loading courses...</p>}
      {error && <p className="text-center text-danger">{error}</p>}
      {!loading && !error && courses.length === 0 && (
        <p className="text-center">No public courses available.</p>
      )}
      <div className="row justify-content-center">
        {courses.map(course => (
          <div className="col-md-6 col-lg-4" key={course.course_id}>
            <CourseCard course={course} enrolledCount={enrolledCounts[course.course_id] || 0} />
          </div>
        ))}
      </div>
    </div>
  );
}
