import React, { useState } from 'react';
import { logAction } from './utils/actionLogger';
import { useNavigate } from 'react-router-dom';
import { BASE_URL, BASE_API } from './utils/config';
import { recordVisit } from './utils/visit-tracker';

const formStyle = {
  maxWidth: '400px',
  margin: '60px auto',
  padding: '32px',
  borderRadius: '16px',
  boxShadow: '0 4px 24px rgba(0,0,0,0.12)',
  background: '#fff',
  display: 'flex',
  flexDirection: 'column',
  gap: '18px',
  animation: 'fadeIn 1s',
};

const inputStyle = {
  padding: '12px',
  borderRadius: '8px',
  border: '1px solid #ccc',
  fontSize: '1rem',
};

const buttonStyle = {
  padding: '14px',
  borderRadius: '8px',
  border: 'none',
  background: 'linear-gradient(90deg, #a259e6 0%, #f75c7a 100%)',
  color: '#fff',
  fontWeight: 'bold',
  fontSize: '1.1rem',
  cursor: 'pointer',
  marginTop: '12px',
  transition: 'background 0.2s',
};

const errorStyle = {
  color: '#f75c7a',
  fontWeight: 'bold',
  textAlign: 'center',
};

function Login() {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [error, setError] = useState('');
  const [showReset, setShowReset] = useState(false);
  const [resetId, setResetId] = useState('');
  const [resetEmail, setResetEmail] = useState('');
  const [resetMsg, setResetMsg] = useState('');
  const [resetError, setResetError] = useState('');
  const navigate = useNavigate();

  React.useEffect(() => {
    recordVisit();
  }, []);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError('');
    try {
      const res = await fetch(`${BASE_API}/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, password })
      });
      const data = await res.json();
      if (res.ok) {
        localStorage.setItem('id_user', data.id_user);
        localStorage.setItem('userId', data.id_user); // Ensure userId is set for dashboard
        localStorage.setItem('role', data.role);
        localStorage.setItem('email', data.email || email);
        // Set enligne=1 for this user
        fetch(`${BASE_API}/api/set-enligne`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ id_user: data.id_user, enligne: 1 })
        });
        // Log login action
        logAction({
          actor_id: data.id_user,
          actor_role: data.role,
          actor_email: data.email || email,
          action_type: 'login',
          action_description: 'User logged in',
          page: '/login',
        });
        if (data.role === 'Administrator') {
          navigate('/Company/Administrator/index');
        } else {
          navigate('/Company/10000/index');
        }
      } else {
        setError(data.error || 'Login failed');
      }
    } catch (err) {
      setError('Server error');
    }
  };

  const handleReset = async (e) => {
    e.preventDefault();
    setResetError('');
    setResetMsg('');
    try {
  const res = await fetch(`${BASE_API}/reset-password`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id_user: resetId, email: resetEmail })
      });
      const data = await res.json();
      if (res.ok) {
        setResetMsg('Magic link sent to your email!');
      } else {
        setResetError(data.error || 'Reset failed');
      }
    } catch (err) {
      setResetError('Server error');
    }
  };

  return (
    <div>
      <div style={{textAlign:'center', marginBottom:'8px'}}>
        <span style={{color:'#a259e6', cursor:'pointer', fontWeight:'bold'}} onClick={() => window.location.href = BASE_URL}>← Back to main page</span>
      </div>
      {!showReset ? (
        <form style={formStyle} onSubmit={handleSubmit}>
          <h2 style={{textAlign:'center', color:'#a259e6'}}>Login</h2>
          <input style={inputStyle} type="email" placeholder="Email" value={email} onChange={e => setEmail(e.target.value.toLowerCase())} required />
          <input style={inputStyle} type="password" placeholder="Password" value={password} onChange={e => setPassword(e.target.value)} required />
          <button style={buttonStyle} type="submit">Login</button>
          {error && <div style={errorStyle}>{error}</div>}
          <div style={{textAlign:'center', marginTop:'8px'}}>
            Don't have an account? <span style={{color:'#a259e6', cursor:'pointer'}} onClick={()=>navigate('/register')}>Register</span>
          </div>
          <div style={{textAlign:'center', marginTop:'8px'}}>
            <span style={{color:'#f75c7a', cursor:'pointer', textDecoration:'underline'}} onClick={()=>setShowReset(true)}>Forgot password?</span>
          </div>
        </form>
      ) : (
        <form style={formStyle} onSubmit={handleReset}>
          <h2 style={{textAlign:'center', color:'#a259e6'}}>Reset Password</h2>
          <input style={inputStyle} type="number" placeholder="User ID" value={resetId} onChange={e => setResetId(e.target.value)} required />
          <input style={inputStyle} type="email" placeholder="Email" value={resetEmail} onChange={e => setResetEmail(e.target.value)} required />
          <button style={buttonStyle} type="submit">Send Magic Link</button>
          {resetError && <div style={errorStyle}>{resetError}</div>}
          {resetMsg && <div style={{color:'#27ae60', fontWeight:'bold', textAlign:'center'}}>{resetMsg}</div>}
          <div style={{textAlign:'center', marginTop:'8px'}}>
            <span style={{color:'#a259e6', cursor:'pointer'}} onClick={()=>setShowReset(false)}>Back to Login</span>
          </div>
        </form>
      )}
    </div>
  );
}

export default Login;
