import React, { useEffect, useState, useRef } from 'react';
import { Box, Typography, TextField, Button, Avatar, CircularProgress, MenuItem, IconButton } from '@mui/material';
import PhotoCamera from '@mui/icons-material/PhotoCamera';
import axios from 'axios';
import { BASE_API } from '../utils/config';

const genderOptions = [
  { value: 'M', label: 'Male' },
  { value: 'F', label: 'Female' },
  { value: 'O', label: 'Other' },
];


function ProfilePage({ currentUser, onAvatarChange }) {
  // Backend URL for serving profile images
  const backendUrl = BASE_API; // Use config value

  const [profile, setProfile] = useState(null);
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState('');
  const [uploading, setUploading] = useState(false);
  const fileInputRef = useRef();

  const [previewImg, setPreviewImg] = useState(null);


  useEffect(() => {
    if (currentUser?.id) {
      fetchProfile();
    }
    // eslint-disable-next-line
  }, [currentUser]);

  const fetchProfile = async () => {
    setLoading(true);
    setError('');
    try {
      const res = await axios.get(`${BASE_API}/api/users/by-iduser/${currentUser.id}`);
      setProfile(res.data.user);
      // Notify parent of avatar change if available
      if (onAvatarChange && res.data.user && res.data.user.img) {
        const imgPath = res.data.user.img.startsWith('/profile/') ? BASE_API + res.data.user.img : res.data.user.img;
        onAvatarChange(imgPath);
      }
    } catch (err) {
      setError('Failed to load profile.');
    }
    setLoading(false);
  };

  const handleChange = (e) => {
    setProfile({ ...profile, [e.target.name]: e.target.value });
  };

  const handleImageChange = async (e) => {
    const file = e.target.files[0];
    if (!file) return;
    setUploading(true);
    setError('');
    // Preview image immediately
    const reader = new FileReader();
    reader.onloadend = () => {
      setPreviewImg(reader.result);
    };
    reader.readAsDataURL(file);
    try {
      // Compose filename: prenom_nom.ext
      const ext = file.name.split('.').pop();
      const prenom = (profile.prenom || 'user').replace(/\s+/g, '_');
      const nom = (profile.nom || 'profile').replace(/\s+/g, '_');
      const filename = `${prenom}_${nom}.${ext}`;
      const formData = new FormData();
      formData.append('image', file, filename);
      formData.append('id_user', profile.id_user || profile.id || '');
      // Upload endpoint should save to backend/public/profile
      const res = await axios.post(`${BASE_API}/api/users/upload-profile-image`, formData, {
        headers: { 'Content-Type': 'multipart/form-data' },
      });
      // Update profile image path in state
      setProfile({ ...profile, img: res.data.imagePath });
      setError('Profile image updated!');
      setPreviewImg(null); // Use server image after upload
      // Notify parent of avatar change
      if (onAvatarChange && res.data.imagePath) {
        const imgPath = res.data.imagePath.startsWith('/profile/') ? BASE_API + res.data.imagePath : res.data.imagePath;
        onAvatarChange(imgPath);
      }
    } catch (err) {
      setError('Failed to upload image.');
    }
    setUploading(false);
  };

  const handleSave = async () => {
    setSaving(true);
    setError('');
    try {
      await axios.put(`${BASE_API}/api/users/by-iduser/${currentUser.id}`, profile);
      // Log action to backend
      await axios.post(`${BASE_API}/api/log_action`, {
        actor_id: currentUser.id,
        actor_role: currentUser.role,
        actor_email: profile.email,
        action_type: 'update_profile',
        action_description: 'Updated profile information',
        page: window.location.pathname
      });
      setError('Profile updated successfully!');
      // Refresh profile after update
      await fetchProfile();
    } catch (err) {
      setError('Failed to update profile.');
    }
    setSaving(false);
  };

  if (loading || !profile) return <Box sx={{ p: 4, textAlign: 'center' }}><CircularProgress /></Box>;

  // Compute avatar src with backend URL if needed
  const avatarSrc = profile.img
    ? profile.img.startsWith('/profile/')
      ? backendUrl + profile.img
      : profile.img
    : undefined;

  return (
    <Box sx={{ maxWidth: 500, mx: 'auto', p: 3 }}>
      <Typography variant="h4" gutterBottom>My Profile</Typography>
      <Box sx={{ display: 'flex', flexDirection: 'column', alignItems: 'center', mb: 2 }}>
        <Avatar src={avatarSrc} sx={{ width: 80, height: 80, mb: 2 }} />
        <input
          type="file"
          accept="image/*"
          style={{ display: 'none' }}
          ref={fileInputRef}
          onChange={handleImageChange}
        />
        <IconButton color="primary" component="span" onClick={() => fileInputRef.current.click()} disabled={uploading}>
          <PhotoCamera />
        </IconButton>
        <Typography variant="caption" sx={{ mb: 1 }}>Upload profile image</Typography>
      </Box>
      <TextField label="First Name" name="prenom" value={profile.prenom || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
      <TextField label="Last Name" name="nom" value={profile.nom || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
      <TextField label="Email" name="email" value={profile.email || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} disabled />
      <TextField label="Phone" name="tel" value={profile.tel || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
      <TextField label="Address" name="adresse" value={profile.adresse || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
      <TextField label="Country" name="country" value={profile.country || ''} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
      <TextField label="Gender" name="sexe" value={profile.sexe || ''} onChange={handleChange} select fullWidth sx={{ mb: 2 }}>
        {genderOptions.map(opt => <MenuItem key={opt.value} value={opt.value}>{opt.label}</MenuItem>)}
      </TextField>
      <TextField label="About" name="about" value={profile.about || ''} onChange={handleChange} fullWidth multiline rows={3} sx={{ mb: 2 }} />
      {error && <Typography color={error.includes('success') || error.includes('updated') ? 'green' : 'error'} sx={{ mb: 2 }}>{error}</Typography>}
      <Button variant="contained" color="primary" onClick={handleSave} disabled={saving}>{saving ? 'Saving...' : 'Save Changes'}</Button>
    </Box>
  );
}

export default ProfilePage;
