import React, { useEffect, useState, useRef } from 'react';
import html2pdf from 'html2pdf.js';
import SignatureCanvas from 'react-signature-canvas';
import { BASE_API } from '../../utils/config';

// Utility to find all signature/name/date fields in HTML
function findFields(html) {
  const div = document.createElement('div');
  div.innerHTML = html;
  const signatureSpans = div.querySelectorAll('span');
  const fields = [];
  signatureSpans.forEach(span => {
    if (span.textContent === '[Signature Here]' || span.textContent === '[Name Here]' || span.textContent === '[Date Here]') {
      let type = 'name';
      if (span.textContent.includes('Signature')) type = 'signature';
      if (span.textContent.includes('Date')) type = 'date';
      fields.push({
        type,
        html: span.outerHTML,
        index: fields.length,
      });
    }
  });
  return fields;
}

const SignDocument = ({ docId, token }) => {
  const [doc, setDoc] = useState(null);
  const [fields, setFields] = useState([]);
  const [currentField, setCurrentField] = useState(0);
  const [fieldValues, setFieldValues] = useState({});
  const [signedHtml, setSignedHtml] = useState('');
  const [showInput, setShowInput] = useState(false);
  const [inputValue, setInputValue] = useState('');
  const [done, setDone] = useState(false);
  const [showPdf, setShowPdf] = useState(false);
  const [showSignaturePad, setShowSignaturePad] = useState(false);
  const [signatureData, setSignatureData] = useState('');
  const [signatureMode, setSignatureMode] = useState('choose');
  const sigPadRef = useRef(null);

  useEffect(() => {
    fetch(`${BASE_API}/sign/${docId}?token=${token}`)
      .then(res => res.json())
      .then(data => {
        setDoc(data);
        let html = data.document_content;
        try {
          html = typeof html === 'string' ? html.replace(/\\u003C/g, '<').replace(/\\u003E/g, '>') : html;
        } catch {}
        setFields(findFields(html));
        setSignedHtml(html);
      });
  }, [docId, token]);

 // Highlight current field in preview and make fields clickable
 useEffect(() => {
   if (!fields.length) return;
   let html = doc ? doc.document_content : '';
   fields.forEach((field, idx) => {
     if (fieldValues[idx]) {
       html = html.replace(field.html, `<span style="font-family: 'Pacifico', cursive; color: #1565c0; font-size: 1.2em;">${fieldValues[idx]}</span>`);
     } else {
       html = html.replace(field.html, `<span style="border:2px solid #f75c7a; background:#fffbe6; cursor:pointer;" data-field-type="${field.type}" data-field-idx="${idx}">${field.type === 'signature' ? '[Signature Here]' : field.type === 'date' ? '[Date Here]' : '[Name Here]'}</span>`);
     }
   });
   setSignedHtml(html);
 }, [fieldValues, fields, doc]);

  const handleNext = () => {
    setShowInput(false);
    setInputValue('');
    if (currentField < fields.length - 1) {
      setCurrentField(currentField + 1);
    } else {
      setDone(true);
    }
  };

  const handleSaveField = (idx, value) => {
    setFieldValues({ ...fieldValues, [idx]: value });
    setShowInput(false);
    setInputValue('');
  };

  const handleFieldClick = (e) => {
    const target = e.target;
    if (target.hasAttribute('data-field-type')) {
      const type = target.getAttribute('data-field-type');
      const idx = parseInt(target.getAttribute('data-field-idx'));
      if (type === 'date') setShowInput({type:'date', idx});
      else if (type === 'name') setShowInput({type:'name', idx});
      else if (type === 'signature') { setShowInput({type:'signature', idx}); setSignatureMode('choose'); }
    }
  };

  const handleDownloadPDF = () => {
    const element = document.createElement('div');
    element.innerHTML = signedHtml;
    html2pdf().from(element).save(`signed_document_${docId}.pdf`);
  };

  const handleSubmit = async () => {
    // Send signedHtml to backend
    await fetch(`${BASE_API}/sign/${docId}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ signed_content: signedHtml, signer_email: doc.email })
    });
    alert('Signed document submitted!');
  };

  if (!doc) return <div>Loading...</div>;

  return (
    <div style={{padding:'32px'}}>
      <h2>Sign Document</h2>
      <div style={{marginBottom:'16px', fontWeight:'bold', fontSize:'1.1em', background:'#eaf6ff', borderRadius:'8px', padding:'12px 18px', boxShadow:'0 2px 8px rgba(21,101,192,0.08)'}}>
        <div>Document Name: <span style={{color:'#1565c0'}}>{doc.name ? doc.name : `Document #${doc.id}`}</span></div>
        <div>Company Code: <span style={{color:'#27ae60'}}>{doc.company_code || 'N/A'}</span></div>
        <div>Recipient Email: <span style={{color:'#f75c7a'}}>{doc.email}</span></div>
      </div>
      <div style={{marginBottom:'24px'}}>
        <button onClick={handleDownloadPDF} style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer', marginRight:'8px'}}>Download PDF</button>
        <button onClick={()=>setShowPdf(!showPdf)} style={{background:'#27ae60', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer', marginRight:'8px'}}>{showPdf ? 'Hide PDF Preview' : 'Show PDF Preview'}</button>
        <button onClick={()=>setShowSignaturePad(!showSignaturePad)} style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>{showSignaturePad ? 'Hide Signature Pad' : 'Draw Signature'}</button>
        {done ? (
          <button onClick={handleSubmit} style={{background:'#27ae60', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Submit Signed Document</button>
        ) : (
          <button onClick={handleNext} style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Next Field</button>
        )}
      </div>
      {showPdf && (
        <div style={{marginBottom:'24px'}}>
          <iframe
            src={`${BASE_API}/documents/pdf-preview/${docId}`}
            title="PDF Preview"
            width="100%"
            height="500px"
            style={{border:'1px solid #1565c0', borderRadius:'8px'}}
          />
        </div>
      )}
      <div style={{background:'#fff', borderRadius:'8px', boxShadow:'0 2px 8px rgba(0,0,0,0.08)', padding:'16px', minHeight:'300px'}}>
        <div
          contentEditable={true}
          suppressContentEditableWarning={true}
          style={{outline:'none'}}
          dangerouslySetInnerHTML={{__html: signedHtml}}
          onInput={e => setSignedHtml(e.currentTarget.innerHTML)}
          onClick={handleFieldClick}
        />
      </div>
      {showInput && showInput.type === 'date' && (
        <div style={{marginTop:'24px', background:'#fffbe6', border:'2px solid #a259e6', borderRadius:'8px', padding:'16px'}}>
          <input type="date" value={inputValue} onChange={e=>setInputValue(e.target.value)} style={{padding:'8px', fontSize:'1.1em', borderRadius:'6px', border:'1px solid #ccc'}} />
          <button onClick={()=>handleSaveField(showInput.idx, inputValue ? new Date(inputValue).toLocaleDateString() : '')} style={{marginLeft:'12px', background:'#a259e6', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Save</button>
        </div>
      )}
      {showInput && showInput.type === 'name' && (
        <div style={{marginTop:'24px', background:'#fffbe6', border:'2px solid #27ae60', borderRadius:'8px', padding:'16px', position:'relative'}}>
          <span style={{position:'absolute', top:'12px', right:'18px', fontSize:'2em', cursor:'pointer'}} onClick={()=>setShowInput(false)}>&#10006;</span>
          <h4>Enter Name</h4>
          <input type="text" value={inputValue} onChange={e=>setInputValue(e.target.value)} style={{padding:'8px', fontSize:'1.1em', borderRadius:'6px', border:'1px solid #ccc'}} />
          <button onClick={()=>handleSaveField(showInput.idx, inputValue)} style={{marginLeft:'12px', background:'#27ae60', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Save</button>
        </div>
      )}
      {showInput && showInput.type === 'signature' && (
        <div style={{marginTop:'24px', background:'#fffbe6', border:'2px solid #1565c0', borderRadius:'8px', padding:'16px', position:'relative'}}>
          <span style={{position:'absolute', top:'12px', right:'18px', fontSize:'2em', cursor:'pointer'}} onClick={()=>{setShowInput(false); setShowSignaturePad(false); setSignatureMode('choose');}}>&#10006;</span>
          {signatureMode === 'choose' ? (
            <>
              <h4>Choose Signature Style</h4>
              <div style={{marginBottom:'16px'}}>
                <button style={{marginRight:'12px', background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>setSignatureMode('typed')}>Typed</button>
                <button style={{background:'#27ae60', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>setSignatureMode('draw')}>Draw</button>
              </div>
            </>
          ) : signatureMode === 'typed' ? (
            <div style={{marginTop:'16px'}}>
              <h4>Type Your Signature</h4>
              <input type="text" value={inputValue} onChange={e=>setInputValue(e.target.value)} placeholder="Signature" style={{padding:'8px', fontSize:'1.1em', borderRadius:'6px', border:'1px solid #ccc'}} />
              <div style={{marginTop:'12px'}}>
                <span style={{
                  fontFamily: signatureMode === 'pacifico' ? 'Pacifico, cursive' : signatureMode === 'serif' ? 'serif' : signatureMode === 'monospace' ? 'monospace' : signatureMode === 'bold' ? 'inherit' : 'inherit',
                  fontWeight: signatureMode === 'bold' ? 'bold' : 'normal',
                  fontSize:'1.5em', color:'#1565c0', background:'#fff', padding:'6px 18px', borderRadius:'8px', marginRight:'12px'
                }}>{inputValue}</span>
                <span
                  title="Switch signature style"
                  style={{cursor:'pointer', fontSize:'1.7em', verticalAlign:'middle'}}
                  onClick={e => {
                    e.stopPropagation();
                    const styles = ['pacifico','serif','monospace','bold'];
                    const currentIdx = styles.indexOf(signatureMode);
                    setSignatureMode(styles[(currentIdx+1)%styles.length]);
                  }}
                >&#8635;</span>
              </div>
              <button style={{marginTop:'12px', background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>{
              let fontFamily = signatureMode === 'pacifico' ? 'Pacifico, cursive' : signatureMode === 'serif' ? 'serif' : signatureMode === 'monospace' ? 'monospace' : 'inherit';
              let fontWeight = signatureMode === 'bold' ? 'bold' : 'normal';
              handleSaveField(showInput.idx, `<span style=\"font-family:${fontFamily}; font-weight:${fontWeight}; font-size:1.5em; color:#1565c0; background:#fff; padding:6px 18px; border-radius:8px;\">${inputValue}</span>`);
              setShowInput(false);
              setSignatureMode('choose');
            }}>Save Typed Signature</button>
              <button style={{marginLeft:'12px', background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>setSignatureMode('choose')}>Back</button>
            </div>
          ) : (
            <div style={{marginTop:'16px'}}>
              <h4>Draw Your Signature</h4>
              <SignatureCanvas
                ref={sigPadRef}
                penColor="#1565c0"
                canvasProps={{width: 400, height: 120, className: 'sigCanvas'}}
              />
              <button style={{marginLeft:'12px', background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>{
                handleSaveField(showInput.idx, `<img src="${sigPadRef.current.toDataURL('image/png')}" alt="Signature" style="border:2px solid #1565c0; border-radius:8px; background:#fff; padding:8px;" />`);
                setShowSignaturePad(false);
                setShowInput(false);
                setSignatureMode('choose');
              }}>Save Drawn Signature</button>
              <button style={{marginLeft:'12px', background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>sigPadRef.current.clear()}>Clear</button>
              <button style={{marginLeft:'12px', background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>setSignatureMode('choose')}>Back</button>
            </div>
          )}
        </div>
      )}
      {signatureData && (
        <div style={{marginTop:'24px'}}>
          <h4>Signature Image Saved:</h4>
          <img src={signatureData} alt="Signature" style={{border:'2px solid #1565c0', borderRadius:'8px', background:'#fff', padding:'8px'}} />
        </div>
      )}
      {done && <div style={{marginTop:'24px', color:'#27ae60', fontWeight:'bold'}}>All fields completed! You can now submit the signed document.</div>}
    </div>
  );
};

export default SignDocument;
