import React, { useEffect, useState } from 'react';
import { Box, Typography, Chip, Card, CardContent, CircularProgress } from '@mui/material';
import axios from 'axios';

function AnnouncementPage({ currentUser }) {
  const [announcements, setAnnouncements] = useState([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    fetchAnnouncements();
  }, []);

  const fetchAnnouncements = async () => {
    setLoading(true);
    try {
      const res = await axios.get('/api/announcements');
      setAnnouncements(res.data.announcements || []);
    } catch (err) {}
    setLoading(false);
  };

  // Filtering logic for public, role, user
  const getVisibleAnnouncements = () => {
    const userId = currentUser?.id;
    const userEmail = currentUser?.email;
    const userRole = currentUser?.role;
    return announcements.filter(a => {
      if (a.is_active !== 1) return false;
      // Public: both target_roles and target_users are empty or whitespace
      if (!a.target_roles?.trim() && !a.target_users?.trim()) return true;
      // Role: target_roles includes userRole
      if (a.target_roles && a.target_roles.split(',').map(r => r.trim()).includes(userRole)) return true;
      // User: target_users includes userId or userEmail
      if (a.target_users && (a.target_users.split(',').map(u => u.trim()).includes(String(userId)) || a.target_users.split(',').map(u => u.trim()).includes(userEmail))) return true;
      return false;
    });
  };

  const visibleAnnouncements = getVisibleAnnouncements();

  return (
    visibleAnnouncements.length === 0 ? null : (
      <Box sx={{ p: 3 }}>
        <Typography variant="h4" gutterBottom>Announcements</Typography>
        {loading ? <CircularProgress /> : (
          visibleAnnouncements.map(a => (
            <Card key={a.id} sx={{ mb: 3, boxShadow: 2 }}>
              <CardContent>
                <Typography variant="h6" sx={{ mb: 1 }}>{a.title}</Typography>
                <div dangerouslySetInnerHTML={{ __html: a.content }} style={{ marginBottom: 8 }} />
                <Box sx={{ mb: 1 }}>
                  {a.target_roles && a.target_roles.split(',').map(role => (
                    <Chip key={role} label={role} size="small" sx={{ mr: 1 }} />
                  ))}
                  {a.target_users && a.target_users.split(',').map(user => (
                    <Chip key={user} label={user} size="small" color="info" sx={{ mr: 1 }} />
                  ))}
                </Box>
                {a.expiration_date && (
                  <Typography variant="caption" sx={{ color: 'red' }}>Expires: {a.expiration_date}</Typography>
                )}
              </CardContent>
            </Card>
          ))
        )}
      </Box>
    )
  );
}

export default AnnouncementPage;
