import Modal from 'antd/lib/modal/Modal';
import React, { useState, useEffect } from 'react';
import { Table, Button, Input } from 'antd';
import { BASE_API } from '../../utils/config';

const userTabs = [
  { key: 'all', label: 'Users' },
  { key: 'admin', label: 'Administrator' },
  { key: 'bpo', label: 'BPOs' },
  { key: 'staff', label: 'Staff' },
  { key: 'students', label: 'Students' },
  { key: 'interpreters', label: 'Interpreters' },
  { key: 'new', label: 'New Users' },
];

const API_URL = `${BASE_API}/api/users`;

const { Search } = Input;

const UsersManager = () => {
  const [activeTab, setActiveTab] = useState('all');
  const [editUser, setEditUser] = useState(null);
  const [users, setUsers] = useState([]);
  const [filteredUsers, setFilteredUsers] = useState(users || []);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [searchText, setSearchText] = useState('');
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [currentPage, setCurrentPage] = useState(1);
  const [selectedRole, setSelectedRole] = useState('');

  useEffect(() => {
    setLoading(true);
    fetch(API_URL)
      .then(res => {
        if (!res.ok) throw new Error('Failed to fetch users');
        return res.json();
      })
      .then(data => {
        setUsers(data);
        setFilteredUsers(data);
        setLoading(false);
      })
      .catch(err => {
        setError(err.message);
        setLoading(false);
      });
  }, []);

  useEffect(() => {
    setFilteredUsers(users);
  }, [users]);

  // Fetch all roles from backend
  const [allRoles, setAllRoles] = useState([]);
  useEffect(() => {
    fetch(`${BASE_API}/api/roles/list`)
      .then(res => res.json())
      .then(data => setAllRoles(Array.isArray(data.roles) ? data.roles : []))
      .catch(() => setAllRoles([]));
  }, []);

  const handleSearch = () => {
    if (!searchText && !selectedRole) {
      setFilteredUsers(users);
      return;
    }
    const lower = searchText.toLowerCase();
    setFilteredUsers(
      users.filter(
        u =>
          ((u.name && u.name.toLowerCase().includes(lower)) ||
            (u.email && u.email.toLowerCase().includes(lower)) ||
            (u.role && u.role.toLowerCase().includes(lower))) &&
          (!selectedRole || u.role === selectedRole)
      )
    );
  };

  // Update filtered users when role changes
  useEffect(() => {
    handleSearch();
    // eslint-disable-next-line
  }, [selectedRole]);

  const filtered = filteredUsers.filter(user => {
    if (activeTab === 'all') return true;
    if (activeTab === 'admin') return user.role === 'Administrator';
    if (activeTab === 'bpo') return user.role === 'Bpo';
    if (activeTab === 'staff') return user.role === 'Staff';
    if (activeTab === 'students') return user.role === 'Student';
    if (activeTab === 'interpreters') return user.role === 'Interpreter';
    if (activeTab === 'new') return user.role === 'Default' && user.code === 'new';
    return false;
  });

  // Pagination logic
  const totalRows = filtered.length;
  const pageSizes = [10, 25, 50, 100, 'All'];
  // For New Users tab, always show sorted by date_created DESC
  const sortedFiltered = activeTab === 'new'
    ? [...filtered].sort((a, b) => new Date(b.date_created) - new Date(a.date_created))
    : filtered;
  const paginatedRows = rowsPerPage === 'All' ? sortedFiltered : sortedFiltered.slice((currentPage - 1) * rowsPerPage, currentPage * rowsPerPage);
  const totalPages = rowsPerPage === 'All' ? 1 : Math.ceil(totalRows / rowsPerPage);

  const handleRowsPerPageChange = e => {
    const value = e.target.value === 'All' ? 'All' : parseInt(e.target.value, 10);
    setRowsPerPage(value);
    setCurrentPage(1);
  };
  const handlePageChange = newPage => setCurrentPage(newPage);
  const handlePrint = () => {
    // Add print style to head
    let style = document.getElementById('print-table-style');
    if (!style) {
      style = document.createElement('style');
      style.id = 'print-table-style';
      style.innerHTML = printTableStyle;
      document.head.appendChild(style);
    }
    window.print();
  };

  // Print CSS for table-only printing
  const printTableStyle = `
    @media print {
      body * { visibility: hidden !important; }
      #user-table-print, #user-table-print * { visibility: visible !important; }
      #user-table-print { position: absolute !important; left: 0; top: 0; width: 100vw; background: #fff; z-index: 9999; }
    }
  `;

  if (loading) return <div>Loading...</div>;
  if (error) return <div>Error: {error}</div>;

  return (
    <div style={{padding:'20px'}}>
      <h2>Users Manager</h2>
      <div style={{display:'flex', marginBottom:'20px'}}>
        {userTabs.map(tab => {
          let count = 0;
          if (tab.key === 'all') {
            count = users.length;
          } else if (tab.key === 'admin') {
            count = users.filter(u => u.role === 'Administrator').length;
          } else if (tab.key === 'bpo') {
            count = users.filter(u => u.role === 'BPOs').length;
          } else if (tab.key === 'staff') {
            count = users.filter(u => u.role === 'Staff').length;
          } else if (tab.key === 'students') {
            count = users.filter(u => u.role === 'Student').length;
          } else if (tab.key === 'interpreters') {
            count = users.filter(u => u.role === 'Interpreter').length;
          } else if (tab.key === 'new') {
            count = users.filter(u => u.role === 'Default' && u.code === 'new').length;
          }
          return (
            <button
              key={tab.key}
              onClick={() => setActiveTab(tab.key)}
              style={{
                padding:'10px 20px',
                marginRight:'10px',
                background: activeTab === tab.key ? '#007bff' : '#f8f9fa',
                color: activeTab === tab.key ? '#fff' : '#000',
                border:'none',
                borderRadius:'5px',
                cursor:'pointer',
                position: 'relative'
              }}
            >
              {tab.label}
              <span style={{
                color: 'red',
                fontWeight: 'bold',
                marginLeft: 6,
                fontSize: '1rem',
                verticalAlign: 'middle',
                position: 'relative',
                top: '-2px'
              }}>
                ({count})
              </span>
            </button>
          );
        })}
      </div>
      <div>
        {activeTab === 'all' && (
          <div style={{ display: 'flex', alignItems: 'center', marginBottom: 16 }}>
            <Search
              placeholder="Search by name, email, or role"
              value={searchText}
              onChange={e => setSearchText(e.target.value)}
              onSearch={handleSearch}
              enterButton="Search"
              style={{ width: 300, marginRight: 8 }}
            />
            <select
              value={selectedRole}
              onChange={e => setSelectedRole(e.target.value)}
              style={{ marginLeft: 8, padding: '4px 8px', minWidth: 120 }}
            >
              <option value="">All Roles</option>
              {allRoles.map(role => (
                <option key={role} value={role}>{role}</option>
              ))}
            </select>
            <div style={{ marginLeft: 16, display: 'flex', alignItems: 'center', gap: 8 }}>
              <label htmlFor="rowsPerPage">Rows per page:</label>
              <select id="rowsPerPage" value={rowsPerPage} onChange={handleRowsPerPageChange} style={{ padding: '4px 8px' }}>
                {pageSizes.map(size => (
                  <option key={size} value={size}>{size}</option>
                ))}
              </select>
              <Button onClick={handlePrint} type="primary" style={{ marginLeft: 16 }}>Print</Button>
            </div>
          </div>
        )}
        {filtered.length === 0 ? (
          <p>No users found.</p>
        ) : (
          <>
          <table id="user-table-print" style={{width:'100%', borderCollapse:'collapse'}}>
            <thead>
              <tr>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>#</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>ID</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Name</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Email</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Role</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Status</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Date Created</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Last Login</th>
                <th style={{border:'1px solid #ddd', padding:'8px'}}>Actions</th>
              </tr>
            </thead>
            <tbody>
              {paginatedRows.map((user, idx) => (
                <tr key={user.id} style={{borderBottom:'1px solid #eee'}}>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{(rowsPerPage === 'All' ? idx + 1 : (currentPage - 1) * rowsPerPage + idx + 1)}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.Id}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.name}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.email}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.role}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.is_new ? 'New' : 'Active'}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.date_created ? new Date(user.date_created).toLocaleString() : ''}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>{user.llogin ? new Date(user.llogin).toLocaleString() : ''}</td>
                  <td style={{border:'1px solid #ddd', padding:'8px'}}>
                    <button style={{background:'#eaf6ff', color:'#1565c0', border:'none', borderRadius:'6px', padding:'4px 10px', cursor:'pointer'}} title="Edit User" onClick={()=>setEditUser(user)}>
                      <svg width="18" height="18" fill="none" viewBox="0 0 24 24"><path d="M4 17.25V21h3.75l11.06-11.06a1.003 1.003 0 0 0 0-1.42l-2.83-2.83a1.003 1.003 0 0 0-1.42 0L4 17.25z" stroke="#1565c0" strokeWidth="2"/></svg>
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
          {/* Pagination controls */}
          {rowsPerPage !== 'All' && totalPages > 1 && (
            <div style={{ display: 'flex', justifyContent: 'flex-end', alignItems: 'center', marginTop: 12, gap: 8 }}>
              <Button onClick={() => handlePageChange(1)} disabled={currentPage === 1}>First</Button>
              <Button onClick={() => handlePageChange(currentPage - 1)} disabled={currentPage === 1}>Prev</Button>
              <span>Page {currentPage} of {totalPages}</span>
              <Button onClick={() => handlePageChange(currentPage + 1)} disabled={currentPage === totalPages}>Next</Button>
              <Button onClick={() => handlePageChange(totalPages)} disabled={currentPage === totalPages}>Last</Button>
            </div>
          )}
          </>
        )}
      </div>
      {editUser && (
        <EditUserPopup
          user={editUser}
          onClose={()=>setEditUser(null)}
          onSave={updatedUser=>{
            setEditUser(null);
            setUsers(users => users.map(u => u.id === updatedUser.id ? updatedUser : u));
          }}
        />
      )}
    </div>
  );
};


const EditUserPopup = ({ user, onClose, onSave }) => {
  const [showAddRate, setShowAddRate] = useState(false);
  const [rateForm, setRateForm] = useState({ language1: '', language2: '', rate: '', per_minute: false, hourly: false, comment: '' });
  const [rateLoading, setRateLoading] = useState(false);
  const [rateError, setRateError] = useState(null);
  const [languages, setLanguages] = useState([]);
  useEffect(() => {
    if (showAddRate) {
      fetch('https://restcountries.com/v3.1/all?fields=languages')
        .then(res => res.json())
        .then(data => {
          // Extract all language names from the API response
          const langSet = new Set();
          data.forEach(country => {
            if (country.languages) {
              Object.values(country.languages).forEach(l => langSet.add(l));
            }
          });
          setLanguages(Array.from(langSet).sort((a, b) => a.localeCompare(b)));
        });
    }
  }, [showAddRate]);
  const [activeTab, setActiveTab] = useState('edit');
  const tabList = [
    { key: 'overview', label: 'Overview' },
    { key: 'edit', label: 'Edit Profile' },
    { key: 'settings', label: 'Settings' },
    { key: 'reset', label: 'Reset Password' },
    { key: 'payment', label: 'Payment Account' },
    { key: 'access', label: 'Access' }
  ];
  const [form, setForm] = useState(user);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [photoPreview, setPhotoPreview] = useState(user?.photo_url || '');
  const [workDuration, setWorkDuration] = useState('');
  const [companies, setCompanies] = useState([]);
  const [roles, setRoles] = useState([]);

  useEffect(() => {
    fetch(`${BASE_API}/api/companies`)
      .then(res => res.json())
      .then(setCompanies);
    fetch(`${BASE_API}/api/roles`)
      .then(res => res.json())
      .then(setRoles);
  }, []);

  if (!user) return null;

  const handlePhotoChange = e => {
    const file = e.target.files[0];
    setForm({...form, photo: file});
    if (file) {
      const reader = new FileReader();
      reader.onload = ev => setPhotoPreview(ev.target.result);
      reader.readAsDataURL(file);
    }
  };

  const handleSave = () => {
    setLoading(true);
    setError(null);
    // handle photo upload separately if needed
    fetch(`${BASE_API}/api/users/${user.id}`, {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(form)
    })
      .then(res => {
        if (!res.ok) throw new Error('Failed to update user');
        return res.json();
      })
      .then(data => {
        setLoading(false);
        onSave(data);
      })
      .catch(err => {
        setError(err.message);
        setLoading(false);
      });
  };

  const handleWorkDuration = () => {
    if (!form.Start_Date) return;
    const start = new Date(form.Start_Date);
    const now = new Date();
    const diff = now - start;
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const years = Math.floor(days / 365);
    const months = Math.floor((days % 365) / 30);
    setWorkDuration(`${years} years, ${months} months, ${days % 30} days`);
  };

  return (
    <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={onClose}>
      <div style={{background:'#fff', borderRadius:'18px', padding:'32px', minWidth:'600px', maxWidth:'1000px', boxShadow:'0 4px 24px rgba(0,0,0,0.18)', width:'100%', maxHeight:'95vh', overflowY:'auto'}} onClick={e=>e.stopPropagation()}>
        <div style={{display:'flex', gap:'12px', marginBottom:'24px', justifyContent:'center', flexWrap:'wrap'}}>
          {tabList.map(tab => (
            <button
              key={tab.key}
              onClick={()=>setActiveTab(tab.key)}
              style={{
                background: activeTab===tab.key ? '#a259e6' : '#f7f8fa',
                color: activeTab===tab.key ? '#fff' : '#333',
                border:'none',
                borderRadius:'8px',
                padding:'10px 24px',
                fontWeight:'bold',
                fontSize:'1rem',
                cursor:'pointer',
                boxShadow: activeTab===tab.key ? '0 2px 8px rgba(162,89,230,0.12)' : 'none'
              }}
            >
              {tab.label}
            </button>
          ))}
        </div>
        {activeTab === 'overview' && (
          <div style={{textAlign:'center', padding:'24px'}}>
            <h3>Overview</h3>
            <p>Profile summary and quick stats will appear here.</p>
          </div>
        )}
        {activeTab === 'edit' && (
          <>
            <h2 style={{marginBottom:'24px', textAlign:'center', fontWeight:'bold'}}>Edit User Profile</h2>
            <form style={{display:'grid', gridTemplateColumns:'1fr 1fr', gap:'20px'}}>
              <div style={{gridColumn:'span 2', display:'flex', alignItems:'center', gap:'18px', marginBottom:'10px'}}>
                <div style={{position:'relative', width:'90px', height:'90px'}}>
                  <label htmlFor="photo-upload" style={{position:'absolute', top:0, left:0, width:'100%', height:'100%', cursor:'pointer', zIndex:2}}></label>
                  <input id="photo-upload" type="file" accept="image/*" onChange={handlePhotoChange} style={{display:'none'}} />
                  <div style={{width:'90px', height:'90px', borderRadius:'50%', border:'3px solid #d32f2f', overflow:'hidden', boxShadow:'0 2px 8px rgba(0,0,0,0.10)'}}>
                    {photoPreview ? (
                      <img src={photoPreview} alt="Preview" style={{width:'100%', height:'100%', objectFit:'cover'}} />
                    ) : (
                      <div style={{width:'100%', height:'100%', display:'flex', alignItems:'center', justifyContent:'center', color:'#aaa'}}>No Photo</div>
                    )}
                  </div>
                  <div style={{position:'absolute', top:'8px', right:'8px', background:'#22c55e', borderRadius:'50%', width:'28px', height:'28px', display:'flex', alignItems:'center', justifyContent:'center', zIndex:3}}>
                    <svg width="18" height="18" viewBox="0 0 24 24" fill="none"><circle cx="12" cy="12" r="12" fill="none"/><path d="M7 12v5a2 2 0 0 0 2 2h6a2 2 0 0 0 2-2v-5" stroke="#fff" strokeWidth="2"/><rect x="9" y="7" width="6" height="5" rx="1" fill="#fff"/></svg>
                  </div>
                </div>
                <div style={{marginLeft:'18px'}}>
                  <label style={{fontWeight:'bold'}}>Photo:</label><br/>
                  <span style={{fontSize:'0.95rem', color:'#888'}}>Click the circle to upload</span>
                </div>
              </div>
              <div>
                <label>Prénom:</label>
                <input type="text" value={form.prenom || ''} onChange={e=>setForm({...form, prenom:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Nom:</label>
                <input type="text" value={form.nom || ''} onChange={e=>setForm({...form, nom:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>NIF:</label>
                <input type="text" value={form.nif || ''} onChange={e=>setForm({...form, nif:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Birth Date:</label>
                <input type="date" value={form.bdates || ''} onChange={e=>setForm({...form, bdates:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>ID Date:</label>
                <input type="date" value={form.Id_date || ''} onChange={e=>setForm({...form, Id_date:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Expiry Date:</label>
                <input type="date" value={form.Exp_date || ''} onChange={e=>setForm({...form, Exp_date:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Sexe:</label>
                <select value={form.Sexe || ''} onChange={e=>setForm({...form, Sexe:e.target.value})} style={{width:'100%', padding:'8px'}}>
                  <option value="">Select</option>
                  <option value="Male">Male</option>
                  <option value="Female">Female</option>
                  <option value="Other">Other</option>
                </select>
              </div>
              <div style={{gridColumn:'span 2'}}>
                <label>About:</label>
                <textarea value={form.About || ''} onChange={e=>setForm({...form, About:e.target.value})} style={{width:'100%', padding:'8px', minHeight:'48px'}} />
              </div>
              <div>
                <label>Team Code:</label>
                <input type="text" value={form.Team_Code || ''} onChange={e=>setForm({...form, Team_Code:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Company:</label>
                <select value={form.company_id || ''} onChange={e=>setForm({...form, company_id:e.target.value})} style={{width:'100%', padding:'8px'}}>
                  <option value="">Select</option>
                  {companies.map(c => <option key={c.id} value={c.id}>{c.fullname}</option>)}
                </select>
              </div>
              <div>
                <label>Role:</label>
                <select value={form.role_id || ''} onChange={e=>setForm({...form, role_id:e.target.value})} style={{width:'100%', padding:'8px'}}>
                  <option value="">Select</option>
                  {roles.map(r => <option key={r.id} value={r.id}>{r.role_name}</option>)}
                </select>
              </div>
              <div>
                <label>Country:</label>
                <input type="text" value={form.Country || ''} onChange={e=>setForm({...form, Country:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Address:</label>
                <input type="text" value={form.Address || ''} onChange={e=>setForm({...form, Address:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Phone:</label>
                <input type="text" value={form.Phone || ''} onChange={e=>setForm({...form, Phone:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Email:</label>
                <input type="email" value={form.Email || ''} onChange={e=>setForm({...form, Email:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div>
                <label>Start Date:</label>
                <input type="date" value={form.Start_Date || ''} onChange={e=>setForm({...form, Start_Date:e.target.value})} style={{width:'100%', padding:'8px'}} />
              </div>
              <div style={{display:'flex', alignItems:'center', gap:'10px'}}>
                <button type="button" style={{background:'#eaf6ff', color:'#1565c0', border:'none', borderRadius:'8px', padding:'8px 18px', fontWeight:'bold', cursor:'pointer'}} onClick={handleWorkDuration}>Show Work Duration</button>
                {workDuration && <span style={{marginLeft:'8px', fontWeight:'bold'}}>{workDuration}</span>}
              </div>
              <div>
                <label>Status:</label>
                <select value={form.is_new ? 'New' : 'Active'} onChange={e=>setForm({...form, is_new:e.target.value==='New'})} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}}>
                  <option value="Active">Active</option>
                  <option value="New">New</option>
                </select>
              </div>
            </form>
            {error && <div style={{color:'#f75c7a', margin:'12px 0'}}>{error}</div>}
            <div style={{display:'flex', gap:'16px', marginTop:'28px', justifyContent:'center'}}>
              <button style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'10px 32px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={handleSave} disabled={loading}>{loading ? 'Saving...' : 'Save'}</button>
              <button style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'10px 32px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={onClose} disabled={loading}>Cancel</button>
            </div>
          </>
        )}
        {activeTab === 'settings' && (
          <div style={{textAlign:'center', padding:'24px'}}>
            <h3>Settings</h3>
            <p>Settings content goes here.</p>
          </div>
        )}
        {activeTab === 'reset' && (
          <div style={{textAlign:'center', padding:'24px'}}>
            <h3>Reset Password</h3>
            <form style={{maxWidth:'340px', margin:'32px auto', display:'flex', flexDirection:'column', gap:'18px'}} onSubmit={e => {
              e.preventDefault();
              setLoading(true);
              setError(null);
              fetch(`${BASE_API}/api/users/${user.id}/reset-password`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ password: e.target.password.value })
              })
                .then(res => {
                  if (!res.ok) throw new Error('Failed to reset password');
                  return res.json();
                })
                .then(() => {
                  setLoading(false);
                  alert('Password updated successfully!');
                })
                .catch(err => {
                  setError(err.message);
                  setLoading(false);
                });
            }}>
              <input type="password" name="password" placeholder="New Password" style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} required />
              <input type="password" name="confirm" placeholder="Confirm Password" style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} required />
              {error && <div style={{color:'#f75c7a'}}>{error}</div>}
              <button type="submit" style={{background:'#a259e6', color:'#fff', border:'none', borderRadius:'8px', padding:'10px 0', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} disabled={loading}>{loading ? 'Updating...' : 'Update Password'}</button>
            </form>
          </div>
        )}
        {activeTab === 'payment' && (
          <div style={{padding:'32px 0', display:'flex', flexDirection:'column', alignItems:'center'}}>
            <button type="button" style={{marginBottom: 18, background:'#a259e6', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={()=>setShowAddRate(true)}>+ Add Rate</button>
            <form style={{maxWidth:'480px', width:'100%', background:'', display:'grid', gridTemplateColumns:'1fr 1fr', gap:'18px'}} onSubmit={e => {
              e.preventDefault();
              setLoading(true);
              setError(null);
              fetch(`${BASE_API}/api/users/${user.id}/payment-account`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                  payment_method: e.target.payment_method.value,
                  devise: e.target.devise.value,
                  account: e.target.account.value,
                  rate: e.target.rate.value,
                  accept_invoice: e.target.accept_invoice.checked,
                  pay_per_hour: e.target.pay_per_hour.checked,
                  pay_per_minutes: e.target.pay_per_minutes.checked
                })
              })
                .then(res => {
                  if (!res.ok) throw new Error('Failed to update payment account');
                  return res.json();
                })
                .then(() => {
                  setLoading(false);
                  alert('Payment account updated!');
                })
                .catch(err => {
                  setError(err.message);
                  setLoading(false);
                });
            }}>
              <label style={{gridColumn:'1/2', alignSelf:'center'}}>Payment Methods</label>
              <input name="payment_method" type="text" defaultValue={user.payment_method || ''} style={{gridColumn:'2/3', padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
              <label style={{gridColumn:'1/2', alignSelf:'center'}}>Devise</label>
              <input name="devise" type="text" defaultValue={user.devise || ''} style={{gridColumn:'2/3', padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
              <label style={{gridColumn:'1/2', alignSelf:'center'}}>Account#</label>
              <input name="account" type="text" defaultValue={user.account || ''} style={{gridColumn:'2/3', padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
              <label style={{gridColumn:'1/2', alignSelf:'center'}}>Rate</label>
              <input name="rate" type="text" defaultValue={user.rate || ''} style={{gridColumn:'2/3', padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
              <div style={{gridColumn:'2/3', display:'flex', flexDirection:'column', gap:'8px', marginTop:'8px'}}>
                <label><input type="checkbox" name="accept_invoice" defaultChecked={user.accept_invoice} /> Accept invoice</label>
                <label><input type="checkbox" name="pay_per_hour" defaultChecked={user.pay_per_hour} /> Pay per hour</label>
                <label><input type="checkbox" name="pay_per_minutes" defaultChecked={user.pay_per_minutes} /> Pay per minutes</label>
              </div>
              {error && <div style={{gridColumn:'1/3', color:'#f75c7a', marginTop:'8px'}}>{error}</div>}
              <button type="submit" style={{gridColumn:'1/3', margin:'32px auto 0 auto', background:'linear-gradient(90deg,#6a5af9,#a259e6)', color:'#fff', border:'none', borderRadius:'10px', padding:'12px 0', fontWeight:'bold', fontSize:'1.1rem', boxShadow:'0 2px 8px rgba(162,89,230,0.12)', width:'180px', cursor:'pointer'}} disabled={loading}>{loading ? 'Updating...' : 'UPDATE'}</button>
            </form>
            {/* Add Rate Modal */}
            {showAddRate && (
              <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:1000, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={()=>setShowAddRate(false)}>
                <div style={{background:'#fff', borderRadius:'12px', padding:'32px', minWidth:'340px', boxShadow:'0 4px 24px rgba(0,0,0,0.18)', position:'relative'}} onClick={e=>e.stopPropagation()}>
                  <h3 style={{marginBottom:'18px'}}>Add Rate</h3>
                  <form onSubmit={e => {
                    e.preventDefault();
                    setRateLoading(true);
                    setRateError(null);
                    fetch(`${BASE_API}/api/rates`, {
                      method: 'POST',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify({
                        user_id: user.Id,
                        language1: rateForm.language1,
                        language2: rateForm.language2,
                        rate: rateForm.rate,
                        per_minute: rateForm.per_minute,
                        hourly: rateForm.hourly,
                        comment: rateForm.comment
                      })
                    })
                      .then(res => {
                        if (!res.ok) throw new Error('Failed to add rate');
                        return res.json();
                      })
                      .then(() => {
                        // Refresh rates after add
                        fetch(`${BASE_API}/api/rates?user_id=${user.Id}`)
                          .then(res => res.json())
                          .then(data => {
                            if (data && data.rates) {
                              user.ratesList = data.rates;
                            }
                          });
                        setRateLoading(false);
                        setShowAddRate(false);
                        setRateForm({ language1: '', language2: '', rate: '', per_minute: false, hourly: false, comment: '' });
                        alert('Rate added!');
                      })
                      .catch(err => {
                        setRateError(err.message);
                        setRateLoading(false);
                      });
                  }}>
                    <div style={{marginBottom:'12px'}}>
                      <label>Language 1:</label><br/>
                      <select value={rateForm.language1} onChange={e=>setRateForm(f=>({...f, language1:e.target.value, language2: f.language2 === e.target.value ? '' : f.language2}))} style={{width:'100%', padding:'8px'}} required>
                        <option value="">Select language</option>
                        {languages.filter(l => l !== rateForm.language2).map(l => (
                          <option key={l} value={l}>{l}</option>
                        ))}
                      </select>
                    </div>
                    <div style={{marginBottom:'12px'}}>
                      <label>Language 2:</label><br/>
                      <select value={rateForm.language2} onChange={e=>setRateForm(f=>({...f, language2:e.target.value, language1: f.language1 === e.target.value ? '' : f.language1}))} style={{width:'100%', padding:'8px'}} required>
                        <option value="">Select language</option>
                        {languages.filter(l => l !== rateForm.language1).map(l => (
                          <option key={l} value={l}>{l}</option>
                        ))}
                      </select>
                    </div>
                    <div style={{marginBottom:'12px'}}>
                      <label>Rate:</label><br/>
                      <input type="number" value={rateForm.rate} onChange={e=>setRateForm(f=>({...f, rate:e.target.value}))} style={{width:'100%', padding:'8px'}} required />
                    </div>
                    <div style={{marginBottom:'12px', display:'flex', gap:'18px'}}>
                      <label><input type="checkbox" checked={rateForm.per_minute} onChange={e=>setRateForm(f=>({...f, per_minute:e.target.checked, hourly: e.target.checked ? false : f.hourly}))} /> Per Minute</label>
                      <label><input type="checkbox" checked={rateForm.hourly} onChange={e=>setRateForm(f=>({...f, hourly:e.target.checked, per_minute: e.target.checked ? false : f.per_minute}))} /> Hourly</label>
                    </div>
                    <div style={{marginBottom:'12px'}}>
                      <label>Comment:</label><br/>
                      <textarea value={rateForm.comment} onChange={e=>setRateForm(f=>({...f, comment:e.target.value}))} style={{width:'100%', padding:'8px'}} rows={2} />
                    </div>
                    {rateError && <div style={{color:'#f75c7a', marginBottom:'8px'}}>{rateError}</div>}
                    <div style={{display:'flex', justifyContent:'flex-end', gap:'12px'}}>
                      <button type="button" onClick={()=>setShowAddRate(false)} style={{background:'#eee', color:'#333', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Cancel</button>
                      <button type="submit" style={{background:'#a259e6', color:'#fff', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} disabled={rateLoading}>{rateLoading ? 'Saving...' : 'Save'}</button>
                    </div>
                  </form>
                </div>
              </div>
            )}
          </div>
        )}
        {activeTab === 'access' && (
          <div style={{padding:'32px 0', display:'flex', flexDirection:'column', alignItems:'center'}}>
            <form style={{maxWidth:'900px', width:'100%', display:'grid', gridTemplateColumns:'1fr 1fr', gap:'24px'}} onSubmit={e => {
              e.preventDefault();
              setLoading(true);
              setError(null);
              // Collect checked permissions
              const adminPanel = Array.from(e.target.adminPanel).filter(cb => cb.checked).map(cb => cb.value);
              const usersPanel = Array.from(e.target.usersPanel).filter(cb => cb.checked).map(cb => cb.value);
              fetch(`${BASE_API}/api/users/${user.id}/access`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ adminPanel, usersPanel })
              })
                .then(res => {
                  if (!res.ok) throw new Error('Failed to update access');
                  return res.json();
                })
                .then(() => {
                  setLoading(false);
                  alert('Access updated!');
                })
                .catch(err => {
                  setError(err.message);
                  setLoading(false);
                });
            }}>
              <div>
                <div style={{fontWeight:'bold', fontSize:'1.2rem', marginBottom:'12px'}}><span style={{marginRight:'8px'}}>👤</span>Admin Panel</div>
                <div style={{display:'grid', gridTemplateColumns:'1fr 1fr', gap:'8px'}}>
                  {['Add access','Add career','Add course','Add course conten...','Add document','Add document Tuto...','Add Medical Word','Add page','Add service','Edit career','Edit course conte...','Edit insight','Edit partner','Edit User','Manage page','Manage Tutorial','Tasks','Add badge','Add client servic...','Add course access','Add detail servic...','Add document acce...','Add insight','Add or remove acc...','Add partner','Chart Admin','Edit course','Edit document','Edit Medical Word','Edit service','Edit Word','Manage Quiz','Manage Tutorial'].map((perm,i)=>(
                    <label key={perm} style={{display:'flex',alignItems:'center'}}>
                      <input type="checkbox" name="adminPanel" value={perm} defaultChecked={user.adminPanel?.includes(perm)} style={{accentColor:'#6a5af9',marginRight:'6px'}} /> {perm}
                    </label>
                  ))}
                </div>
              </div>
              <div>
                <div style={{fontWeight:'bold', fontSize:'1.2rem', marginBottom:'12px'}}><span style={{marginRight:'8px'}}>👥</span>Users Panel</div>
                <div style={{display:'grid', gridTemplateColumns:'1fr 1fr', gap:'8px'}}>
                  {['Chart Employee','Chart Staff','Courses','Dashboard','Document','Interp. Submissio...','MAKE-IT Center','Medical Document','Medical Interpret...','Quick Links','Translation','Video Tutorial','View Blog','View News'].map((perm,i)=>(
                    <label key={perm} style={{display:'flex',alignItems:'center'}}>
                      <input type="checkbox" name="usersPanel" value={perm} defaultChecked={user.usersPanel?.includes(perm)} style={{accentColor:'#6a5af9',marginRight:'6px'}} /> {perm}
                    </label>
                  ))}
                </div>
              </div>
              {error && <div style={{gridColumn:'1/3', color:'#f75c7a', marginTop:'8px'}}>{error}</div>}
              <button type="submit" style={{gridColumn:'1/3', margin:'32px auto 0 auto', background:'linear-gradient(90deg,#6a5af9,#a259e6)', color:'#fff', border:'none', borderRadius:'10px', padding:'12px 0', fontWeight:'bold', fontSize:'1.1rem', boxShadow:'0 2px 8px rgba(162,89,230,0.12)', width:'180px', cursor:'pointer'}} disabled={loading}>{loading ? 'Updating...' : 'UPDATE'}</button>
            </form>
          </div>
        )}
      </div>
    </div>
  );
};

export default UsersManager;
