import React, { useEffect, useState, useRef, useMemo } from 'react';
import axios from 'axios';
import { logAction } from '../../utils/actionLogger';
import { BASE_API } from '../../utils/config';

// possibleAccess will be fetched from backend


const UserAccessManager = () => {
  const [users, setUsers] = useState([]);
  const [search, setSearch] = useState('');
  const [debouncedSearch, setDebouncedSearch] = useState('');
  const debounceTimeout = useRef();
  const [accessList, setAccessList] = useState([]);
  const [loading, setLoading] = useState(false);
  const [feedback, setFeedback] = useState({ type: '', message: '' });
  const [selectedUser, setSelectedUser] = useState(null);
  const [possibleAccess, setPossibleAccess] = useState([]);
  const [viewMode, setViewMode] = useState('manage'); // 'manage' or 'viewAccess'
  const [accessUserMap, setAccessUserMap] = useState([]);
  const [userNamesMap, setUserNamesMap] = useState({});
  const [accessViewSearch, setAccessViewSearch] = useState('');
  // Fetch all users for each access (for View Access tab)
  const fetchAccessUserMap = async () => {
    try {
      const [accessRes, userAccessRes, userNamesRes] = await Promise.all([
        axios.get('/api/access_names'),
        axios.get('/api/user_access'),
        axios.get('/api/user_names'),
      ]);
      const accessList = accessRes.data || [];
      const userAccessList = userAccessRes.data || [];
      const userNamesList = userNamesRes.data || [];
      // Build a map for fast lookup: userId -> prenom + nom
      const userMap = {};
      userNamesList.forEach(u => {
        userMap[String(u.id_user)] = ((u.prenom || '') + (u.nom ? ' ' + u.nom : '')).trim();
      });
      setUserNamesMap(userMap);
      // Map access_code to users (with prenom/nom, fallback to 'Unknown')
      const map = accessList.map(access => {
        const usersWithAccess = userAccessList
          .filter(ua => String(ua.accessKey) === String(access.access_code) || ua.accessKey === access.name)
          .map(ua => {
            const name = userMap[String(ua.userId)];
            return name || 'Unknown';
          })
          .filter(Boolean);
        return {
          name: access.name,
          access_code: access.access_code,
          users: usersWithAccess,
        };
      });
      setAccessUserMap(map);
    } catch (e) {
      setAccessUserMap([]);
    }
  };
  // CSV Export for View Access
  const handleExportCSV = () => {
    const header = ['Access Name', 'Access Code', 'User Count', 'Users'];
    const rows = accessUserMapFiltered.map(a => [
      a.name,
      a.access_code,
      a.users.length,
      a.users.join('; ')
    ]);
    const csv = [header, ...rows].map(r => r.map(x => '"' + String(x).replace(/"/g, '""') + '"').join(',')).join('\n');
    const blob = new Blob([csv], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'access_user_map.csv';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  // PDF Export for View Access
  const handleExportPDF = async () => {
    // Dynamically import jsPDF and autotable
    const jsPDF = (await import('jspdf')).default;
    const autoTable = (await import('jspdf-autotable')).default;
    const doc = new jsPDF();
    const tableColumn = ['Access Name', 'Access Code', 'User Count', 'Users'];
    const tableRows = accessUserMapFiltered.map(a => [
      a.name,
      a.access_code,
      a.users.length,
      a.users.join('; ')
    ]);
    autoTable(doc, {
      head: [tableColumn],
      body: tableRows,
      styles: { fontSize: 9 },
      headStyles: { fillColor: [41, 128, 185] },
      margin: { top: 20 },
    });
    doc.save('access_user_map.pdf');
  };
  // Filtered accessUserMap for search
  const accessUserMapFiltered = useMemo(() => {
    const s = accessViewSearch.trim().toLowerCase();
    if (!s) return accessUserMap;
    return accessUserMap.filter(a =>
      a.name.toLowerCase().includes(s) ||
      String(a.access_code).includes(s) ||
      a.users.some(u => u.toLowerCase().includes(s))
    );
  }, [accessUserMap, accessViewSearch]);
  // Fetch access-user map when switching to viewAccess
  useEffect(() => {
    if (viewMode === 'viewAccess') {
      fetchAccessUserMap();
    }
  }, [viewMode]);
  const [newAccessName, setNewAccessName] = useState('');
  const [newAccessCode, setNewAccessCode] = useState('');
  const [showAddAccessModal, setShowAddAccessModal] = useState(false);


  useEffect(() => {
    fetchUsers();
    fetchAccessList();
    fetchPossibleAccess();
    logAction({
      actor_id: localStorage.getItem('userId'),
      actor_role: localStorage.getItem('role'),
      actor_email: localStorage.getItem('email'),
      action_type: 'view_access_manager',
      action_description: 'Viewed user access management page',
      page: window.location.pathname
    });
  }, []);

  // Fetch possible access names from backend
  const fetchPossibleAccess = async () => {
    try {
      const res = await axios.get('/api/access_names');
      setPossibleAccess(res.data || []);
    } catch (e) {
      setPossibleAccess([]);
    }
  };

  // Add new access name and code
  const handleAddAccessName = async () => {
    if (!newAccessName.trim() || !newAccessCode.trim()) return;
    setLoading(true);
    try {
      await axios.post('/api/access_names', { name: newAccessName.trim(), access_code: Number(newAccessCode) });
      setFeedback({ type: 'success', message: 'Access name added.' });
      setNewAccessName('');
      setNewAccessCode('');
      setShowAddAccessModal(false);
      fetchPossibleAccess();
      // Log the action with user info
      const actor_id = localStorage.getItem('userId');
      const actor_role = localStorage.getItem('role');
      const actor_email = localStorage.getItem('email');
      await axios.post(`${BASE_API}/api/log_action`, {
        actor_id,
        actor_role,
        actor_email,
        action_type: 'add_access_name',
        action_description: `User ${actor_email || actor_id} added new access name: '${newAccessName.trim()}' (code: ${newAccessCode})`,
        page: window.location.pathname
      });
    } catch (e) {
      setFeedback({ type: 'error', message: 'Failed to add access name.' });
    }
    setLoading(false);
  };


  // Debounce search input
  useEffect(() => {
    if (debounceTimeout.current) clearTimeout(debounceTimeout.current);
    debounceTimeout.current = setTimeout(() => {
      setDebouncedSearch(search);
    }, 300);
    return () => clearTimeout(debounceTimeout.current);
  }, [search]);


  const fetchUsers = async () => {
    const res = await axios.get('/api/users?limit=50');
    setUsers(res.data || []);
  };
  const fetchAccessList = async () => {
    const res = await axios.get('/api/user_access');
    setAccessList(res.data || []);
  };


  const handleToggleAccess = async (userId, accessKey, enabled) => {
    if (!userId) {
      setFeedback({ type: 'error', message: 'Invalid user. Please refresh and try again.' });
      return;
    }
    setLoading(true);
    let prevAccessList = accessList;
    setAccessList(list => {
      if (enabled) {
        return [...list, { userId, accessKey }];
      } else {
        return list.filter(acc => !(acc.userId === userId && acc.accessKey === accessKey));
      }
    });
    try {
      const validUserId = String(userId);
      await axios.post('/api/user_access/update', { userId: validUserId, accessKey, enabled });
      // Log the grant/revoke action
      const actor_id = localStorage.getItem('userId');
      const actor_role = localStorage.getItem('role');
      const actor_email = localStorage.getItem('email');
      await logAction({
        actor_id,
        actor_role,
        actor_email,
        action_type: enabled ? 'grant_access' : 'revoke_access',
        action_description: `${enabled ? 'Granted' : 'Revoked'} access '${accessKey}' for user ${validUserId}`,
        target_id: validUserId,
        page: window.location.pathname
      });
      setFeedback({ type: 'success', message: `Access ${enabled ? 'granted' : 'revoked'} successfully.` });
      await fetchAccessList();
    } catch (err) {
      setAccessList(prevAccessList); // revert
      setFeedback({ type: 'error', message: 'Failed to update access.' });
    }
    setLoading(false);
  };

  const handleAddAccess = async () => {
    if (!selectedUser || !newAccess) return;
    const validUserId = String(selectedUser);
    setLoading(true);
    // Optimistically update UI
    let prevAccessList = accessList;
    setAccessList(list => [...list, { userId: validUserId, accessKey: newAccess }]);
    try {
      await axios.post('/api/user_access/add', { userId: validUserId, accessKey: newAccess });
  const user = users.find(u => String(u.Id) === validUserId);
      const target_email = user ? (user.email || user.username || '') : '';
      const admin_email = localStorage.getItem('email');
      if (!target_email || target_email === admin_email) {
        setFeedback({ type: 'warning', message: 'Warning: Target user email missing or is admin. Action not logged.' });
      } else {
        await logAction({
          actor_id: localStorage.getItem('userId'),
          actor_role: localStorage.getItem('role'),
          actor_email: admin_email,
          action_type: 'add_access',
          action_description: `Added ${newAccess} for user ${validUserId}`,
          target_id: validUserId,
          target_email,
          page: 'user_access_manager'
        });
      }
      setFeedback({ type: 'success', message: `Access to ${newAccess} added.` });
      setAddDialogOpen(false);
      setNewAccess('');
      await fetchAccessList();
      await fetchLogs();
    } catch (err) {
      setAccessList(prevAccessList); // revert
      setFeedback({ type: 'error', message: 'Failed to add access.' });
    }
    setLoading(false);
  };

  const handleRemoveAccess = async (userId, accessKey) => {
    setLoading(true);
    // Optimistically update UI
    let prevAccessList = accessList;
    setAccessList(list => list.filter(acc => !(acc.userId === userId && acc.accessKey === accessKey)));
    try {
      await axios.post('/api/user_access/remove', { userId, accessKey });
  const user = users.find(u => String(u.Id) === String(userId));
      const target_email = user ? (user.email || user.username || '') : '';
      const admin_email = localStorage.getItem('email');
      if (!target_email || target_email === admin_email) {
        setFeedback({ type: 'warning', message: 'Warning: Target user email missing or is admin. Action not logged.' });
      } else {
        await logAction({
          actor_id: localStorage.getItem('userId'),
          actor_role: localStorage.getItem('role'),
          actor_email: admin_email,
          action_type: 'remove_access',
          action_description: `Removed ${accessKey} for user ${userId}`,
          target_id: userId,
          target_email,
          page: 'user_access_manager'
        });
      }
      setFeedback({ type: 'success', message: `Access to ${accessKey} removed.` });
      await fetchAccessList();
      await fetchLogs();
    } catch (err) {
      setAccessList(prevAccessList); // revert
      setFeedback({ type: 'error', message: 'Failed to remove access.' });
    }
    setLoading(false);
  };


  // Memoized filtered users for performance
  const filteredUsers = useMemo(() => {
    const s = debouncedSearch.trim().toLowerCase();
    let filtered = users.filter(u => typeof u.Id !== 'undefined');
    if (!s) {
      // Show all users if no search
      return filtered;
    }
    filtered = filtered.filter(u => {
      const email = (u.email || '').toLowerCase();
      const prenom = (u.prenom || '').toLowerCase();
      const nom = (u.nom || '').toLowerCase();
      const id = (typeof u.Id !== 'undefined' ? String(u.Id) : '');
      return email.includes(s) || prenom.includes(s) || nom.includes(s) || id.includes(s);
    });
    return filtered;
  }, [users, debouncedSearch]);

  // Edit Access Modal state
  const [showEditAccessModal, setShowEditAccessModal] = useState(false);
  const [editAccessList, setEditAccessList] = useState([]);
  const [editAccessName, setEditAccessName] = useState('');
  const [editAccessCode, setEditAccessCode] = useState('');
  const [editAccessId, setEditAccessId] = useState(null);

  // Open edit modal and load access list
  useEffect(() => {
    if (showEditAccessModal) {
      setEditAccessList(possibleAccess);
      setEditAccessName('');
      setEditAccessCode('');
      setEditAccessId(null);
    }
  }, [showEditAccessModal, possibleAccess]);

  // Handle selecting an access to edit
  const handleSelectEditAccess = (access) => {
    setEditAccessId(access.access_code);
    setEditAccessName(access.name);
    setEditAccessCode(access.access_code);
  };

  // Handle saving edited access
  const handleSaveEditAccess = async () => {
    if (!editAccessName.trim() || !editAccessCode) return;
    setLoading(true);
    try {
      await axios.post('/api/access_names/edit', { access_code: editAccessId, name: editAccessName.trim(), new_code: editAccessCode });
      // Log the edit action
      const actor_id = localStorage.getItem('userId');
      const actor_role = localStorage.getItem('role');
      const actor_email = localStorage.getItem('email');
      await logAction({
        actor_id,
        actor_role,
        actor_email,
        action_type: 'edit_access_type',
        action_description: `Edited access name/code: id=${editAccessId}, new name='${editAccessName.trim()}', new code='${editAccessCode}'`,
        page: window.location.pathname
      });
      setFeedback({ type: 'success', message: 'Access updated.' });
      setShowEditAccessModal(false);
      fetchPossibleAccess();
    } catch (e) {
      setFeedback({ type: 'error', message: 'Failed to update access.' });
    }
    setLoading(false);
  };

  // Handle deleting access
  const handleDeleteAccess = async () => {
    if (!editAccessId) return;
    setLoading(true);
    try {
      await axios.post('/api/access_names/delete', { access_code: editAccessId });
      // Log the delete action
      const actor_id = localStorage.getItem('userId');
      const actor_role = localStorage.getItem('role');
      const actor_email = localStorage.getItem('email');
      await logAction({
        actor_id,
        actor_role,
        actor_email,
        action_type: 'delete_access_type',
        action_description: `Deleted access name/code: id=${editAccessId}, name='${editAccessName}', code='${editAccessCode}'`,
        page: window.location.pathname
      });
      setFeedback({ type: 'success', message: 'Access deleted.' });
      setShowEditAccessModal(false);
      fetchPossibleAccess();
    } catch (e) {
      setFeedback({ type: 'error', message: 'Failed to delete access.' });
    }
    setLoading(false);
  };

  return (
    <div className="container py-4">
      <div className="d-flex align-items-center mb-4 gap-2">
        <h2 className="mb-0 me-3">User Access Management</h2>
        <button className={`btn btn-${viewMode === 'manage' ? 'primary' : 'outline-primary'}`} onClick={() => setViewMode('manage')}>
          Manage Access
        </button>
        <button className={`btn btn-${viewMode === 'viewAccess' ? 'primary' : 'outline-primary'}`} onClick={() => setViewMode('viewAccess')}>
          View Access
        </button>
        <button className="btn btn-primary ms-2" onClick={() => {
          // Auto-generate next available unique access code
          let code = 100;
          const usedCodes = new Set(possibleAccess.map(a => Number(a.access_code || a.code || a)));
          while (usedCodes.has(code)) code++;
          setNewAccessCode(String(code));
          setShowAddAccessModal(true);
        }}>
          Add Access
        </button>
        <button className="btn btn-outline-warning ms-2" style={{fontWeight:'bold'}} onClick={() => setShowEditAccessModal(true)}>
          Edit Access
        </button>
      {/* Edit Access Modal */}
      {showEditAccessModal && (
        <div className="modal show fade" style={{display:'block', background:'rgba(0,0,0,0.3)', position:'fixed', top:0, left:0, width:'100vw', height:'100vh', zIndex:1050}} tabIndex="-1" role="dialog">
          <div className="modal-dialog modal-dialog-centered" role="document" style={{maxWidth:'400px', margin:'auto'}}>
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit or Delete Access</h5>
                <button type="button" className="btn-close" aria-label="Close" onClick={() => setShowEditAccessModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-3">
                  <label className="form-label">Select Access</label>
                  <select className="form-select" value={editAccessId || ''} onChange={e => {
                    const selected = editAccessList.find(a => String(a.access_code) === e.target.value);
                    handleSelectEditAccess(selected || {});
                  }}>
                    <option value="">-- Select --</option>
                    {editAccessList.map(a => (
                      <option key={a.access_code} value={a.access_code}>{a.name} ({a.access_code})</option>
                    ))}
                  </select>
                </div>
                {editAccessId && (
                  <>
                    <div className="mb-3">
                      <label className="form-label">Access Name</label>
                      <input
                        type="text"
                        className="form-control"
                        value={editAccessName}
                        onChange={e => setEditAccessName(e.target.value)}
                        disabled={loading}
                      />
                    </div>
                    <div className="mb-3">
                      <label className="form-label">Access Code</label>
                      <input
                        type="number"
                        className="form-control"
                        value={editAccessCode}
                        onChange={e => setEditAccessCode(e.target.value)}
                        disabled={loading}
                      />
                    </div>
                  </>
                )}
              </div>
              <div className="modal-footer d-flex justify-content-between">
                <button type="button" className="btn btn-secondary" onClick={() => setShowEditAccessModal(false)} disabled={loading}>Cancel</button>
                {editAccessId && (
                  <>
                    <button type="button" className="btn btn-danger" onClick={handleDeleteAccess} disabled={loading}>Delete</button>
                    <button type="button" className="btn btn-primary" onClick={handleSaveEditAccess} disabled={loading || !editAccessName.trim() || !editAccessCode}>Save</button>
                  </>
                )}
              </div>
            </div>
          </div>
        </div>
      )}
      </div>
      {viewMode === 'manage' && (
        <div className="row mb-3">
          <div className="col-md-8">
            <input
              type="text"
              className="form-control"
              placeholder="Search by ID, email, prenom, or nom"
              value={search}
              onChange={e => setSearch(e.target.value)}
            />
          </div>
        </div>
      )}
      {feedback.message && (
        <div className={`alert alert-${feedback.type === 'success' ? 'success' : 'danger'} mb-3`} role="alert">
          {feedback.message}
        </div>
      )}

      {viewMode === 'manage' && filteredUsers.length === 0 && (
        <div className="alert alert-secondary">No users found.</div>
      )}
      {viewMode === 'manage' && filteredUsers.length > 0 && (
        <div className="table-responsive">
          <table className="table table-bordered align-middle">
            <thead className="table-light">
              <tr>
                <th>ID</th>
                <th>Prenom</th>
                <th>Nom</th>
                <th>Access</th>
              </tr>
            </thead>
            <tbody>
              {filteredUsers.map(user => {
                const userAccessKeys = possibleAccess.map(accessObj => accessObj.access_code || accessObj.code || accessObj);
                const allChecked = userAccessKeys.every(accessKey => accessList.some(acc => acc.userId == user.Id && (acc.accessKey == accessKey)));
                const someChecked = userAccessKeys.some(accessKey => accessList.some(acc => acc.userId == user.Id && (acc.accessKey == accessKey)));
                return (
                  <tr key={user.Id}>
                    <td>{user.Id}</td>
                    <td>{user.prenom}</td>
                    <td>{user.nom}</td>
                    <td>
                      <div className="d-flex flex-wrap gap-2 align-items-center">
                        <div className="form-check form-check-inline">
                          <input
                            className="form-check-input"
                            type="checkbox"
                            id={`access-all-${user.Id}`}
                            checked={allChecked}
                            ref={el => {
                              if (el) el.indeterminate = !allChecked && someChecked;
                            }}
                            onChange={e => {
                              possibleAccess.forEach(accessObj => {
                                const accessKey = accessObj.access_code || accessObj.code || accessObj;
                                const checked = accessList.some(acc => acc.userId == user.Id && (acc.accessKey == accessKey));
                                if (e.target.checked && !checked) {
                                  handleToggleAccess(user.Id, accessKey, true);
                                } else if (!e.target.checked && checked) {
                                  handleToggleAccess(user.Id, accessKey, false);
                                }
                              });
                            }}
                            disabled={loading}
                          />
                          <label className="form-check-label" htmlFor={`access-all-${user.Id}`}>All</label>
                        </div>
                        {possibleAccess.length === 0 ? (
                          <span className="text-muted">No access types defined</span>
                        ) : (
                          possibleAccess.map(accessObj => {
                            const accessKey = accessObj.access_code || accessObj.code || accessObj;
                            const accessLabel = accessObj.name || accessKey;
                            const checked = accessList.some(acc => acc.userId == user.Id && (acc.accessKey == accessKey || acc.accessKey == accessLabel));
                            return (
                              <div className="form-check form-check-inline" key={accessKey}>
                                <input
                                  className="form-check-input"
                                  type="checkbox"
                                  id={`access-${user.Id}-${accessKey}`}
                                  checked={checked}
                                  disabled={loading}
                                  onChange={e => handleToggleAccess(user.Id, accessKey, e.target.checked)}
                                />
                                <label className="form-check-label" htmlFor={`access-${user.Id}-${accessKey}`}>{accessLabel}</label>
                              </div>
                            );
                          })
                        )}
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>
      )}

      {viewMode === 'viewAccess' && (
        <>
          <div className="row mb-3">
            <div className="col-md-6">
              <input
                type="text"
                className="form-control"
                placeholder="Search access, code, or user..."
                value={accessViewSearch}
                onChange={e => setAccessViewSearch(e.target.value)}
              />
            </div>
            <div className="col-md-6 text-end d-flex justify-content-end gap-2">
              <button className="btn btn-outline-success" onClick={handleExportCSV}>
                Export CSV
              </button>
              <button className="btn btn-outline-danger" onClick={handleExportPDF}>
                Export PDF
              </button>
            </div>
          </div>
          <div className="table-responsive">
            <table className="table table-bordered align-middle">
              <thead className="table-light">
                <tr>
                  <th>Access Name</th>
                  <th>Access Code</th>
                  <th>User Count</th>
                  <th>Users</th>
                </tr>
              </thead>
              <tbody>
                {accessUserMapFiltered.length === 0 ? (
                  <tr>
                    <td colSpan="4" className="text-center text-muted">No access types found.</td>
                  </tr>
                ) : (
                  accessUserMapFiltered.map(access => (
                    <tr key={access.access_code}>
                      <td>{access.name}</td>
                      <td>{access.access_code}</td>
                      <td>{access.users.length}</td>
                      <td>{access.users.length > 0 ? access.users.join(', ') : <span className="text-muted">No users</span>}</td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </>
      )}

      {/* Add new access name - Popup Modal */}
      {/* Modal */}
      {showAddAccessModal && (
        <div className="modal show fade" style={{display:'block', background:'rgba(0,0,0,0.3)', position:'fixed', top:0, left:0, width:'100vw', height:'100vh', zIndex:1050}} tabIndex="-1" role="dialog">
          <div className="modal-dialog modal-dialog-centered" role="document" style={{maxWidth:'400px', margin:'auto'}}>
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Add New Access</h5>
                <button type="button" className="btn-close" aria-label="Close" onClick={() => setShowAddAccessModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-3">
                  <label className="form-label">Access Name</label>
                  <input
                    type="text"
                    className="form-control"
                    placeholder="Enter new access name"
                    value={newAccessName}
                    onChange={e => setNewAccessName(e.target.value)}
                    disabled={loading}
                  />
                </div>
                <div className="mb-3">
                  <label className="form-label">Access Code (number, unique)</label>
                  <input
                    type="number"
                    className="form-control"
                    placeholder="Enter unique access code"
                    value={newAccessCode}
                    disabled={true}
                  />
                </div>
              </div>
              <div className="modal-footer d-flex justify-content-between">
                <button type="button" className="btn btn-secondary" onClick={() => setShowAddAccessModal(false)} disabled={loading}>Cancel</button>
                <button type="button" className="btn btn-primary" onClick={handleAddAccessName} disabled={loading || !newAccessName.trim() || !newAccessCode.trim()}>
                  Add Access
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default UserAccessManager;
