import React, { useEffect, useState } from 'react';
import { BASE_API } from '../../utils/config';

const TracesManager = () => {
  const [traces, setTraces] = useState([]);
  const [loading, setLoading] = useState(false);
  const [filter, setFilter] = useState({
    actor: '',
    target: '',
    action_type: '',
    details: '',
    dateFrom: '',
    dateTo: ''
  });
  const [selectedTraces, setSelectedTraces] = useState([]);
  const [reviewedTraces, setReviewedTraces] = useState([]);

  useEffect(() => {
    setLoading(true);
    fetch(`${BASE_API}/api/trace`)
      .then(res => res.json())
      .then(data => {
        setTraces(data);
        setLoading(false);
      })
      .catch(() => setLoading(false));
  }, []);

  const handleReviewTrace = (traceId) => {
    setReviewedTraces([...reviewedTraces, traceId]);
  };

  const handleSelectTrace = (traceId) => {
    setSelectedTraces(selectedTraces.includes(traceId)
      ? selectedTraces.filter(id => id !== traceId)
      : [...selectedTraces, traceId]);
  };

  const handleSelectAll = (checked) => {
    setSelectedTraces(checked ? filteredTraces.map(trace => trace.id) : []);
  };

  const handleDeleteSelected = () => {
    // For demo: just remove from UI
    setTraces(traces.filter(trace => !selectedTraces.includes(trace.id)));
    setSelectedTraces([]);
    // For real: send DELETE request to backend for each selected id
  };

  const filteredTraces = traces.filter(trace => {
    const matchActor = filter.actor ? (trace.actor || '').toLowerCase().includes(filter.actor.toLowerCase()) : true;
    const matchTarget = filter.target ? (trace.target || '').toLowerCase().includes(filter.target.toLowerCase()) : true;
    const matchType = filter.action_type ? (trace.action_type || '').toLowerCase().includes(filter.action_type.toLowerCase()) : true;
    const matchDetails = filter.details ? (trace.details || '').toLowerCase().includes(filter.details.toLowerCase()) : true;
    let matchDate = true;
    if (filter.dateFrom) {
      matchDate = matchDate && new Date(trace.timestamp) >= new Date(filter.dateFrom);
    }
    if (filter.dateTo) {
      matchDate = matchDate && new Date(trace.timestamp) <= new Date(filter.dateTo);
    }
    return matchActor && matchTarget && matchType && matchDetails && matchDate;
  });

  return (
    <div style={{padding:'32px'}}>
      <h2>Reviewable Traces</h2>
      <div style={{marginBottom:'24px', display:'flex', gap:'12px', flexWrap:'wrap'}}>
        <input type="text" placeholder="Actor" value={filter.actor} onChange={e=>setFilter(f=>({...f, actor:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc', minWidth:'120px'}} />
        <input type="text" placeholder="Target" value={filter.target} onChange={e=>setFilter(f=>({...f, target:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc', minWidth:'120px'}} />
        <input type="text" placeholder="Action Type" value={filter.action_type} onChange={e=>setFilter(f=>({...f, action_type:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc', minWidth:'120px'}} />
        <input type="text" placeholder="Details" value={filter.details} onChange={e=>setFilter(f=>({...f, details:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc', minWidth:'120px'}} />
        <input type="date" value={filter.dateFrom} onChange={e=>setFilter(f=>({...f, dateFrom:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
        <input type="date" value={filter.dateTo} onChange={e=>setFilter(f=>({...f, dateTo:e.target.value}))} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
        <button style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 18px', fontWeight:'bold', cursor:'pointer'}} disabled={selectedTraces.length===0} onClick={handleDeleteSelected}>Delete Selected</button>
      </div>
      {loading ? (
        <div>Loading traces...</div>
      ) : (
        <table style={{width:'100%', borderCollapse:'collapse', marginTop:'12px'}}>
          <thead>
            <tr>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>
                <input type="checkbox" checked={selectedTraces.length === filteredTraces.length && filteredTraces.length > 0} onChange={e=>handleSelectAll(e.target.checked)} />
              </th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Actor</th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Target</th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Action Type</th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Details</th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Timestamp</th>
              <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Review</th>
            </tr>
          </thead>
          <tbody>
            {filteredTraces.map(trace => (
              <tr key={trace.id || trace.timestamp} style={{background: reviewedTraces.includes(trace.id) ? '#eaf6ff' : 'inherit'}}>
                <td style={{border:'1px solid #eee', padding:'10px'}}>
                  <input type="checkbox" checked={selectedTraces.includes(trace.id)} onChange={()=>handleSelectTrace(trace.id)} />
                </td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>{trace.actor}</td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>{trace.target}</td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>{trace.action_type}</td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>{trace.details}</td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>{new Date(trace.timestamp).toLocaleString()}</td>
                <td style={{border:'1px solid #eee', padding:'10px'}}>
                  {reviewedTraces.includes(trace.id) ? (
                    <span style={{color:'#1565c0', fontWeight:'bold'}}>Reviewed</span>
                  ) : (
                    <button style={{background:'#a259e6', color:'#fff', border:'none', borderRadius:'6px', padding:'6px 16px', cursor:'pointer'}} onClick={()=>handleReviewTrace(trace.id)}>
                      Mark as Reviewed
                    </button>
                  )}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      )}
    </div>
  );
};

export default TracesManager;
