import React, { useEffect, useState } from 'react';
import { Editor } from '@tinymce/tinymce-react';
import { BASE_API } from '../../utils/config';

const defaultQuizSettings = {
  name: '',
  course_id: '',
  module_name: '', // Added module_name
  chapter_id: '',
  duration_minutes: 30,
  num_questions: 5,
  allowed_attempts: 1,
  unlock_next_on_success: 1,
  retry_wait_hours: 20,
};


function QuizManager() {
  const [quizzes, setQuizzes] = useState([]);
  const [selectedQuiz, setSelectedQuiz] = useState(null);
  const [quizForm, setQuizForm] = useState(defaultQuizSettings);
  const [questions, setQuestions] = useState([]);
  const [questionForm, setQuestionForm] = useState({
    content: '',
    option1: '',
    option2: '',
    option3: '',
    option4: '',
    option5: '',
    correct_answer: '',
  });
  const [showQuizForm, setShowQuizForm] = useState(false);
  const [showQuestionForm, setShowQuestionForm] = useState(false);
  const [courses, setCourses] = useState([]);
  const [modules, setModules] = useState([]);
  const [chapters, setChapters] = useState([]);

  // Fetch quizzes
  useEffect(() => {
    fetch(`${BASE_API}/api/quizzes`)
      .then(res => res.json())
      .then(data => setQuizzes(data.quizzes || []));
    fetch(`${BASE_API}/api/courses`)
      .then(res => res.json())
      .then(data => setCourses(data.courses || []));
  }, []);

  // TEMP: Fetch all modules for testing
  useEffect(() => {
    fetch(`${BASE_API}/api/debug/modules`)
      .then(res => res.json())
      .then(data => {
        console.log('DEBUG: ALL modules API response', data);
        setModules(data.modules || []);
      });
    setQuizForm(q => ({ ...q, chapter_id: '' }));
    setChapters([]);
  }, []);

  // TEMP: Fetch all chapters for testing
  useEffect(() => {
    fetch(`${BASE_API}/api/debug/chapters`)
      .then(res => res.json())
      .then(data => {
        console.log('DEBUG: ALL chapters API response', data);
        setChapters(data.chapters || []);
      });
    setQuizForm(q => ({ ...q, chapter_id: '' }));
  }, []);

  // Fetch questions for selected quiz
  useEffect(() => {
    if (selectedQuiz) {
      const params = new URLSearchParams({
        quiz_id: selectedQuiz.id,
        module_id: selectedQuiz.module_id,
        chapter_id: selectedQuiz.chapter_id
      });
      fetch(`${BASE_API}/api/quiz_questions?${params.toString()}`)
        .then(res => res.json())
        .then(data => setQuestions(data.questions || []));
    } else {
      setQuestions([]);
    }
  }, [selectedQuiz]);

  // Handle quiz form submit
  const handleQuizSubmit = (e) => {
    e.preventDefault();
    const method = quizForm.id ? 'PUT' : 'POST';
    const url = quizForm.id ? `${BASE_API}/api/quizzes/${quizForm.id}` : `${BASE_API}/api/quizzes`;
    // Prepare payload with module_name and module_id set to the selected module name
    const payload = { ...quizForm, module_id: quizForm.module_name };
    fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload),
    })
      .then(res => res.json())
      .then(() => {
        setShowQuizForm(false);
        setQuizForm(defaultQuizSettings);
        fetch(`${BASE_API}/api/quizzes`)
          .then(res => res.json())
          .then(data => setQuizzes(data.quizzes || []));
      });
  };

  // Handle question form submit
  const handleQuestionSubmit = (e) => {
    e.preventDefault();
    const method = questionForm.id ? 'PUT' : 'POST';
    const url = questionForm.id ? `${BASE_API}/api/quiz_questions/${questionForm.id}` : `${BASE_API}/api/quiz_questions`;
  // Combine options into a single string (always send 5, even if empty)
  const optionsArr = [questionForm.option1, questionForm.option2, questionForm.option3, questionForm.option4, questionForm.option5];
  const optionsStr = optionsArr.join(',');
    fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        content: questionForm.content,
        options: optionsStr,
        correct_answer: questionForm.correct_answer,
        quiz_id: selectedQuiz.id,
        course_id: selectedQuiz.course_id,
        module_id: selectedQuiz.module_id,
        chapter_id: selectedQuiz.chapter_id
      }),
    })
      .then(res => res.json())
      .then(() => {
        setShowQuestionForm(false);
        setQuestionForm({ content: '', option1: '', option2: '', option3: '', option4: '', option5: '', correct_answer: '' });
        const params = new URLSearchParams({
          quiz_id: selectedQuiz.id,
          module_id: selectedQuiz.module_id,
          chapter_id: selectedQuiz.chapter_id
        });
        fetch(`${BASE_API}/api/quiz_questions?${params.toString()}`)
          .then(res => res.json())
          .then(data => setQuestions(data.questions || []));
      });
  };

  // Render
  return (
    <div style={{ padding: '24px' }}>
      <h2 style={{ color: '#1976d2', marginBottom: '18px' }}>Quiz Management</h2>
      <button onClick={() => { setShowQuizForm(true); setQuizForm(defaultQuizSettings); }}>+ Add Quiz</button>
      <table style={{ width: '100%', marginTop: '18px', background: '#fff', borderRadius: '8px', boxShadow: '0 2px 8px rgba(0,0,0,0.08)' }}>
        <thead>
          <tr style={{ background: '#e3f2fd' }}>
            <th>Name</th>
            <th>Course</th>
            <th>Module</th>
            <th>Chapter</th>
            <th>Duration (min)</th>
            <th># Questions</th>
            <th>Attempts</th>
            <th>Unlock Next</th>
            <th>Retry Wait (h)</th>
            <th>Passing Score</th>
            <th>Created At</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          {quizzes.map(q => (
            <tr key={q.id} style={{ cursor: 'pointer', background: selectedQuiz && selectedQuiz.id === q.id ? '#bbdefb' : 'none' }}>
              <td>{q.name}</td>
              <td>{q.course_id}</td>
              <td>{q.module_id}</td>
              <td>{q.chapter_id}</td>
              <td>{q.duration_minutes}</td>
              <td>{q.num_questions}</td>
              <td>{q.allowed_attempts}</td>
              <td>{q.unlock_next_on_success ? 'Yes' : 'No'}</td>
              <td>{q.retry_wait_hours}</td>
              <td>{q.passing_score}</td>
              <td>{q.created_at}</td>
              <td>
                <button onClick={() => { setQuizForm(q); setShowQuizForm(true); }}>Edit</button>
                <button onClick={() => {
                  fetch(`${BASE_API}/api/quizzes/${q.id}`, { method: 'DELETE' })
                    .then(() => fetch(`${BASE_API}/api/quizzes`).then(res => res.json()).then(data => setQuizzes(data.quizzes || [])));
                }}>Delete</button>
                <button onClick={() => setSelectedQuiz(q)}>Questions</button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
      {/* Quiz Form Modal */}
      {showQuizForm && (
        <div style={{ position: 'fixed', top: 0, left: 0, width: '100vw', height: '100vh', background: 'rgba(0,0,0,0.18)', zIndex: 9999, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
          <form style={{ background: '#fff', borderRadius: '16px', boxShadow: '0 4px 32px rgba(25, 118, 210, 0.12)', minWidth: '480px', maxWidth: '95vw', maxHeight: '90vh', overflowY: 'auto', padding: '36px 40px', position: 'relative', display: 'flex', flexDirection: 'column', gap: '18px' }} onSubmit={handleQuizSubmit}>
            <h3 style={{ color: '#1976d2', fontWeight: 600, marginBottom: '12px', letterSpacing: '0.5px' }}>{quizForm.id ? 'Edit Quiz' : 'Add Quiz'}</h3>
            <div style={{ display: 'flex', flexDirection: 'column', gap: '16px' }}>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Name
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} placeholder="Name" value={quizForm.name} onChange={e => setQuizForm({ ...quizForm, name: e.target.value })} required />
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Course
                <select style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} value={quizForm.course_id} onChange={e => setQuizForm({ ...quizForm, course_id: e.target.value, module_name: '', chapter_id: '' })} required>
                  <option value="">Select Course</option>
                  {courses.map(course => (
                    <option key={course.id} value={course.id}>{course.course_name}</option>
                  ))}
                </select>
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Module
                <select style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} value={quizForm.module_name || ''} onChange={e => setQuizForm({ ...quizForm, module_name: e.target.value, chapter_id: '' })} required>
                  <option value="">Select Module</option>
                  {modules.map(module => (
                    <option key={module.id} value={module.name}>{module.name}</option>
                  ))}
                </select>
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Chapter
                <select style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} value={String(quizForm.chapter_id || '')} onChange={e => setQuizForm({ ...quizForm, chapter_id: e.target.value })} required>
                  <option value="">Select Chapter</option>
                  {chapters.map(chapter => (
                    <option key={chapter.id} value={String(chapter.id)}>{chapter.name}</option>
                  ))}
                </select>
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Duration (minutes)
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} type="number" placeholder="Duration (minutes)" value={quizForm.duration_minutes} onChange={e => setQuizForm({ ...quizForm, duration_minutes: e.target.value })} required />
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}># Questions
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} type="number" placeholder="# Questions" value={quizForm.num_questions} onChange={e => setQuizForm({ ...quizForm, num_questions: e.target.value })} required />
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Allowed Attempts
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} type="number" placeholder="Allowed Attempts" value={quizForm.allowed_attempts} onChange={e => setQuizForm({ ...quizForm, allowed_attempts: e.target.value })} required />
              </label>
              <label style={{ display: 'flex', alignItems: 'center', fontSize: '1rem', color: '#1976d2', fontWeight: 500, marginTop: '8px' }}>
                <input type="checkbox" checked={!!quizForm.unlock_next_on_success} onChange={e => setQuizForm({ ...quizForm, unlock_next_on_success: e.target.checked ? 1 : 0 })} style={{ marginRight: '8px' }} />
                Unlock next chapter on success
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Retry Wait (hours)
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} type="number" placeholder="Retry Wait (hours)" value={quizForm.retry_wait_hours} onChange={e => setQuizForm({ ...quizForm, retry_wait_hours: e.target.value })} required />
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Passing Score
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px' }} type="number" placeholder="Passing Score" value={quizForm.passing_score || ''} onChange={e => setQuizForm({ ...quizForm, passing_score: e.target.value })} required />
              </label>
              <label style={{ fontWeight: 500, color: '#333', marginBottom: '2px' }}>Created At
                <input style={{ width: '100%', fontSize: '1rem', padding: '8px', borderRadius: '6px', border: '1px solid #bdbdbd', marginTop: '4px', background: '#f5f5f5' }} type="text" value={quizForm.created_at || ''} readOnly />
              </label>
            </div>
            <div style={{ display: 'flex', gap: '16px', justifyContent: 'flex-end', marginTop: '18px' }}>
              <button type="submit" style={{ background: '#1976d2', color: '#fff', border: 'none', borderRadius: '6px', padding: '8px 24px', fontSize: '1rem', fontWeight: 500, cursor: 'pointer', boxShadow: '0 2px 8px rgba(25, 118, 210, 0.08)' }}>Save</button>
              <button type="button" style={{ background: '#fff', color: '#1976d2', border: '1px solid #1976d2', borderRadius: '6px', padding: '8px 24px', fontSize: '1rem', fontWeight: 500, cursor: 'pointer' }} onClick={() => setShowQuizForm(false)}>Cancel</button>
            </div>
          </form>
        </div>
      )}
      {/* Questions Table */}
      {selectedQuiz && (
        <div style={{ marginTop: '32px' }}>
          <h3>Questions for: {selectedQuiz.name}</h3>
          <button onClick={() => setShowQuestionForm(true)}>+ Add Question</button>
          <table style={{ width: '100%', marginTop: '12px', background: '#fff', borderRadius: '8px', boxShadow: '0 2px 8px rgba(0,0,0,0.08)' }}>
            <thead>
              <tr style={{ background: '#e3f2fd' }}>
                <th>Content</th>
                <th>Options</th>
                <th>Correct Answer</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {questions.map(q => (
                <tr key={q.id}>
                  <td>{q.content}</td>
                  <td>{q.options}</td>
                  <td>{q.correct_answer}</td>
                  <td>
                    <button onClick={() => {
                      // Split options string into array and assign to option1-5
                      const opts = (q.options || '').split(',');
                      setQuestionForm({
                        ...q,
                        option1: opts[0] || '',
                        option2: opts[1] || '',
                        option3: opts[2] || '',
                        option4: opts[3] || '',
                        option5: opts[4] || '',
                      });
                      setShowQuestionForm(true);
                    }}>Edit</button>
                    <button onClick={() => {
                      fetch(`${BASE_API}/api/quiz_questions/${q.id}`, { method: 'DELETE' })
                        .then(() => fetch(`${BASE_API}/api/quiz_questions?quiz_id=${selectedQuiz.id}`).then(res => res.json()).then(data => setQuestions(data.questions || [])));
                    }}>Delete</button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}
      {/* Question Form Modal */}
      {showQuestionForm && (
        <div style={{ position: 'fixed', top: 0, left: 0, width: '100vw', height: '100vh', background: 'rgba(0,0,0,0.18)', zIndex: 9999, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
          <form style={{ background: '#fff', borderRadius: '12px', boxShadow: '0 2px 24px rgba(0,0,0,0.18)', minWidth: '420px', maxWidth: '90vw', maxHeight: '90vh', overflowY: 'auto', padding: '32px', position: 'relative' }} onSubmit={handleQuestionSubmit}>
            <h3>{questionForm.id ? 'Edit Question' : 'Add Question'}</h3>
            {/* TinyMCE Editor for rich content */}
            <Editor
  apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
  value={questionForm.content}
  init={{
    height: 200,
    menubar: false,
    plugins: [
      'advlist autolink lists link image charmap print preview anchor',
      'searchreplace visualblocks code fullscreen',
      'insertdatetime media table paste code help wordcount'
    ],
    toolbar:
      'undo redo | formatselect | bold italic backcolor | \
      alignleft aligncenter alignright alignjustify | \
      bullist numlist outdent indent | removeformat | help | image',
    images_upload_url: `${BASE_API}/api/images/upload`, // Your backend image upload endpoint
    automatic_uploads: true,
    images_upload_handler: function (blobInfo, success, failure) {
      // Custom upload handler (optional, for advanced use)
    }
  }}
  onEditorChange={content => setQuestionForm({ ...questionForm, content })}
/>
            {/* Professional row for options and correct answer */}
            <div style={{ display: 'flex', gap: '12px', margin: '16px 0' }}>
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px' }} placeholder="Option 1" value={questionForm.option1} onChange={e => setQuestionForm({ ...questionForm, option1: e.target.value })} />
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px' }} placeholder="Option 2" value={questionForm.option2} onChange={e => setQuestionForm({ ...questionForm, option2: e.target.value })} />
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px' }} placeholder="Option 3" value={questionForm.option3} onChange={e => setQuestionForm({ ...questionForm, option3: e.target.value })} />
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px' }} placeholder="Option 4" value={questionForm.option4} onChange={e => setQuestionForm({ ...questionForm, option4: e.target.value })} />
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px' }} placeholder="Option 5" value={questionForm.option5} onChange={e => setQuestionForm({ ...questionForm, option5: e.target.value })} />
              <textarea style={{ flex: 1, minHeight: '40px', resize: 'vertical', fontSize: '1rem', padding: '8px', fontWeight: 'bold', borderColor: '#1976d2' }} placeholder="Correct Answer (can be long)" value={questionForm.correct_answer} onChange={e => setQuestionForm({ ...questionForm, correct_answer: e.target.value })} required />
            </div>
            {/* Removed redundant options input field. Options are now handled by the 5 textareas above. */}
            <input placeholder="Correct Answer" value={questionForm.correct_answer} onChange={e => setQuestionForm({ ...questionForm, correct_answer: e.target.value })} required />
            <div style={{ marginTop: '18px' }}>
              <button type="submit">Save</button>
              <button type="button" onClick={() => setShowQuestionForm(false)}>Cancel</button>
            </div>
          </form>
        </div>
      )}
    </div>
  );
}

export default QuizManager;
