
import React, { useEffect, useState } from 'react';
import jsPDF from 'jspdf';
import 'jspdf-autotable';
import Papa from 'papaparse';
import axios from 'axios';
import {
  Box, Typography, Button, TextField, MenuItem, Select, InputLabel, FormControl, CircularProgress, Paper, Table, TableBody, TableCell, TableContainer, TableHead, TableRow, Avatar, IconButton
} from '@mui/material';
import RefreshIcon from '@mui/icons-material/Refresh';

const PaymentUserMethod = () => {
  const [users, setUsers] = useState([]);
  const [companies, setCompanies] = useState([]);
  const [methods, setMethods] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedColumns, setSelectedColumns] = useState([
    'photo', 'id_user', 'fullname', 'email', 'amount', 'qte', 'rates', 'role', 'date_to_pay', 'paid_devise', 'bank_name', 'company_name'
  ]);
  const [filters, setFilters] = useState({
    paymentMethod: '',
    company: '',
    criteria: '',
    month: new Date().getMonth() + 1, // JS months are 0-based
    year: new Date().getFullYear(),
    search: '',
    showImages: true,
    displayRows: 25
  });

  useEffect(() => {
    setLoading(true);
    const params = new URLSearchParams({
      month: filters.month,
      year: filters.year
    });
    Promise.all([
      axios.get(`/api/payment-users?${params.toString()}`),
      axios.get('/api/companies'),
      axios.get('/api/payment-methods')
    ]).then(([usersRes, companiesRes, methodsRes]) => {
      setUsers(usersRes.data || []);
      setCompanies(companiesRes.data || []);
      setMethods(methodsRes.data || []);
    }).finally(() => setLoading(false));
  }, [filters.month, filters.year]);

  // Filtering logic
  const filteredUsers = users.filter(u => {
    // Payment method filter
    if (filters.paymentMethod && u.bank_name !== filters.paymentMethod) return false;
    // Company filter
    if (filters.company && u.code !== filters.company) return false;
    // Criteria filter (role, etc.)
    if (filters.criteria && u.role !== filters.criteria) return false;
    // Search filter
    if (filters.search && !(
      (u.nom && u.nom.toLowerCase().includes(filters.search.toLowerCase())) ||
      (u.prenom && u.prenom.toLowerCase().includes(filters.search.toLowerCase())) ||
      (u.email && u.email.toLowerCase().includes(filters.search.toLowerCase())) ||
      (u.id_user && String(u.id_user).includes(filters.search))
    )) return false;
    return true;
  });

  // Displayed rows
  const displayRows = filters.displayRows === -1 ? filteredUsers : filteredUsers.slice(0, filters.displayRows);

  // Column definitions for export and display
  const allColumns = [
    { key: 'photo', label: 'Photo' },
    { key: 'id_user', label: 'ID' },
    { key: 'fullname', label: 'Fullname' },
    { key: 'email', label: 'Email' },
    { key: 'amount', label: 'Amount' },
    { key: 'qte', label: 'Qte' },
    { key: 'rates', label: 'Rate' },
    { key: 'role', label: 'Project' },
    { key: 'date_to_pay', label: 'Date to Pay' },
    { key: 'paid_devise', label: 'Devise' },
    { key: 'bank_name', label: 'Method' },
    { key: 'company_name', label: 'Company' },
  ];

  // Export handlers
  const handleExportCSV = () => {
    const data = displayRows.map(u => {
      const row = {};
      selectedColumns.forEach(col => {
        if (col === 'fullname') {
          row['Fullname'] = `${u.nom || ''} ${u.prenom || ''}`.trim();
        } else if (col === 'photo') {
          row['Photo'] = u.img_url || '';
        } else if (col === 'amount') {
          row['Amount'] = u.amount;
        } else if (col === 'qte') {
          row['Qte'] = u.qte;
        } else if (col === 'rates') {
          row['Rate'] = u.rates;
        } else if (col === 'role') {
          row['Project'] = u.role;
        } else if (col === 'date_to_pay') {
          row['Date to Pay'] = u.date_to_pay;
        } else if (col === 'paid_devise') {
          row['Devise'] = u.paid_devise;
        } else if (col === 'bank_name') {
          row['Method'] = u.bank_name;
        } else if (col === 'company_name') {
          row['Company'] = u.company_name;
        } else {
          row[allColumns.find(c => c.key === col)?.label || col] = u[col];
        }
      });
      return row;
    });
    const csv = Papa.unparse(data);
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.setAttribute('download', 'payment_users.csv');
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  const handleExportPDF = () => {
    const doc = new jsPDF();
    const tableColumn = selectedColumns.map(col => allColumns.find(c => c.key === col)?.label || col);
    const tableRows = displayRows.map(u => selectedColumns.map(col => {
      if (col === 'fullname') return `${u.nom || ''} ${u.prenom || ''}`.trim();
      if (col === 'photo') return '';
      return u[col];
    }));
    doc.autoTable({ head: [tableColumn], body: tableRows });
    doc.save('payment_users.pdf');
  };

  return (
    <Box sx={{ p: 3 }}>
      <Typography variant="h4" gutterBottom>Search by payment method</Typography>
      {/* Export Options */}
      <Paper sx={{ p: 2, mb: 2, background: 'linear-gradient(90deg, #6a11cb 0%, #2575fc 100%)', color: '#fff' }}>
        <Box sx={{ display: 'flex', gap: 2, alignItems: 'center' }}>
          <Button variant="contained" color="success" onClick={handleExportCSV} sx={{ fontWeight: 'bold' }}>Export to XLSX</Button>
          <Button variant="contained" color="error" onClick={handleExportPDF} sx={{ fontWeight: 'bold' }}>Export to PDF</Button>
        </Box>
      </Paper>
      {loading ? (
        <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
          <CircularProgress />
        </Box>
      ) : (
        <Paper sx={{ p: 2 }}>
          {/* Filter controls */}
          <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 2, mb: 2 }}>
            {/* Payment Method Filter */}
            <FormControl sx={{ minWidth: 180 }}>
              <InputLabel>Payment Method</InputLabel>
              <Select
                value={filters.paymentMethod}
                label="Payment Method"
                onChange={e => setFilters(f => ({ ...f, paymentMethod: e.target.value }))}
              >
                <MenuItem value="">All Methods</MenuItem>
                {methods.map(m => <MenuItem key={m} value={m}>{m}</MenuItem>)}
              </Select>
            </FormControl>
            {/* Company Filter */}
            <FormControl sx={{ minWidth: 180 }}>
              <InputLabel>Company</InputLabel>
              <Select
                value={filters.company}
                label="Company"
                onChange={e => setFilters(f => ({ ...f, company: e.target.value }))}
              >
                <MenuItem value="">All Companies</MenuItem>
                {companies.map(c => <MenuItem key={c.code} value={c.code}>{c.fullname}</MenuItem>)}
              </Select>
            </FormControl>
            {/* Criteria Filter (Role) */}
            <FormControl sx={{ minWidth: 150 }}>
              <InputLabel>Criteria</InputLabel>
              <Select
                value={filters.criteria}
                label="Criteria"
                onChange={e => setFilters(f => ({ ...f, criteria: e.target.value }))}
              >
                <MenuItem value="">All Criteria</MenuItem>
                <MenuItem value="Student">Student</MenuItem>
                <MenuItem value="Interpreter">Interpreter</MenuItem>
                <MenuItem value="Administrator">Administrator</MenuItem>
                <MenuItem value="Default">Default</MenuItem>
              </Select>
            </FormControl>
            {/* Month Filter */}
            <FormControl sx={{ minWidth: 120 }}>
              <InputLabel>Month</InputLabel>
              <Select
                value={filters.month}
                label="Month"
                onChange={e => setFilters(f => ({ ...f, month: Number(e.target.value) }))}
              >
                {[...Array(12)].map((_, i) => (
                  <MenuItem key={i+1} value={i+1}>{new Date(0, i).toLocaleString('default', { month: 'long' })}</MenuItem>
                ))}
              </Select>
            </FormControl>
            {/* Year Filter */}
            <FormControl sx={{ minWidth: 100 }}>
              <InputLabel>Year</InputLabel>
              <Select
                value={filters.year}
                label="Year"
                onChange={e => setFilters(f => ({ ...f, year: Number(e.target.value) }))}
              >
                {[...Array(5)].map((_, i) => {
                  const y = new Date().getFullYear() - i;
                  return <MenuItem key={y} value={y}>{y}</MenuItem>;
                })}
              </Select>
            </FormControl>
            {/* Quick Search */}
            <TextField
              label="Quick Search"
              value={filters.search}
              onChange={e => setFilters(f => ({ ...f, search: e.target.value }))}
              sx={{ minWidth: 200 }}
            />
            {/* Rows Filter */}
            <FormControl sx={{ minWidth: 120 }}>
              <InputLabel>Rows</InputLabel>
              <Select
                value={filters.displayRows}
                label="Rows"
                onChange={e => setFilters(f => ({ ...f, displayRows: Number(e.target.value) }))}
              >
                <MenuItem value={25}>Show 25</MenuItem>
                <MenuItem value={50}>Show 50</MenuItem>
                <MenuItem value={100}>Show 100</MenuItem>
                <MenuItem value={-1}>Show All</MenuItem>
              </Select>
            </FormControl>
            {/* Column selection for export */}
            <FormControl sx={{ minWidth: 180 }}>
              <InputLabel>Select Columns</InputLabel>
              <Select
                multiple
                value={selectedColumns}
                onChange={e => setSelectedColumns(e.target.value)}
                renderValue={selected => selected.map(col => allColumns.find(c => c.key === col)?.label || col).join(', ')}
              >
                {allColumns.map(col => (
                  <MenuItem key={col.key} value={col.key}>
                    <input
                      type="checkbox"
                      checked={selectedColumns.includes(col.key)}
                      onChange={() => {}}
                      style={{ marginRight: 8 }}
                    />
                    {col.label}
                  </MenuItem>
                ))}
              </Select>
            </FormControl>
          </Box>
          {/* Table */}
          <TableContainer component={Paper}>
            <Table size="small">
              <TableHead>
                <TableRow>
                  {selectedColumns.map(col => (
                    <TableCell key={col}>{allColumns.find(c => c.key === col)?.label || col}</TableCell>
                  ))}
                </TableRow>
              </TableHead>
              <TableBody>
                {displayRows.map(u => (
                  <TableRow key={u.id_user}>
                    {selectedColumns.map(col => (
                      <TableCell key={col}>
                        {col === 'photo' ? (
                          filters.showImages ? <Avatar src={u.img_url || ''} alt={u.nom || ''} /> : null
                        ) : col === 'fullname' ? (
                          `${u.nom || ''} ${u.prenom || ''}`.trim()
                        ) : col === 'amount' ? (
                          <span style={{ color: 'green', fontWeight: 'bold' }}>${u.amount}</span>
                        ) : col === 'rates' ? (
                          <span>${u.rates}</span>
                        ) : col === 'qte' ? (
                          <span>{u.qte}</span>
                        ) : (
                          u[col]
                        )}
                      </TableCell>
                    ))}
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        </Paper>
      )}
    </Box>
  );
};

export default PaymentUserMethod;
