import React, { useEffect, useState, useRef } from 'react';
import MedicalDocumentGlossary from '../../MedicalDocumentGlossary';
import { useEffect as useEffectReact, useState as useStateReact } from 'react';
import Modal from 'react-modal';
import 'bootstrap/dist/css/bootstrap.min.css';

const MyDocument = ({ userId, role }) => {
  // For medical word count
  const [medicalWordCount, setMedicalWordCount] = useStateReact(0);
  // --- access control for glossary ---
  const [canShowGlossary, setCanShowGlossary] = useState(false);

  useEffect(() => {
    if (!userId) return;
    fetch(`/api/user_access_block/${userId}`)
      .then(res => res.json())
      .then(data => {
        // Check if access_code 126 is present
        const has126 = (data.blocks || []).some(b => b.access_code === 126);
        setCanShowGlossary(has126);
      })
      .catch(() => setCanShowGlossary(false));
  }, [userId]);

  useEffectReact(() => {
    fetch('/api/medical_words?page=1&page_size=1')
      .then(res => res.json())
      .then(data => setMedicalWordCount(data.total || 0));
  }, []);
  const [documents, setDocuments] = useState([]);
  const [selectedDoc, setSelectedDoc] = useState(null);
  const [modalOpen, setModalOpen] = useState(false);
  const [pdfFailed, setPdfFailed] = useState(false);
  const [showGlossary, setShowGlossary] = useState(false);
  const iframeRef = useRef(null);

  useEffect(() => {
    if (!userId) return;
    if (role === 'Administrator' || role === '10000') {
      fetch('/api/insights')
        .then(res => res.json())
        .then(data => setDocuments(data.insights || []));
    } else {
      // For other users, fetch assigned documents
      fetch(`/api/assigned_insights/${userId}`)
        .then(res => res.json())
        .then(data => setDocuments(data.insights || []));
    }
  }, [userId, role]);

  const handleView = doc => {
    setSelectedDoc(doc);
    setModalOpen(true);
    setPdfFailed(false);
  };

  // PDF fallback logic: if iframe is blank after 2 seconds, show fallback
  useEffect(() => {
    if (!selectedDoc || !selectedDoc.document_url) return;
    if (!(/^https?:\/\//.test(selectedDoc.document_url) || selectedDoc.document_url.startsWith('/documents/'))) return;
    setPdfFailed(false);
    const timer = setTimeout(() => {
      if (iframeRef.current && iframeRef.current.contentDocument) {
        // If PDF did not load, contentDocument will be null or empty
        const doc = iframeRef.current.contentDocument;
        if (!doc || doc.body.childNodes.length === 0) {
          setPdfFailed(true);
        }
      }
    }, 2000);
    return () => clearTimeout(timer);
  }, [selectedDoc]);

  return (
    <div className="container">
      <h3
        className="mb-4"
        style={{
          color: '#1976d2',
          fontWeight: 'bold',
          fontSize: '2.2rem',
          letterSpacing: '0.01em',
          cursor: showGlossary ? 'pointer' : 'default',
          textDecoration: showGlossary ? 'underline' : 'none',
          textAlign: 'center'
        }}
        onClick={() => showGlossary && setShowGlossary(false)}
        title={showGlossary ? "Return to document cards" : ""}
      >
        My Documents
      </h3>
      {showGlossary ? (
        <MedicalDocumentGlossary userId={userId} />
      ) : (
        <div className="row">
          {/* Medical Glossary Card - only show if access_key 126 */}
          {canShowGlossary && (
            <div className="col-12 col-sm-6 col-md-4 col-lg-3 mb-4">
              <div className="card h-100 shadow-sm">
                <div className="card-img-top d-flex align-items-center justify-content-center" style={{ background: '#e3f2fd', height: '140px' }}>
                  <img src={process.env.PUBLIC_URL + '/images/medical_glossary_Picture.png'} alt="Medical Glossary" style={{ height: '110px', width: '110px', objectFit: 'contain' }} />
                </div>
                <div className="card-body d-flex flex-column justify-content-between">
                  <h5 className="card-title text-primary">Medical Glossary</h5>
                  <p className="card-text text-secondary">Browse medical terms and translations</p>
                  <div className="d-flex justify-content-between align-items-center mt-3">
                    <span className="badge bg-warning text-dark px-3 py-2">{medicalWordCount} terms</span>
                    <button className="btn btn-dark" onClick={() => setShowGlossary(true)}>Open</button>
                  </div>
                </div>
              </div>
            </div>
          )}
          {/* Document Cards */}
          {documents.length > 0 ? documents.map(doc => (
            <div key={doc.id} className="col-12 col-sm-6 col-md-4 col-lg-3 mb-4">
              <div className="card h-100 shadow-sm">
                {doc.image_url ? (
                  <img src={doc.image_url} className="card-img-top" alt="cover" style={{ height: '140px', objectFit: 'cover' }} />
                ) : (
                  <div className="card-img-top d-flex align-items-center justify-content-center" style={{ background: '#e3f2fd', height: '140px', fontSize: '2.8rem', color: '#1976d2' }}>📄</div>
                )}
                <div className="card-body d-flex flex-column justify-content-between">
                  <h5 className="card-title text-primary" title={doc.title} style={{ whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' }}>{doc.title}</h5>
                  <div className="mb-2">
                    {doc.document_url ? (
                      <span className="text-danger fw-bold"><span style={{ fontSize: '1.3rem' }}>📄</span> PDF/Document</span>
                    ) : (
                      <span className="text-primary fw-bold"><span style={{ fontSize: '1.3rem' }}>📝</span> Content</span>
                    )}
                  </div>
                  {doc.content && (
                    <div className="text-muted mb-2" style={{ fontSize: '0.97rem', maxHeight: '40px', overflow: 'hidden', textOverflow: 'ellipsis' }} dangerouslySetInnerHTML={{ __html: doc.content }} />
                  )}
                  {doc.document_url && (
                    <div className="text-secondary mb-2" style={{ fontSize: '0.97rem', wordBreak: 'break-all' }}>{doc.document_url}</div>
                  )}
                  <div className="d-flex justify-content-end">
                    <button className="btn btn-primary" onClick={() => handleView(doc)}>View</button>
                  </div>
                </div>
              </div>
            </div>
          )) : (
            <div className="col-12 text-center text-muted mt-4" style={{ fontSize: '1.2rem' }}>
              {role === 'Administrator' || role === '10000'
                ? 'No documents available.'
                : 'You do not have any assigned documents. Please contact your administrator if you believe this is an error.'}
            </div>
          )}
        </div>
      )}
      {/* Document Modal Popup */}
      <Modal
        isOpen={modalOpen}
        onRequestClose={() => setModalOpen(false)}
        contentLabel="Document Preview"
        style={{
          content: {
            maxWidth: '700px',
            margin: 'auto',
            borderRadius: '16px',
            padding: '32px',
            boxShadow: '0 6px 32px rgba(25,118,210,0.13)',
            border: '1.5px solid #e3f2fd'
          }
        }}
      >
        <div>
          <h4 className="mb-3 text-primary">{selectedDoc?.title}</h4>
          {selectedDoc?.document_url ? (
            <iframe
              src={selectedDoc.document_url}
              title="Document"
              style={{ width: '100%', height: '400px', border: '1px solid #e3f2fd', borderRadius: '8px' }}
              ref={iframeRef}
            />
          ) : (
            <div className="mb-3" dangerouslySetInnerHTML={{ __html: selectedDoc?.content || '' }} />
          )}
          <div className="d-flex justify-content-end mt-4">
            <button className="btn btn-secondary" onClick={() => setModalOpen(false)}>Close</button>
          </div>
        </div>
      </Modal>
    </div>
  );
};

export default MyDocument;
