import React, { useEffect, useState } from 'react';
import { useSearchParams } from 'react-router-dom';

function MedicalCertificatePrintView() {
  const [searchParams] = useSearchParams();
  const cerf_id = searchParams.get('cerf_id');
  const [data, setData] = useState(null);

  useEffect(() => {
    if (cerf_id) {
      fetch(`/api/certificates?cerf_id=${cerf_id}`)
        .then(res => res.json())
        .then(res => {
          if (Array.isArray(res)) {
            setData(res[0] || null);
          } else {
            setData(res);
          }
        });
    }
  }, [cerf_id]);

  const handlePrint = () => {
    window.print();
  };

  const handleDownloadPDF = () => {
    alert('To save as PDF: Press Ctrl+P (Cmd+P on Mac), select "Save as PDF", set paper size to 11x8.5in Landscape, set margins to None, enable Background graphics, then Save.');
    setTimeout(() => window.print(), 500);
  };

  function formatDate(dateStr) {
    if (!dateStr) return '';
    const d = new Date(dateStr);
    if (isNaN(d)) return dateStr;
    return d.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });
  }

  function getExpirationDate(dateStr) {
    if (!dateStr) return '';
    const d = new Date(dateStr);
    if (isNaN(d)) return '';
    d.setFullYear(d.getFullYear() + 2);
    return formatDate(d.toISOString());
  }

  if (!data) return <div>Loading...</div>;

  // Dynamic fields
  const recipientName = data?.fullname || 'Name not found';
  const certificateId = data?.cerf_id || 'Certificate ID';
  const issueDate = formatDate(data?.date_validity);
  const validThrough = getExpirationDate(data?.date_validity);
  const languagePair = `BETWEEN ${data?.language_from?.toUpperCase()} AND ${data?.language_to?.toUpperCase()}`;
  const qrCodeSrc = data?.qr_code_filename ? data.qr_code_filename : `/Company/qr_certificate/qr_medical_${cerf_id}.png`;
  const logoSrc = process.env.PUBLIC_URL + '/images/make-it_services_logo.png';
  const signatureSrc = process.env.PUBLIC_URL + '/images/dr_daniel_charles_signature.png';

  return (
    <div>
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Roboto:wght@300;400;500&display=swap');
        body { font-family: 'Times New Roman', serif; background: #fff; width: 11in; height: 8.5in; margin: 0; padding: 0; position: relative; overflow: hidden; }
        .certificate-container { width: 100%; height: 100%; background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); position: relative; padding: 15px; box-sizing: border-box; }
        .decorative-corner { position: absolute; width: 40px; height: 40px; background: #000000; z-index: 10; }
        .corner-tl { top: 15px; left: 15px; clip-path: polygon(0 0, 100% 0, 0 100%); }
        .corner-tr { top: 15px; right: 15px; clip-path: polygon(0 0, 100% 0, 100% 100%); }
        .corner-bl { bottom: 15px; left: 15px; clip-path: polygon(0 0, 0 100%, 100% 100%); }
        .corner-br { bottom: 15px; right: 15px; clip-path: polygon(0 100%, 100% 0, 100% 100%); }
  .inner-certificate { background: #ffffff; width: 100%; height: 750px; border: 2px solid #000; position: relative; padding: 30px 40px; box-sizing: border-box; }
        .inner-border { border: 1px solid #ccc; height: calc(100% - 16px); width: calc(100% - 16px); position: absolute; top: 8px; left: 8px; pointer-events: none; }
        .header { text-align: center; margin-bottom: 25px; position: relative; z-index: 5; }
        .title { font-size: 28px; font-weight: bold; color: #333; text-transform: uppercase; letter-spacing: 4px; margin-bottom: 5px; line-height: 1.1; }
        .subtitle { font-size: 28px; color: #333; text-transform: uppercase; letter-spacing: 4px; font-weight: bold; }
        .recipient-name { font-size: 36px; font-weight: normal; color: #333; margin: 30px 0 25px; font-family: 'Brush Script MT', cursive; text-align: center; font-style: italic; position: relative; z-index: 5; }
        .completion-text { font-size: 16px; color: #333; margin: 20px 40px; font-weight: normal; text-transform: none; letter-spacing: normal; text-align: center; position: relative; z-index: 5; line-height: 1.5; font-style: italic; }
        .language-pair { font-size: 20px; color: #dc2626; font-weight: bold; margin: 15px 0; text-align: center; position: relative; z-index: 5; text-transform: uppercase; letter-spacing: 1px; }
        .requirements-text { font-size: 12px; color: #555; margin: 25px 50px; line-height: 1.4; font-style: italic; text-align: center; position: relative; z-index: 5; }
        .date-certified { font-size: 15px; color: #333; font-weight: bold; margin: 10px 0 0 0; text-align: center; position: relative; z-index: 5; }
        .certificate-id { font-size: 15px; color: #333; font-weight: bold; margin: 10px 0 0 0; text-align: center; position: relative; z-index: 5; }
  .footer-section { position: absolute; bottom: 110px; left: 50px; right: 50px; display: flex; justify-content: space-between; align-items: flex-end; z-index: 5; }
        .qr-section { display: flex; flex-direction: column; align-items: center; border: 3px solid #dc2626; padding: 8px; background: #fff; border-radius: 5px; width: 100px; }
        .qr-code { width: 70px; height: 70px; background: #f0f0f0; border: 1px solid #ccc; display: flex; align-items: center; justify-content: center; margin-bottom: 5px; border-radius: 3px; }
        .qr-code img { width: 68px; height: 68px; object-fit: contain; }
        .scan-text { color: #dc2626; font-weight: bold; font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px; }
        .signature-block { display: flex; flex-direction: column; align-items: flex-end; }
        .signature-image-container { height: 50px; display: flex; align-items: center; margin-bottom: -8px; }
        .legal-signature-img { max-width: 120px; max-height: 45px; opacity: 0.8; }
        .signature-line-legal { width: 140px; height: 1px; background-color: #333; margin: 5px 0 10px 0; }
        .legal-signatory-info { text-align: right; }
        .signatory-name { font-family: Arial, sans-serif; font-size: 12px; font-weight: 600; color: #333; margin-bottom: 3px; }
        .signatory-title { font-family: Arial, sans-serif; font-size: 10px; color: #666; margin-bottom: 2px; }
        .company-footer { position: absolute; bottom: 25px; left: 50%; transform: translateX(-50%); font-size: 16px; font-weight: bold; color: #333; text-transform: uppercase; letter-spacing: 2px; text-align: center; z-index: 5; }
  .center-logo { position: absolute; left: 50%; bottom: 60px; transform: translateX(-50%); text-align: center; z-index: 3; }
        .company-logo { max-width: 120px; max-height: 80px; object-fit: contain; }
        .print-button-container { position: fixed; top: 20px; right: 20px; z-index: 1000; display: flex; gap: 10px; }
        .print-btn { background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; font-family: 'Arial', sans-serif; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 8px; box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3); transition: all 0.3s ease; text-decoration: none; letter-spacing: 0.5px; text-transform: uppercase; }
        .print-btn:hover { background: linear-gradient(135deg, #b91c1c 0%, #991b1b 100%); transform: translateY(-2px); box-shadow: 0 6px 16px rgba(220, 38, 38, 0.4); }
        .print-btn:active { transform: translateY(0); box-shadow: 0 2px 8px rgba(220, 38, 38, 0.3); }
        .print-icon { font-size: 16px; font-weight: bold; }
        .download-btn { background: linear-gradient(135deg, #059669 0%, #047857 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; font-family: 'Arial', sans-serif; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 8px; box-shadow: 0 4px 12px rgba(5, 150, 105, 0.3); transition: all 0.3s ease; text-decoration: none; letter-spacing: 0.5px; text-transform: uppercase; }
        .download-btn:hover { background: linear-gradient(135deg, #047857 0%, #065f46 100%); transform: translateY(-2px); box-shadow: 0 6px 16px rgba(5, 150, 105, 0.4); }
        .download-btn:active { transform: translateY(0); box-shadow: 0 2px 8px rgba(5, 150, 105, 0.3); }
        @media print { .print-button-container { display: none !important; } body { background: white !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } .certificate-container { background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%) !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } .decorative-corner { background: #000000 !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } }
      `}</style>
      <div className="print-button-container">
        <button className="print-btn" onClick={handlePrint}>
          <span className="print-icon">🖨️</span> Print Certificate
        </button>
        <button className="download-btn" onClick={handleDownloadPDF}>
          <span className="print-icon">📄</span> Save as PDF
        </button>
      </div>
      <div className="certificate-container">
        <div className="decorative-corner corner-tl"></div>
        <div className="decorative-corner corner-tr"></div>
        <div className="decorative-corner corner-bl"></div>
        <div className="decorative-corner corner-br"></div>
        <div className="inner-certificate">
          <div className="inner-border"></div>
          <div className="header">
            <div className="title">PROFESSIONAL</div>
            <div className="subtitle">INTERPRETER CERTIFICATE</div>
          </div>
          <div className="recipient-name">{recipientName}</div>
          <div className="completion-text">
            To successfully complete the 60-hour Medical Interpreter Training course and to demonstrate proficiency in professional medical interpretation.
          </div>
          <div className="language-pair">{languagePair}</div>
          <div className="requirements-text">
            This certificate acknowledges completion of all training requirements including medical terminology, cultural competency, ethics, and practical interpretation skills as established by professional standards for medical interpreters.
          </div>
          <div className="date-certified">Issued on {issueDate}</div>
          <div className="date-certified">Valid Through {validThrough}</div>
          <div className="certificate-id">ID No. {certificateId}</div>
          <div className="center-logo">
            <img src={logoSrc} alt="Make-IT Services Logo" className="company-logo" />
          </div>
          <div className="footer-section">
            <div className="qr-section">
              <div className="qr-code">
                <img src={qrCodeSrc} alt="QR Code" />
              </div>
              <div className="scan-text">SCAN ME!</div>
            </div>
            <div className="signature-block">
              <div className="signature-image-container">
                <img src={signatureSrc} alt="Signature" className="legal-signature-img" />
              </div>
              <div className="signature-line-legal"></div>
              <div className="legal-signatory-info">
                <div className="signatory-name">Dr. Daniel CHARLES</div>
                <div className="signatory-title">Compliance and Training</div>
              </div>
            </div>
          </div>
          <div className="company-footer">MAKE-IT SERVICES LEARNING CENTER</div>
        </div>
      </div>
    </div>
  );
}

export default MedicalCertificatePrintView;
