import React, { useEffect, useState } from 'react';
import { useSearchParams } from 'react-router-dom';

function LegalCertificatePrintView() {
  const [searchParams] = useSearchParams();
  const cerf_id = searchParams.get('cerf_id');
  const [data, setData] = useState(null);

  useEffect(() => {
    if (cerf_id) {
      fetch(`/api/certificates?cerf_id=${cerf_id}`)
        .then(res => res.json())
        .then(res => {
          // If response is array, use first item; else use object
          if (Array.isArray(res)) {
            setData(res[0] || null);
          } else {
            setData(res);
          }
        });
    }
  }, [cerf_id]);

  const handlePrint = () => {
    window.print();
  };

  const handleDownloadPDF = () => {
    alert('To save as PDF: Press Ctrl+P (Cmd+P on Mac), select "Save as PDF", set paper size to 11x8.5in Landscape, set margins to None, enable Background graphics, then Save.');
    setTimeout(() => window.print(), 500);
  };

  // Helper to format date
  function formatDate(dateStr) {
    if (!dateStr) return '';
    const d = new Date(dateStr);
    if (isNaN(d)) return dateStr;
    return d.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });
  }

  // Helper to calculate expiration date (add 2 years)
  function getExpirationDate(dateStr) {
    if (!dateStr) return '';
    const d = new Date(dateStr);
    if (isNaN(d)) return '';
    d.setFullYear(d.getFullYear() + 2);
    return formatDate(d.toISOString());
  }

  if (!data) return <div>Loading...</div>;

  return (
    <div>
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Roboto:wght@300;400;500&display=swap');
        body {
          font-family: 'Roboto', sans-serif;
          background: white;
          width: 11in;
          height: 8.5in;
          margin: 0;
          padding: 0.5in;
          display: flex;
          align-items: center;
          justify-content: center;
        }
        .certificate-container {
          background: white;
          width: 10in;
          height: 7.5in;
          border: 15px solid #800080;
          border-radius: 20px;
          position: relative;
          overflow: hidden;
          display: flex;
          flex-direction: column;
        }
        .watermark {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%) rotate(-45deg);
          font-size: 6rem;
          color: rgba(0,0,0,0.03);
          z-index: 1;
          font-family: 'Playfair Display', serif;
          letter-spacing: 0.2em;
          font-weight: 900;
        }
        .certificate-header {
          background: linear-gradient(135deg, #800080 0%, #9370DB 100%);
          color: white;
          text-align: center;
          padding: 20px 20px 15px 20px;
          position: relative;
          z-index: 2;
        }
        .certificate-title {
          font-family: 'Playfair Display', serif;
          font-size: 2.0rem;
          font-weight: 700;
          margin-bottom: 8px;
          color: #fff;
        }
        .certificate-subtitle {
          font-size: 1.0rem;
          font-weight: 300;
          opacity: 0.9;
          color: #fff;
        }
        .certificate-body {
          padding: 25px 40px;
          text-align: center;
          position: relative;
          flex: 1;
          display: flex;
          flex-direction: column;
          justify-content: space-between;
          z-index: 2;
        }
        .main-content {
          flex: 1;
          display: flex;
          flex-direction: column;
          justify-content: center;
        }
        .certificate-text {
          font-size: 1.0rem;
          line-height: 1.6;
          color: #333;
          margin-bottom: 15px;
        }
        .recipient-name {
          font-family: 'Playfair Display', serif;
          font-size: 2.4rem;
          font-weight: 700;
          color: #800080;
          margin: 2px 0;
          text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
          border-bottom: 3px solid #9370DB;
          padding-bottom: 10px;
          display: inline-block;
        }
        .legal-simple-container {
          margin-top: 20px;
        }
        .legal-subtitle {
          font-family: 'Playfair Display', serif;
          font-size: 1.2rem;
          color: #800080;
          font-weight: 600;
          margin-bottom: 20px;
        }
        .legal-verification-text {
          font-size: 0.95rem;
          color: #555;
          line-height: 1.6;
          margin-bottom: 25px;
          max-width: 85%;
          margin-left: auto;
          margin-right: auto;
        }
        .legal-signature-section {
          display: flex;
          justify-content: space-between;
          align-items: center;
          margin-top: 20px;
          background: linear-gradient(135deg, rgba(128, 0, 128, 0.03) 0%, rgba(147, 112, 219, 0.05) 100%);
          border-radius: 15px;
          padding: 20px;
          border: 2px solid #800080;
        }
        .legal-signature-left {
          flex: 1;
          text-align: center;
        }
        .signature-image-container {
          margin-bottom: -22px;
        }
        .legal-signature-img {
          max-width: 280px;
          max-height: 50px;
          opacity: 1.1;
        }
        .signature-line-legal {
          width: 300px;
          height: 1px;
          background-color: #000;
          margin: 0 auto 8px auto;
        }
        .legal-signatory-info {
          text-align: center;
        }
        .signatory-name {
          font-family: Arial, sans-serif;
          font-size: 14px;
          font-weight: bold;
          color: #000;
          margin-bottom: 3px;
        }
        .signatory-title {
          font-family: Arial, sans-serif;
          font-size: 12px;
          color: #666;
          line-height: 1.3;
        }
        .signatory-org {
          font-family: Arial, sans-serif;
          font-size: 10px;
          color: #666;
          font-weight: 500;
        }
        .signature-dates-container {
          margin-top: 15px;
          display: flex;
          justify-content: center;
          gap: 20px;
        }
        .signature-date-item {
          background: white;
          border: 2px solid #800080;
          border-radius: 25px;
          padding: 8px 16px;
          text-align: center;
          box-shadow: 0 2px 8px rgba(128, 0, 128, 0.15);
          min-width: 120px;
        }
        .signature-date-label {
          font-family: 'Playfair Display', serif;
          font-size: 0.75rem;
          font-weight: 700;
          color: #800080;
          text-transform: uppercase;
          letter-spacing: 1px;
          margin-bottom: 2px;
        }
        .signature-date-value {
          font-family: 'Roboto', sans-serif;
          font-size: 0.85rem;
          font-weight: 600;
          color: #333;
        }
        .legal-qr-center {
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          margin: 0 20px;
        }
        .qr-code-legal {
          width: 80px;
          height: 80px;
          border: 3px solid #800080;
          border-radius: 8px;
        }
        .qr-text {
          font-size: 0.8rem;
          color: #666;
          margin: 5px 0;
        }
        .legal-info-right {
          flex: 1;
          text-align: center;
        }
        .legal-info-box {
          background: white;
          border-radius: 12px;
          padding: 15px;
          box-shadow: 0 4px 15px rgba(128, 0, 128, 0.1);
          border: 1px solid #800080;
        }
        .legal-info-title {
          font-family: 'Playfair Display', serif;
          color: #800080;
          font-size: 1rem;
          font-weight: 700;
          text-transform: uppercase;
          letter-spacing: 1px;
          margin-bottom: 8px;
        }
        .legal-info-desc {
          font-family: 'Roboto', sans-serif;
          font-size: 0.75rem;
          color: #666;
          line-height: 1.4;
          margin-bottom: 8px;
        }
        .legal-info-number {
          font-family: 'Roboto', sans-serif;
          font-size: 0.8rem;
          color: #800080;
          font-weight: 600;
        }
        .print-button-container {
          position: fixed;
          top: 20px;
          right: 20px;
          z-index: 1000;
          display: flex;
          gap: 10px;
        }
        .print-btn {
          background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%);
          color: white;
          border: none;
          padding: 12px 24px;
          border-radius: 8px;
          font-family: 'Arial', sans-serif;
          font-size: 14px;
          font-weight: 600;
          cursor: pointer;
          display: flex;
          align-items: center;
          gap: 8px;
          box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3);
          transition: all 0.3s ease;
          text-decoration: none;
          letter-spacing: 0.5px;
          text-transform: uppercase;
        }
        .print-btn:hover {
          background: linear-gradient(135deg, #b91c1c 0%, #991b1b 100%);
          transform: translateY(-2px);
          box-shadow: 0 6px 16px rgba(220, 38, 38, 0.4);
        }
        .print-btn:active {
          transform: translateY(0);
          box-shadow: 0 2px 8px rgba(220, 38, 38, 0.3);
        }
        .print-icon {
          font-size: 16px;
          font-weight: bold;
        }
        .download-btn {
          background: linear-gradient(135deg, #059669 0%, #047857 100%);
          color: white;
          border: none;
          padding: 12px 24px;
          border-radius: 8px;
          font-family: 'Arial', sans-serif;
          font-size: 14px;
          font-weight: 600;
          cursor: pointer;
          display: flex;
          align-items: center;
          gap: 8px;
          box-shadow: 0 4px 12px rgba(5, 150, 105, 0.3);
          transition: all 0.3s ease;
          text-decoration: none;
          letter-spacing: 0.5px;
          text-transform: uppercase;
        }
        .download-btn:hover {
          background: linear-gradient(135deg, #047857 0%, #065f46 100%);
          transform: translateY(-2px);
          box-shadow: 0 6px 16px rgba(5, 150, 105, 0.4);
        }
        .download-btn:active {
          transform: translateY(0);
          box-shadow: 0 2px 8px rgba(5, 150, 105, 0.3);
        }
        .view-btn {
          background: linear-gradient(135deg, #800080 0%, #9370DB 100%);
          color: white;
          border: none;
          padding: 12px 24px;
          border-radius: 8px;
          font-family: 'Arial', sans-serif;
          font-size: 14px;
          font-weight: 600;
          cursor: pointer;
          display: flex;
          align-items: center;
          gap: 8px;
          box-shadow: 0 4px 12px rgba(128, 0, 128, 0.3);
          transition: all 0.3s ease;
          text-decoration: none;
          letter-spacing: 0.5px;
          text-transform: uppercase;
        }
        .view-btn:hover {
          background: linear-gradient(135deg, #9370DB 0%, #663399 100%);
          transform: translateY(-2px);
          box-shadow: 0 6px 16px rgba(128, 0, 128, 0.4);
        }
        .view-btn:active {
          transform: translateY(0);
          box-shadow: 0 2px 8px rgba(128, 0, 128, 0.3);
        }
        @media print {
          .print-button-container { display: none !important; }
          body { background: white !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; }
          .certificate-container { background: white !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; }
        }
        @media print {
          @page { size: 11in 8.5in landscape; margin: 0.5in; }
          body { -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; }
        }
      `}</style>
      <div className="print-button-container">
        <button className="print-btn" onClick={handlePrint}>
          <span className="print-icon">🖨️</span> Print Certificate
        </button>
        <button className="download-btn" onClick={handleDownloadPDF}>
          <span className="print-icon">📄</span> Save as PDF
        </button>
        <a href={`https://school.makeitservices.org/verify/legal/Legal_certificate_html/legal_certificate_${cerf_id}.html`} target="_blank" className="view-btn">
          <span className="print-icon">👁️</span> View Certificate
        </a>
      </div>
      <div className="certificate-container">
        <div className="watermark">MAKE-IT SERVICES</div>
        <div className="certificate-header">
          <h1 className="certificate-title">Legal Interpreter Certification</h1>
          <p className="certificate-subtitle">40 Hours Legal Interpreter Program</p>
        </div>
        <div className="certificate-body">
          <div className="main-content">
            <p className="certificate-text">This is to certify that</p>
            <div className="recipient-name">{data?.fullname || 'Name not found'}</div>
            <p className="certificate-text">This certificate confirms that the individual has successfully completed the 40-hour Legal Interpreter certification program and is qualified to provide legal interpretation services.</p>
            <div className="legal-simple-container">
              <div className="legal-subtitle">Legal Interpretation in {data.language_from} and {data.language_to}</div>
              <div className="legal-verification-text">This certificate verifies passing scores in MAKE-IT Legal Center' written exam and legal interpretation exam (verbal)</div>
              <div className="legal-signature-section">
                <div className="legal-signature-left">
                  <div className="signature-image-container">
                    <img src={data.signature_url || '/images/ernston_signature_legal.png'} alt="Signature" className="legal-signature-img" />
                  </div>
                  <div className="signature-line-legal"></div>
                  <div className="legal-signatory-info">
                    <div className="signatory-name">Ernston St Pierre</div>
                    <div className="signatory-title">Lead Legal Trainer</div>
                    <div className="signatory-org">MAKE-IT SERVICES TECHNOLOGY legal Center.</div>
                  </div>
                  <div className="signature-dates-container">
                    <div className="signature-date-item">
                      <div className="signature-date-label">Issue #</div>
                      <div className="signature-date-value">{formatDate(data.date_validity)}</div>
                    </div>
                    <div className="signature-date-item">
                      <div className="signature-date-label">Expiration</div>
                      <div className="signature-date-value">{getExpirationDate(data.date_validity)}</div>
                    </div>
                  </div>
                </div>
                <div className="legal-qr-center">
                  <img src={data.qr_code_filename ? data.qr_code_filename : `/Company/qr_certificate/qr_legal_${cerf_id}.png`} alt="Verification QR Code" className="qr-code-legal" />
                  <div className="qr-text">Scan to Verify</div>
                </div>
                <div className="legal-info-right">
                  <div className="legal-info-box">
                    <div className="legal-info-title">Advancing Justice Through Legal Education®</div>
                    <div className="legal-info-desc">is developed and maintained by MAKE-IT LEGAL CENTER, committed to excellence in legal training and aligned with best practices recognized by leading U.S. legal organizations</div>
                    <div className="legal-info-number">Certificate #: {data.cerf_id}</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}

export default LegalCertificatePrintView;
