
import React, { useEffect, useState } from 'react';
import { useSearchParams } from 'react-router-dom';

// const signatureImg = require('/images/dr_daniel_charles_signature.png');

function HipaaCertificatePrintView() {
  const [searchParams] = useSearchParams();
  const cerf_id = searchParams.get('cerf_id');
  const [data, setData] = useState(null);
  const [showPDFModal, setShowPDFModal] = useState(false);

  useEffect(() => {
    if (cerf_id) {
      fetch(`/api/certificates?cerf_id=${cerf_id}`)
        .then(res => res.json())
        .then(res => {
          if (Array.isArray(res)) {
            setData(res[0] || null);
          } else {
            setData(res);
          }
        });
    }
  }, [cerf_id]);

  if (!data) return <div>Loading...</div>;

  // Dynamic fields
  const recipientName = data.fullname || 'Recipient Name';
  const certificateId = data.cerf_id || 'Certificate ID';
  let issueDate = data.date_validity || 'Issue Date';
  if (issueDate && typeof issueDate === 'string' && issueDate.length >= 10) {
    // Try to format ISO date string
    const d = new Date(issueDate);
    if (!isNaN(d)) {
      issueDate = d.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' });
    }
  }
  const languagePair = (data.language_from && data.language_to) ? `${data.language_from} - ${data.language_to}` : 'Language Pair';
  const validity = 'Valid for 1 year from issue date';
  const qrCodeSrc = data.qr_code_filename || '';

  // Print and PDF logic
  const handlePrint = () => window.print();
  const handleDownloadPDF = () => setShowPDFModal(true);
  const handleStartPrintProcess = () => {
    setShowPDFModal(false);
    setTimeout(() => window.print(), 300);
  };

  return (
    <div>
      {/* Print/Download/View Buttons */}
      <div className="print-button-container">
        <button className="print-btn" onClick={handlePrint}>
          <span className="print-icon">🖨️</span>
          Print Certificate
        </button>
        <button className="download-btn" onClick={handleDownloadPDF}>
          <span className="print-icon">📄</span>
          Save as PDF
        </button>
        <a
          href={qrCodeSrc.replace('qr_certificate/qr_', 'Hipaa_certificate_html/hipaa_certificate_print_') + '.html'}
          className="view-btn"
          target="_blank"
          rel="noopener noreferrer"
        >
          <span className="print-icon">👁️</span>
          View Certificate
        </a>
      </div>

      {/* Certificate Layout */}
      <div className="certificate-container">
        <div className="watermark">CERTIFIED</div>
        <div className="certificate-header">
          <h1 className="certificate-title">HIPAA Compliance Certification</h1>
          <p className="certificate-subtitle">Health Insurance Portability and Accountability Act</p>
        </div>
        <div className="certificate-body">
          <div className="hipaa-main-content">
            <p className="certificate-text">This is to certify that</p>
            <div className="recipient-name">{recipientName}</div>
            <p className="certificate-text">
              This certificate confirms that the individual has successfully completed HIPAA compliance training and understands the requirements for protecting patient health information.
            </p>
            <div className="certificate-details">
              <div className="detail-item">
                <div className="detail-label">Certificate ID</div>
                <div className="detail-value">{certificateId}</div>
              </div>
              <div className="detail-item">
                <div className="detail-label">Issue Date</div>
                <div className="detail-value">{issueDate}</div>
              </div>
              <div className="detail-item">
                <div className="detail-label">Language Pair</div>
                <div className="detail-value">{languagePair}</div>
              </div>
              <div className="detail-item">
                <div className="detail-label">Validity</div>
                <div className="detail-value">{validity}</div>
              </div>
            </div>
            <div className="company-name">MAKE-IT LEARNING CENTER</div>
            <div className="legal-signature-section">
              <div className="legal-signature-left">
                <div className="signature-image-container">
                  <img src={process.env.PUBLIC_URL + '/images/dr_daniel_charles_signature.png'} alt="Signature" className="legal-signature-img" />
                </div>
                <div className="signature-line-legal"></div>
                <div className="legal-signatory-info">
                  <div className="signatory-name">Dr. Daniel CHARLES</div>
                  <div className="signatory-title">Compliance and Training</div>
                </div>
              </div>
              <div className="legal-qr-center">
                {qrCodeSrc && (
                  <img src={qrCodeSrc} alt="Verification QR Code" className="qr-code" />
                )}
                <div className="qr-text">Scan to Verify</div>
              </div>
              <div className="legal-info-right">
                <div className="legal-info-box">
                  <div className="legal-info-title">HIPAA Compliance Certified</div>
                  <div className="legal-info-desc">Privacy and Security Training Completed</div>
                  <div className="legal-info-number">Certificate #: {certificateId}</div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* PDF Modal */}
      {showPDFModal && (
        <div id="pdf-modal" style={{
          position: 'fixed', top: 0, left: 0, width: '100%', height: '100%', background: 'rgba(0,0,0,0.8)', zIndex: 10000,
          display: 'flex', alignItems: 'center', justifyContent: 'center', fontFamily: 'Arial, sans-serif'
        }}>
          <div style={{background: 'white', padding: 30, borderRadius: 12, maxWidth: 500, boxShadow: '0 20px 40px rgba(0,0,0,0.3)'}}>
            <div style={{textAlign: 'center', marginBottom: 20}}>
              <h2 style={{color: '#dc2626', margin: '0 0 10px 0', fontSize: 24}}>📄 Save as PDF</h2>
              <p style={{color: '#666', margin: 0, fontSize: 16}}>Follow these steps to save your HIPAA certificate as PDF</p>
            </div>
            <ol style={{color: '#333', fontSize: 14, lineHeight: 1.6, margin: '20px 0', paddingLeft: 20}}>
              <li><strong>Press Ctrl+P</strong> (or Cmd+P on Mac)</li>
              <li>Select <strong>Save as PDF</strong> as destination</li>
              <li>Set paper size to <strong>11 x 8.5 inches Landscape</strong></li>
              <li>Set margins to <strong>None</strong> or <strong>Minimum</strong></li>
              <li>Enable <strong>Background graphics</strong></li>
              <li>Click <strong>Save</strong> or <strong>Print</strong></li>
            </ol>
            <div style={{display: 'flex', gap: 10, justifyContent: 'center', marginTop: 25}}>
              <button onClick={handleStartPrintProcess} style={{background: 'linear-gradient(135deg, #dc2626 0%, #b91c1c 100%)', color: 'white', border: 'none', padding: '12px 24px', borderRadius: 8, fontWeight: 600, cursor: 'pointer', fontFamily: 'Arial, sans-serif'}}>Start Print Process</button>
              <button onClick={() => setShowPDFModal(false)} style={{background: '#6b7280', color: 'white', border: 'none', padding: '12px 24px', borderRadius: 8, fontWeight: 600, cursor: 'pointer', fontFamily: 'Arial, sans-serif'}}>Close</button>
            </div>
          </div>
        </div>
      )}

      {/* Styles */}
      <style>{`
        @import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Roboto:wght@300;400;500&display=swap');
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Roboto', sans-serif; background: white; width: 11in; height: 8.5in; margin: 0; padding: 0.5in; display: flex; align-items: center; justify-content: center; }
        .certificate-container { background: white; width: 10in; height: 7.5in; border: 15px solid #4169E1; border-radius: 20px; position: relative; overflow: hidden; display: flex; flex-direction: column; }
        .watermark { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%) rotate(-45deg); font-size: 6rem; color: rgba(0,0,0,0.03); z-index: 1; font-family: 'Playfair Display', serif; letter-spacing: 0.2em; font-weight: 900; }
        .certificate-header { background: linear-gradient(135deg, #4169E1 0%, #1E90FF 100%); color: white; text-align: center; padding: 25px 20px 20px 20px; position: relative; z-index: 2; }
    .certificate-title { font-family: 'Playfair Display', serif; font-size: 2.2rem; font-weight: 700; margin-bottom: 8px; text-shadow: 2px 2px 4px rgba(0,0,0,0.3); color: #fff; }
  .certificate-subtitle { font-size: 1.1rem; font-weight: 300; opacity: 0.9; color: #fff; }
        .certificate-body { padding: 30px 40px; text-align: center; position: relative; flex: 1; display: flex; flex-direction: column; justify-content: space-between; z-index: 2; }
        .hipaa-main-content { flex: 1; display: flex; flex-direction: column; justify-content: flex-start; padding-top: 20px; }
        .certificate-text { font-size: 1.1rem; line-height: 1.6; color: #333; margin-bottom: 10px; margin-top: -10px; }
        .recipient-name { font-family: 'Playfair Display', serif; font-size: 2.4rem; font-weight: 700; color: #4169E1; margin: 10px 0; text-shadow: 1px 1px 2px rgba(0,0,0,0.1); border-bottom: 3px solid #1E90FF; padding-bottom: 10px; display: inline-block; }
        .certificate-details { display: grid; grid-template-columns: 1fr 1fr 1fr 1fr; gap: 20px; margin: 25px 0 10px 0; text-align: left; }
        .detail-item { background: #F0F8FF; padding: 15px; border-radius: 10px; border-left: 5px solid #4169E1; }
        .detail-label { font-weight: 600; color: #4169E1; margin-bottom: 5px; text-transform: uppercase; font-size: 0.8rem; letter-spacing: 1px; }
        .detail-value { font-size: 1rem; color: #333; font-weight: 500; }
        .company-name { font-family: 'Playfair Display', serif; font-size: 0.9rem; color: #4169E1; font-weight: 600; margin-bottom: 15px; }
        .legal-signature-section { display: flex; justify-content: space-between; align-items: center; margin-top: -5px; background: linear-gradient(135deg, rgba(65, 105, 225, 0.03) 0%, rgba(30, 144, 255, 0.05) 100%); border-radius: 15px; padding: 20px; border: 2px solid #4169E1; }
        .legal-signature-left { flex: 1; text-align: center; }
        .signature-image-container { margin-bottom: -20px; }
        .legal-signature-img { max-width: 180px; max-height: 50px; opacity: 0.8; }
        .signature-line-legal { width: 300px; height: 1px; background-color: #000; margin: 0 auto 8px auto; }
        .legal-signatory-info { text-align: center; }
        .signatory-name { font-family: Arial, sans-serif; font-size: 14px; font-weight: bold; color: #000; margin-bottom: 3px; }
        .signatory-title { font-family: Arial, sans-serif; font-size: 12px; color: #666; line-height: 1.3; }
        .legal-qr-center { display: flex; flex-direction: column; align-items: center; justify-content: center; margin: 0 20px; }
        .qr-code { width: 80px; height: 80px; border: 3px solid #4169E1; border-radius: 8px; }
        .qr-text { font-size: 0.7rem; color: #666; margin-top: 5px; }
        .legal-info-right { flex: 1; text-align: center; }
        .legal-info-box { background: white; border-radius: 12px; padding: 15px; box-shadow: 0 4px 15px rgba(65, 105, 225, 0.1); border: 1px solid #4169E1; }
        .legal-info-title { font-family: 'Playfair Display', serif; color: #4169E1; font-size: 1rem; font-weight: 700; text-transform: uppercase; letter-spacing: 1px; margin-bottom: 8px; }
        .legal-info-desc { font-family: 'Roboto', sans-serif; font-size: 0.85rem; color: #666; line-height: 1.4; margin-bottom: 8px; }
        .legal-info-number { font-family: 'Roboto', sans-serif; font-size: 0.8rem; color: #4169E1; font-weight: 600; }
        .print-button-container { position: fixed; top: 20px; right: 20px; z-index: 1000; display: flex; gap: 10px; }
        .print-btn { background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; font-family: 'Arial', sans-serif; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 8px; box-shadow: 0 4px 12px rgba(220, 38, 38, 0.3); transition: all 0.3s ease; text-decoration: none; letter-spacing: 0.5px; text-transform: uppercase; }
        .print-btn:hover { background: linear-gradient(135deg, #b91c1c 0%, #991b1b 100%); transform: translateY(-2px); box-shadow: 0 6px 16px rgba(220, 38, 38, 0.4); }
        .print-btn:active { transform: translateY(0); box-shadow: 0 2px 8px rgba(220, 38, 38, 0.3); }
        .print-icon { font-size: 16px; font-weight: bold; }
        .download-btn { background: linear-gradient(135deg, #059669 0%, #047857 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; font-family: 'Arial', sans-serif; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 8px; box-shadow: 0 4px 12px rgba(5, 150, 105, 0.3); transition: all 0.3s ease; text-decoration: none; letter-spacing: 0.5px; text-transform: uppercase; }
        .download-btn:hover { background: linear-gradient(135deg, #047857 0%, #065f46 100%); transform: translateY(-2px); box-shadow: 0 6px 16px rgba(5, 150, 105, 0.4); }
        .download-btn:active { transform: translateY(0); box-shadow: 0 2px 8px rgba(5, 150, 105, 0.3); }
        .view-btn { background: linear-gradient(135deg, #4169E1 0%, #1E90FF 100%); color: white; border: none; padding: 12px 24px; border-radius: 8px; font-family: 'Arial', sans-serif; font-size: 14px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 8px; box-shadow: 0 4px 12px rgba(65, 105, 225, 0.3); transition: all 0.3s ease; text-decoration: none; letter-spacing: 0.5px; text-transform: uppercase; }
        .view-btn:hover { background: linear-gradient(135deg, #1E90FF 0%, #0056b3 100%); transform: translateY(-2px); box-shadow: 0 6px 16px rgba(65, 105, 225, 0.4); }
        .view-btn:active { transform: translateY(0); box-shadow: 0 2px 8px rgba(65, 105, 225, 0.3); }
        @media print { .print-button-container { display: none !important; } body { background: white !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } .certificate-container { background: white !important; -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } }
        @media print { @page { size: 11in 8.5in landscape; margin: 0.5in; } body { -webkit-print-color-adjust: exact; color-adjust: exact; print-color-adjust: exact; } }
      `}</style>
    </div>
  );
}

export default HipaaCertificatePrintView;
