import React, { useState, useEffect, useRef } from 'react';
import axios from 'axios';

const sampleQuiz = [
  {
    question: 'What is the main topic of this chapter?',
    options: ['Medical Interpreter', 'Haitian Creole', 'Quiz Example', 'Other'],
    answer: 0,
  },
  {
    question: 'Is this course for Haitian people?',
    options: ['Yes', 'No'],
    answer: 0,
  },
];

const ChapterQuiz = ({ quiz = sampleQuiz, quizSettings = {}, userId, onComplete }) => {
  const [shuffledQuiz, setShuffledQuiz] = useState([]);
  const [answers, setAnswers] = useState([]);
  const [submitted, setSubmitted] = useState(false);
  const [score, setScore] = useState(0);
  const [latestResult, setLatestResult] = useState(null);
  const [timeLeft, setTimeLeft] = useState(quizSettings.duration_minutes ? quizSettings.duration_minutes * 60 : null);
  const [attempts, setAttempts] = useState(0);
  const [retryAvailableAt, setRetryAvailableAt] = useState(null);
  const [currentIdx, setCurrentIdx] = useState(0);
  const [maxAttemptsReached, setMaxAttemptsReached] = useState(false);
  const [quizStarted, setQuizStarted] = useState(false);
  const timerRef = useRef();
  // Shuffle questions and initialize answers
  useEffect(() => {
    // Fetch latest quiz result and quiz settings from backend
    const fetchQuizMeta = async () => {
      try {
        // Fetch latest result for this user and quiz
        const res = await axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${quizSettings.id}`);
        if (res.data && res.data.attempts !== undefined) {
          setAttempts(res.data.attempts);
          setLatestResult(res.data);
          if (quizSettings.allowed_attempts && res.data.attempts >= quizSettings.allowed_attempts) {
            setMaxAttemptsReached(true);
          } else {
            setMaxAttemptsReached(false);
          }
        }
      } catch (err) {
        // If no result, allow quiz
        setMaxAttemptsReached(false);
      }
    };
    fetchQuizMeta();
    // Shuffle quiz array
    const shuffled = [...quiz].sort(() => Math.random() - 0.5);
    setShuffledQuiz(shuffled);
    setAnswers(Array(shuffled.length).fill(null));
    setCurrentIdx(0);
  }, [quiz, quizSettings.id, quizSettings.allowed_attempts, userId]);

  // Timer logic
  useEffect(() => {
    if (timeLeft === null || submitted) return;
    timerRef.current = setInterval(() => {
      setTimeLeft(t => {
        if (t > 0) return t - 1;
        clearInterval(timerRef.current);
        setSubmitted(true);
        return 0;
      });
    }, 1000);
    return () => clearInterval(timerRef.current);
  }, [timeLeft, submitted]);

  // Attempt/retry logic
  useEffect(() => {
    if (submitted && quizSettings.retry_wait_hours && attempts >= quizSettings.allowed_attempts) {
      const now = Date.now();
      setRetryAvailableAt(now + quizSettings.retry_wait_hours * 60 * 60 * 1000);
    }
  }, [submitted, attempts, quizSettings.retry_wait_hours, quizSettings.allowed_attempts]);

  const handleSelect = (qIdx, optIdx) => {
  if (submitted) return;
  const newAnswers = [...answers];
  newAnswers[qIdx] = optIdx;
  setAnswers(newAnswers);
  };

  const handleSubmit = async () => {
    let correct = 0;
    quiz.forEach((q, idx) => {
      if (answers[idx] === q.answer || (q.correct_answer && q.options[answers[idx]] === q.correct_answer)) correct++;
    });
  // Calculate score as percentage out of 100
  const totalScore = quiz.length > 0 ? Math.round((correct / quiz.length) * 100) : 0;
  setScore(totalScore);
    setSubmitted(true);
    setAttempts(a => a + 1);
    // Save result to backend
    try {
      await axios.post('/api/quiz_result', {
        user_id: userId,
        quiz_id: quizSettings.id,
        score: totalScore,
        attempts: attempts + 1,
        is_passed: totalScore >= quizSettings.passing_score ? 1 : 0
      });
      // Fetch latest result from backend
      const res = await axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${quizSettings.id}`);
      setLatestResult(res.data);
    } catch (err) {
      // Handle error if needed
    }
    // Do NOT call onComplete here; user must manually close the quiz after seeing the result
  };

  // Format timer
  const formatTime = secs => {
    const m = Math.floor(secs / 60);
    const s = secs % 60;
    return `${m}:${s < 10 ? '0' : ''}${s}`;
  };

  // Retry logic
  const canRetry = !retryAvailableAt || Date.now() > retryAvailableAt;

  // Handler for closing the quiz (calls parent onComplete if provided)
  const handleCloseQuiz = () => {
    if (onComplete) onComplete('closed');
  };

  return (
    <div style={{ marginTop: 32, background: '#fff', borderRadius: 12, boxShadow: '0 2px 12px rgba(0,0,0,0.08)', padding: 24 }}>
      <h3 style={{ color: '#1976d2', marginBottom: 18 }}>Quiz</h3>
      {/* Start Assessment button logic */}
      {!quizStarted && (
        <div style={{ display: 'flex', flexDirection: 'column', alignItems: 'center', gap: 16 }}>
          <button
            type="button"
            onClick={() => setQuizStarted(true)}
            disabled={maxAttemptsReached}
            style={{ padding: '12px 32px', background: maxAttemptsReached ? '#888' : '#1976d2', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 18, cursor: maxAttemptsReached ? 'not-allowed' : 'pointer', marginBottom: 12 }}
          >
            Start Assessment
          </button>
          {maxAttemptsReached && (
            <button
              type="button"
              onClick={handleCloseQuiz}
              style={{ padding: '10px 24px', background: '#888', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer' }}
            >
              Close Quiz
            </button>
          )}
          {maxAttemptsReached && (
            <div style={{ marginTop: 12, color: '#d32f2f', fontWeight: 'bold', fontSize: 18 }}>
              You have reached the maximum number of attempts for this quiz.
            </div>
          )}
        </div>
      )}
      {/* Quiz UI only appears after starting */}
      {quizStarted && !maxAttemptsReached && (
        <>
          {quizSettings.duration_minutes && !submitted && (
            <div style={{ fontWeight: 'bold', color: '#d32f2f', marginBottom: 12 }}>Time Left: {formatTime(timeLeft)}</div>
          )}
          {quizSettings.allowed_attempts && (
            <div style={{ fontSize: 14, color: '#888', marginBottom: 8 }}>Attempt {attempts + 1} of {quizSettings.allowed_attempts}</div>
          )}
          {/* Show only one question at a time */}
          {shuffledQuiz.length > 0 && !submitted && (
            <div style={{ marginBottom: 18 }}>
              <div style={{ fontWeight: 'bold', marginBottom: 8 }}>
                {currentIdx + 1}. <span dangerouslySetInnerHTML={{ __html: shuffledQuiz[currentIdx].content || shuffledQuiz[currentIdx].question }} />
              </div>
              <div style={{ display: 'flex', flexDirection: 'column', gap: 8 }}>
                {shuffledQuiz[currentIdx].options
                  .map((opt, oIdx) => ({ opt, oIdx }))
                  .filter(({ opt }) => opt && opt.trim() !== '')
                  .map(({ opt, oIdx }) => (
                    <label key={oIdx} style={{ display: 'flex', alignItems: 'center', cursor: submitted ? 'not-allowed' : 'pointer' }}>
                      <input
                        type="radio"
                        name={`q${currentIdx}`}
                        checked={answers[currentIdx] === oIdx}
                        disabled={submitted}
                        onChange={() => handleSelect(currentIdx, oIdx)}
                        style={{ marginRight: 8 }}
                      />
                      {opt}
                    </label>
                  ))}
              </div>
            </div>
          )}
          {/* ...existing code for navigation and submit buttons... */}
          {!submitted && shuffledQuiz.length > 0 ? (
            <div>
              {currentIdx < shuffledQuiz.length - 1 ? (
                <button
                  type="button"
                  onClick={() => setCurrentIdx(currentIdx + 1)}
                  disabled={answers[currentIdx] === null}
                  style={{ padding: '10px 24px', background: '#1976d2', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer', marginTop: 12 }}
                >
                  Next
                </button>
              ) : (
                <button
                  type="button"
                  onClick={handleSubmit}
                  disabled={answers.some(a => a === null) || (quizSettings.allowed_attempts && attempts >= quizSettings.allowed_attempts)}
                  style={{ padding: '10px 24px', background: '#1976d2', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer', marginTop: 12 }}
                >
                  Submit Quiz
                </button>
              )}
            </div>
          ) : (
            <>
              {latestResult && (
                <div style={{ marginTop: 18, fontWeight: 'bold', fontSize: 18, color: latestResult.is_passed ? '#27ae60' : '#f75c7a' }}>
                  Points Earned: {latestResult.score} / 100 <br />
                  Passing Score: {quizSettings.passing_score} <br />
                  {latestResult.is_passed ? '🎉 Passed!' : 'Try again next time!'}
                </div>
              )}
              {/* Show Try Again and Close Quiz if more attempts left, else only Close Quiz */}
              {quizSettings.allowed_attempts && attempts < quizSettings.allowed_attempts && canRetry ? (
                <div style={{ display: 'flex', gap: '16px', marginTop: 12 }}>
                  <button
                    type="button"
                    onClick={() => {
                      setAnswers(Array(shuffledQuiz.length).fill(null));
                      setSubmitted(false);
                      setScore(0);
                      setLatestResult(null);
                      setTimeLeft(quizSettings.duration_minutes ? quizSettings.duration_minutes * 60 : null);
                      setCurrentIdx(0);
                    }}
                    style={{ padding: '10px 24px', background: '#43cea2', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer' }}
                  >
                    Try Again
                  </button>
                  <button
                    type="button"
                    onClick={handleCloseQuiz}
                    style={{ padding: '10px 24px', background: '#888', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer' }}
                  >
                    Close Quiz
                  </button>
                </div>
              ) : (
                <button
                  type="button"
                  onClick={handleCloseQuiz}
                  style={{ padding: '10px 24px', background: '#888', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, cursor: 'pointer', marginTop: 12 }}
                >
                  Close Quiz
                </button>
              )}
              {quizSettings.retry_wait_hours && !canRetry && (
                <div style={{ marginTop: 12, color: '#d32f2f', fontWeight: 'bold' }}>
                  You can retry after {new Date(retryAvailableAt).toLocaleTimeString()}.
                </div>
              )}
            </>
          )}
        </>
      )}
    </div>
  );
};

export default ChapterQuiz;
