import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Editor } from '@tinymce/tinymce-react';


const AddCourseContent = () => {
  const [courses, setCourses] = useState([]);
  const [selectedCourseId, setSelectedCourseId] = useState('');
  const [moduleName, setModuleName] = useState('');
  const [modules, setModules] = useState([]);
  const [chapterName, setChapterName] = useState('');
  const [chapterContent, setChapterContent] = useState('');
  const [chapterInsight, setChapterInsight] = useState('');
  const [selectedModule, setSelectedModule] = useState('');
  const [documentFile, setDocumentFile] = useState(null);
  const [imageFile, setImageFile] = useState(null);
  const [videoFile, setVideoFile] = useState(null);
  const [audioFile, setAudioFile] = useState(null);
  const [chapters, setChapters] = useState([]);
  const [documents, setDocuments] = useState([]);
  const [images, setImages] = useState([]);
  const [videos, setVideos] = useState([]);
  const [audios, setAudios] = useState([]);

  // Fetch courses on mount
  useEffect(() => {
    axios.get('/api/courses').then(res => {
      setCourses(res.data.courses || []);
    });
  }, []);

  // Fetch modules for selected course
  useEffect(() => {
    if (selectedCourseId) {
      axios.get(`/api/modules/${selectedCourseId}`).then(res => {
        setModules(res.data.modules);
      });
    } else {
      setModules([]);
    }
  }, [selectedCourseId]);

  React.useEffect(() => {
    axios.get('/api/images').then(res => setImages(res.data.images || []));
    axios.get('/api/videos').then(res => setVideos(res.data.videos || []));
    axios.get('/api/audios').then(res => setAudios(res.data.audios || []));
    axios.get('/documents/all').then(res => setDocuments(res.data.documents || []));
  }, []);

  // Add module
  const handleAddModule = async () => {
    if (!selectedCourseId) {
      alert('Please select a course!');
      return;
    }
    if (!moduleName) {
      alert('Please enter a module name!');
      return;
    }
    await axios.post('/api/modules', { course_id: selectedCourseId, name: moduleName });
    setModuleName('');
    // Refresh modules
    const res = await axios.get(`/api/modules/${selectedCourseId}`);
    setModules(res.data.modules);
  };

  // Add chapter
  const handleAddChapter = async () => {
    if (!selectedModule || !chapterName) return;
    let documentUrl = documentFile; // No longer uploading, just using the selected ID
    let imageUrl = imageFile;
    let videoUrl = videoFile;
    let audioUrl = audioFile;
    await axios.post('/api/chapters', {
      module_id: selectedModule,
      name: chapterName,
      content: chapterContent,
      insight: chapterInsight,
      document_url: documentUrl,
      image_url: imageUrl,
      video_url: videoUrl,
      audio_url: audioUrl,
    });
    setChapterName('');
    setChapterContent('');
    setChapterInsight('');
    setDocumentFile(null);
    setImageFile(null);
    setVideoFile(null);
    setAudioFile(null);
    // Refresh chapters
    const res = await axios.get(`/api/chapters/${selectedModule}`);
    setChapters(res.data.chapters);
  };

  // Fetch chapters when module changes
  React.useEffect(() => {
    if (selectedModule) {
      axios.get(`/api/chapters/${selectedModule}`).then(res => {
        setChapters(res.data.chapters);
      });
    } else {
      setChapters([]);
    }
  }, [selectedModule]);

  return (
    <div style={{ maxWidth: 800, margin: '0 auto', background: '#fff', borderRadius: 12, boxShadow: '0 2px 12px rgba(0,0,0,0.08)', padding: 32 }}>
      <h2 style={{ color: '#1976d2', marginBottom: 24 }}>Add Course Content</h2>
      <div style={{ marginBottom: 32 }}>
        <h3 style={{ color: '#1565c0', marginBottom: 18 }}>Add Module</h3>
        <div style={{ display: 'flex', alignItems: 'center', gap: 18, marginBottom: 18 }}>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 220 }}>
            <label htmlFor="course-select" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Select a course</label>
            <select
              id="course-select"
              value={selectedCourseId}
              onChange={e => setSelectedCourseId(e.target.value)}
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 16, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            >
              <option value="">-- Select a course --</option>
              {courses.map(course => (
                <option key={course.course_id} value={course.course_id}>
                  {course.course_name}
                </option>
              ))}
            </select>
          </div>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 220 }}>
            <label htmlFor="module-name" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Module Name</label>
            <input
              id="module-name"
              type="text"
              value={moduleName}
              onChange={e => setModuleName(e.target.value)}
              placeholder="Enter module name"
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 16, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            />
          </div>
          <button
            onClick={handleAddModule}
            style={{ padding: '12px 28px', background: 'linear-gradient(90deg,#1976d2 60%,#2196f3 100%)', color: '#fff', border: 'none', borderRadius: 8, fontWeight: 'bold', fontSize: 16, boxShadow: '0 2px 8px rgba(25,118,210,0.08)', cursor: 'pointer', marginTop: 22 }}
          >
            Add Module
          </button>
        </div>
        <div style={{ marginTop: 12 }}>
          <strong style={{ color: '#1565c0' }}>Modules:</strong>
          <ul style={{ marginTop: 6 }}>
            {modules.map(m => (
              <li key={m.id} style={{ margin: '4px 0', fontSize: 15, color: '#333' }}>{m.name}</li>
            ))}
          </ul>
        </div>
      </div>
      <div style={{ marginBottom: 32 }}>
        <h3>Add Chapter</h3>
        <div style={{ marginTop: 12, marginBottom: 18, display: 'flex', gap: 24, flexWrap: 'wrap', alignItems: 'flex-end' }}>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 180 }}>
            <label htmlFor="document-select" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Document</label>
            <select
              id="document-select"
              value={documentFile}
              onChange={e => setDocumentFile(e.target.value)}
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 15, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            >
              <option value="">Select Document</option>
              {documents.map(doc => (
                <option key={doc.id} value={doc.id}>{doc.document_content?.slice(0,30) || 'Document ' + doc.id}</option>
              ))}
            </select>
          </div>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 180 }}>
            <label htmlFor="image-select" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Image</label>
            <select
              id="image-select"
              value={imageFile}
              onChange={e => setImageFile(e.target.value)}
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 15, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            >
              <option value="">Select Image</option>
              {images.map(img => (
                <option key={img.id} value={img.id}>{img.image_name}</option>
              ))}
            </select>
          </div>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 180 }}>
            <label htmlFor="video-select" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Video</label>
            <select
              id="video-select"
              value={videoFile}
              onChange={e => setVideoFile(e.target.value)}
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 15, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            >
              <option value="">Select Video</option>
              {videos.map(vid => (
                <option key={vid.id} value={vid.id}>{vid.video_name}</option>
              ))}
            </select>
          </div>
          <div style={{ display: 'flex', flexDirection: 'column', minWidth: 180 }}>
            <label htmlFor="audio-select" style={{ fontWeight: 'bold', marginBottom: 6, color: '#333' }}>Audio</label>
            <select
              id="audio-select"
              value={audioFile}
              onChange={e => setAudioFile(e.target.value)}
              style={{ padding: '10px 12px', borderRadius: 6, border: '1px solid #bdbdbd', fontSize: 15, background: '#f9f9f9', color: '#333', fontWeight: 500 }}
            >
              <option value="">Select Audio</option>
              {audios.map(aud => (
                <option key={aud.id} value={aud.id}>{aud.audio_name}</option>
              ))}
            </select>
          </div>
        </div>
        <select value={selectedModule} onChange={e => setSelectedModule(e.target.value)} style={{ padding: 8, width: 320, marginBottom: 12 }}>
          <option value="">Select Module</option>
          {modules.map(m => (
            <option key={m.id} value={m.id}>{m.name}</option>
          ))}
        </select>
        <input
          type="text"
          value={chapterName}
          onChange={e => setChapterName(e.target.value)}
          placeholder="Chapter Name"
          style={{ padding: 8, width: 300, marginRight: 12 }}
        />
        <div style={{marginBottom: 16}}>
          <label style={{fontWeight:'bold', display:'block', marginBottom: 4}}>Chapter Content (Document):</label>
          <Editor
            apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
            value={chapterContent}
            onEditorChange={setChapterContent}
            init={{
              height: 120,
              menubar: false,
              plugins: [
                'advlist autolink lists link image charmap print preview anchor',
                'searchreplace visualblocks code fullscreen',
                'insertdatetime media table paste code help wordcount'
              ],
              toolbar:
                'undo redo | formatselect | bold italic backcolor | \
                alignleft aligncenter alignright alignjustify | \
                bullist numlist outdent indent | removeformat | help'
            }}
          />
        </div>
        <div style={{marginBottom: 16}}>
          <label style={{fontWeight:'bold', display:'block', marginBottom: 4}}>Chapter Insight (Rich Text):</label>
          <Editor
            apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
            value={chapterInsight}
            onEditorChange={setChapterInsight}
            init={{
              height: 120,
              menubar: false,
              plugins: [
                'advlist autolink lists link image charmap print preview anchor',
                'searchreplace visualblocks code fullscreen',
                'insertdatetime media table paste code help wordcount'
              ],
              toolbar:
                'undo redo | formatselect | bold italic backcolor | \
                alignleft aligncenter alignright alignjustify | \
                bullist numlist outdent indent | removeformat | help'
            }}
          />
        </div>
        <button onClick={handleAddChapter} style={{ padding: '8px 16px', background: '#1976d2', color: '#fff', border: 'none', borderRadius: 6, marginTop: 12 }}>Add Chapter</button>
        <div style={{ marginTop: 12 }}>
          <strong>Chapters:</strong>
          <ul>
            {chapters.map(c => (
              <li key={c.id} style={{ margin: '4px 0' }}>{c.name}</li>
            ))}
          </ul>
        </div>
      </div>
    </div>
  );
};

export default AddCourseContent;
