import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { useNavigate } from 'react-router-dom';
import { BASE_API } from '../../utils/config';
import { AppBar, Toolbar, Typography, Button, IconButton, Drawer, List, ListItem, ListItemText, Avatar, Badge, Menu, MenuItem, Dialog, DialogTitle, DialogContent, DialogActions } from '@mui/material';
import MenuIcon from '@mui/icons-material/Menu';
import NotificationsIcon from '@mui/icons-material/Notifications';
import DashboardSidebar from '../DashboardSidebar';
import CourseDashboard from '../Administrator/CourseDashboard';
import MyDocument from '../Administrator/MyDocument';
import MyTutorial from '../MyTutorial';
import AnnouncementPage from '../AnnouncementPage';
import ProfilePage from '../ProfilePage';
import { logAction } from '../../utils/actionLogger';
import { BarChart, Bar, XAxis, YAxis, Tooltip, ResponsiveContainer, CartesianGrid, Legend, LabelList, Cell } from 'recharts';

const monthNames = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
const monthColors = [
  '#1976d2', '#43cea2', '#ffa726', '#8e24aa', '#fbc02d', '#e53935', '#27ae60', '#a259e6', '#90caf9', '#185a9d', '#f75c7a', '#b2ebf2'
];

// Announcement type color mapping
const announcementTypeColors = {
  Info: '#1976d2',
  Warning: '#ffa726',
  Alert: '#e53935',
  Update: '#43cea2',
  'Payment Delay': '#fbc02d',
  'User Supension': '#8e24aa',
  Payroll: '#27ae60',
  Account: '#a259e6',
  User: '#90caf9',
};

// Severity color mapping for expiration
const expirationSeverityColors = {
  expired: '#e53935', // red
  twoDays: '#ffa726', // orange
  almost: '#fbc02d',  // yellow
  normal: 'green',      // default
};

// Helper to get expiration severity and message
function getExpirationSeverity(expirationDate) {
  if (!expirationDate) return { severity: 'normal', message: '' };
  const today = new Date();
  const expDate = new Date(expirationDate);
  today.setHours(0,0,0,0);
  expDate.setHours(0,0,0,0);
  const diffDays = Math.ceil((expDate - today) / (1000 * 60 * 60 * 24));
  if (diffDays < 0) {
    return { severity: 'expired', message: 'Expired' };
  } else if (diffDays === 0) {
    return { severity: 'expired', message: 'Expired today' };
  } else if (diffDays === 1) {
    return { severity: 'twoDays', message: 'Expires tomorrow' };
  } else if (diffDays === 2) {
    return { severity: 'twoDays', message: 'Expires in two days' };
  } else if (diffDays >= 3 && diffDays <= 5) {
    return { severity: 'almost', message: 'Almost expired' };
  } else {
    return { severity: 'normal', message: '' };
  }
}

const Index10000 = () => {
  const [role, setRole] = useState('Default');
  const [access, setAccess] = useState([]);
  const [selectedMenu, setSelectedMenu] = useState('dashboard');
  const [userId, setUserId] = useState(null);
  const [idUser, setIdUser] = useState(null);
  const [chartData, setChartData] = useState([]);
  const [selectedMonth, setSelectedMonth] = useState(new Date().getMonth());
  const [selectedYear, setSelectedYear] = useState(new Date().getFullYear());
  const [earnings, setEarnings] = useState({ earning: 0, total_minutes: 0, rate: 0, type: '' });
  const [announcementPopupOpen, setAnnouncementPopupOpen] = useState(false);
  const [popupAnnouncements, setPopupAnnouncements] = useState([]);
  const [recentActions, setRecentActions] = useState([]);
  const [actionsLoading, setActionsLoading] = useState(false);
  const [actionsError, setActionsError] = useState('');
  const [newsBlog, setNewsBlog] = useState([]);
  const [newsBlogLoading, setNewsBlogLoading] = useState(false);
  const [newsBlogError, setNewsBlogError] = useState('');
  const navigate = useNavigate();

  // Fetch recent actions for the current user
  useEffect(() => {
    if (!userId) return;
    setActionsLoading(true);
    setActionsError('');
    fetch(`${BASE_API}/api/recent-actions?user_id=${userId}`)
      .then(res => {
        if (!res.ok) throw new Error('Failed to fetch recent actions');
        return res.json();
      })
      .then(data => {
        setRecentActions(Array.isArray(data.actions) ? data.actions : []);
        setActionsLoading(false);
      })
      .catch(err => {
        setActionsError('Could not load recent actions.');
        setActionsLoading(false);
      });
  }, [userId]);

  useEffect(() => {
    // Get role from localStorage
    const userRole = localStorage.getItem('role') || 'Default';
    setRole(userRole);
  // Get userId and id_user from localStorage (or context)
  const uid = localStorage.getItem('userId');
  setUserId(uid);
  const idu = localStorage.getItem('id_user');
  setIdUser(idu);
    // Role-based access setup
    let accessArr = ['dashboard'];
    if (['Administrator', 'Student', 'Interpreter', 'Staff', 'bpo', 'BPO', 'Translator'].includes(userRole)) {
      accessArr.push('courses');
    }
    if (userRole === 'Student') {
      accessArr.push('my_progress');
    }
    if (userRole === 'Interpreter') {
      accessArr.push('tasks');
    }
    if (userRole === 'Administrator') {
      accessArr.push('edit_user', 'admin_settings');
    }
    setAccess(accessArr);
  }, []);

  useEffect(() => {
    // Fetch interpreter chart data for user
    if (userId) {
      fetch(`${BASE_API}/api/interpreter_chart/${userId}`)
        .then(res => res.json())
        .then(data => {
          // Convert chart object to array for Recharts
          const arr = Object.entries(data.chart || {}).map(([key, value], idx) => {
            const [year, month] = key.split('-');
            const mIdx = parseInt(month,10)-1;
            return {
              name: `${monthNames[mIdx]} ${year}`,
              minutes: value,
              month: mIdx,
              year: parseInt(year,10),
              fill: monthColors[mIdx]
            };
          });
          setChartData(arr);
        })
        .catch(err => console.error('Error fetching chart data:', err));
      // Fetch earnings for selected month/year
      fetch(`${BASE_API}/api/interpreter_earnings/${userId}/${selectedMonth+1}/${selectedYear}`)
        .then(res => res.json())
        .then(data => {
          setEarnings(data);
        })
        .catch(err => console.error('Error fetching earnings:', err));
    }
  }, [userId, selectedMonth, selectedYear]);

  useEffect(() => {
    // Fetch announcements and show popup if public or targeted
    const fetchAnnouncements = async () => {
      try {
        const res = await fetch('/api/announcements');
        const data = await res.json();
        const announcements = data.announcements || [];
        const userEmail = localStorage.getItem('email') || '';
        const userIdStr = String(userId);
        const visible = announcements.filter(a => {
          if (a.is_active !== 1) return false; // Filter out inactive announcements
          if (!a.target_roles?.trim() && !a.target_users?.trim()) return true; // Public
          if (a.target_users && (a.target_users.split(',').map(u => u.trim()).includes(userIdStr) || a.target_users.split(',').map(u => u.trim()).includes(userEmail))) return true;
          return false;
        });
        if (visible.length > 0) {
          setPopupAnnouncements(visible);
          setAnnouncementPopupOpen(true);
        }
      } catch (err) {}
    };
    if (selectedMenu === 'dashboard') {
      fetchAnnouncements();
    }
  }, [selectedMenu, userId]);

  // Fetch news/blog posts for sidebar
  useEffect(() => {
    setNewsBlogLoading(true);
    setNewsBlogError('');
    fetch('/api/news_blog')
      .then(res => {
        if (!res.ok) throw new Error('Failed to fetch news/blog');
        return res.json();
      })
      .then(data => {
        setNewsBlog(Array.isArray(data) ? data : []);
        setNewsBlogLoading(false);
      })
      .catch(err => {
        setNewsBlogError('Could not load news/blog posts.');
        setNewsBlogLoading(false);
      });
  }, []);

  // Calculate percent change from previous month
  const prevMonth = selectedMonth === 0 ? 11 : selectedMonth - 1;
  const prevYear = selectedMonth === 0 ? selectedYear - 1 : selectedYear;
  const prevMonthData = chartData.find(d => d.month === prevMonth && d.year === prevYear);
  const currMonthData = chartData.find(d => d.month === selectedMonth && d.year === selectedYear);
  let percentChange = null;
  if (prevMonthData && currMonthData) {
    if (prevMonthData.minutes === 0 && currMonthData.minutes > 0) {
      percentChange = 100;
    } else if (prevMonthData.minutes > 0) {
      percentChange = ((currMonthData.minutes - prevMonthData.minutes) / prevMonthData.minutes) * 100;
    } else {
      percentChange = 0;
    }
  }

  // Calculate earning based on rate type
  let earningValue = 0;
  let minutes = Number(earnings.minutes_worked);
  const rate = Number(earnings.rate) || 0;
  if (!minutes) {
    // fallback to chartData for current month
    const currMonthData = chartData.find(d => d.month === selectedMonth && d.year === selectedYear);
    minutes = currMonthData ? Number(currMonthData.minutes) : 0;
  }
  if (earnings) {
    if (earnings.type === 'minute') {
      earningValue = minutes * rate;
    } else if (earnings.type === 'hourly') {
      earningValue = (minutes / 60) * rate;
    }
  }

  // Sidebar menu items by role
  const menuItems = [
    { key: 'dashboard', label: 'Dashboard' },
    ...(access.includes('courses') ? [{ key: 'courses', label: 'Courses' }] : []),
    { key: 'my_tutorial', label: 'My Tutorial' },
    { key: 'mydocument', label: 'My Document' },
    { key: 'profile', label: 'Profile' },
    ...(access.includes('my_progress') ? [{ key: 'my_progress', label: 'My Progress' }] : []),
    ...(access.includes('tasks') ? [{ key: 'tasks', label: 'Tasks' }] : []),
    ...(access.includes('edit_user') ? [{ key: 'edit_user', label: 'Edit User' }] : []),
    ...(access.includes('admin_settings') ? [{ key: 'admin_settings', label: 'Admin Settings' }] : []),
  ];

  // Responsive top bar menu state
  const [drawerOpen, setDrawerOpen] = useState(false);
  const [profileOpen, setProfileOpen] = useState(false);
  const [showProfilePage, setShowProfilePage] = useState(false);
  // Avatar state for ProfilePage
  const [avatar, setAvatar] = useState('/avatar.png');
  const handleAvatarChange = (imgUrl) => {
    setAvatar(imgUrl || '/avatar.png');
  };

  // Fetch avatar on initial load (match Administrator logic)
  useEffect(() => {
    const userId = localStorage.getItem('userId');
    if (!userId) return;
    (async () => {
      try {
        const res = await fetch(`/api/users/by-iduser/${userId}`);
        const data = await res.json();
        if (data.user && data.user.img) {
          const imgPath = data.user.img.startsWith('/profile/') ? BASE_API + data.user.img : data.user.img;
          setAvatar(imgPath);
        }
      } catch (e) {
        // ignore
      }
    })();
  }, []);
  // No topMenuItems for top bar
  const topMenuItems = [];

  return (
    <>
      <AppBar position="fixed" color="default" elevation={2} sx={{ zIndex: 1201 }}>
        <Toolbar>
          <Typography variant="h6" sx={{ flexGrow: 1, fontWeight: 700, color: '#1976d2', letterSpacing: '0.04em' }}>
            MAKE-IT COMPANY
          </Typography>
          <div className="desktop-menu" style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
            <IconButton color="primary" sx={{ mr: 1 }}>
              <Badge badgeContent={2} color="error">
                <NotificationsIcon />
              </Badge>
            </IconButton>
            <div style={{position:'relative'}}>
              <IconButton sx={{ ml: 2 }} onClick={() => setProfileOpen(!profileOpen)}>
                <Avatar alt="User" src={avatar} />
              </IconButton>
              {profileOpen && (
                <div style={{
                  position:'absolute', top:'48px', right:0, background:'#fff',
                  boxShadow:'0 8px 32px rgba(67,206,162,0.18)', borderRadius:'14px', minWidth:'200px', zIndex:10002, padding:'16px',
                  border:'1.5px solid #1976d2', display:'flex', flexDirection:'column', alignItems:'stretch',
                }}>
                  <div style={{fontWeight:'bold', marginBottom:'10px', color:'#1976d2', fontSize:'1.08rem'}}>Profile</div>
                  <div style={{marginBottom:'10px', cursor:'pointer', color:'#1976d2', fontWeight:500}}
                    onClick={() => {
                      setShowProfilePage(true);
                      setProfileOpen(false);
                    }}
                  >My Account</div>
                  <div style={{marginBottom:'10px', cursor:'pointer', color:'#1976d2', fontWeight:500}}
                    onClick={() => { setProfileOpen(false); navigate('/settings'); }}
                  >Settings</div>
                  <div style={{marginBottom:'10px', cursor:'pointer', color:'#f75c7a', fontWeight:600}}
                    onClick={async () => {
                      // Log logout action
                      try {
                        const actor_id = localStorage.getItem('userId');
                        const actor_role = localStorage.getItem('role');
                        const actor_email = localStorage.getItem('email');
                        if (actor_id && actor_role && actor_email) {
                          await logAction({
                            actor_id,
                            actor_role,
                            actor_email,
                            action_type: 'logout',
                            action_description: 'User logged out',
                            page: window.location.pathname,
                          });
                        }
                      } catch (e) {}
                      localStorage.clear();
                      navigate('/Login');
                    }}
                  >Logout</div>
                </div>
              )}
            </div>
          </div>
          <div className="mobile-menu">
            <IconButton edge="end" color="inherit" aria-label="menu" onClick={() => setDrawerOpen(true)}>
              <MenuIcon />
            </IconButton>
          </div>
          <Drawer anchor="right" open={drawerOpen} onClose={() => setDrawerOpen(false)}>
            <List sx={{ width: 220 }}>
              {menuItems.map(item => (
                <ListItem button key={item.key} onClick={() => { setSelectedMenu(item.key); setDrawerOpen(false); }}>
                  <ListItemText primary={item.label} />
                </ListItem>
              ))}
              <ListItem button onClick={() => { handleProfileClose(); setDrawerOpen(false); }}>
                <ListItemText primary="Profile" />
              </ListItem>
              <ListItem button onClick={() => { handleProfileClose(); navigate('/settings'); setDrawerOpen(false); }}>
                <ListItemText primary="Settings" />
              </ListItem>
              <ListItem button onClick={() => { handleProfileClose(); navigate('/logout'); setDrawerOpen(false); }}>
                <ListItemText primary="Log Out" sx={{ color: 'red' }} />
              </ListItem>
            </List>
          </Drawer>
        </Toolbar>
        <style>{`
          @media (max-width: 900px) {
            .desktop-menu { display: none !important; }
            .mobile-menu { display: block !important; }
          }
          @media (min-width: 901px) {
            .desktop-menu { display: flex !important; gap: 8px; }
            .mobile-menu { display: none !important; }
          }
        `}</style>
      </AppBar>
      <div style={{
        display: 'flex',
        minHeight: '100vh',
        width: '100vw',
        overflow: 'hidden',
        flexDirection: 'row',
        marginTop: '64px', // shift down for AppBar
      }}>
        {/* Left Sidebar */}
        <div style={{
          width: '240px',
          background: '#fff',
          borderRadius: '0 24px 24px 0',
          boxShadow: '0 2px 12px rgba(0,0,0,0.08)',
          padding: '32px 0',
          minHeight: 'calc(100vh - 64px)',
          position: 'fixed',
          left: 0,
          top: 64,
          zIndex: 99,
          transition: 'width 0.3s',
        }}
        className="sidebar-left"
        >
          {menuItems.map(item => (
            <div
              key={item.key}
              style={{margin:'0', fontWeight:'bold', color:'#1976d2', padding:'20px 18px 8px 18px', cursor:'pointer', background:selectedMenu===item.key?'#e3f2fd':'none', borderRadius:'8px', fontSize:'1.15rem', textAlign:'left'}}
              onClick={() => {
                setSelectedMenu(item.key);
                setShowProfilePage(false); // Hide profile page when switching menu
              }}
            >
              {item.label}
            </div>
          ))}
        </div>

        {/* Main Content */}
        <main style={{
          marginLeft: '240px',
          marginRight: '340px',
          padding: '32px',
          background: '#f7f8fa',
          minHeight: 'calc(100vh - 64px)',
          width: 'calc(100vw - 240px - 340px)',
          boxSizing: 'border-box',
          transition: 'margin 0.3s, width 0.3s',
        }}
        className="main-content"
        >
          {/* Show only ProfilePage if showProfilePage is true, else show dashboard content */}
          {showProfilePage ? (
            <ProfilePage
              currentUser={{ id: idUser, email: localStorage.getItem('email') || '', role }}
              onAvatarChange={handleAvatarChange}
            />
          ) : (
            <>
              {selectedMenu === 'my_tutorial' && (
                <MyTutorial userId={userId} />
              )}
              {selectedMenu === 'mydocument' && (
                <MyDocument userId={userId} role={role} />
              )}
              {selectedMenu === 'announcementsPage' && (
                <AnnouncementPage currentUser={{ id: userId, email: localStorage.getItem('email') || '', role }} />
              )}
              {selectedMenu === 'profile' && (
                <ProfilePage
                  currentUser={{ id: idUser, email: localStorage.getItem('email') || '', role }}
                  onAvatarChange={handleAvatarChange}
                />
              )}
              {selectedMenu === 'dashboard' && (
                <>
                  {/* Top Cards Row */}
                  <div style={{display:'flex', gap:'24px', marginBottom:'32px', flexWrap:'wrap'}}>
                    {/* Enrolled Courses Card */}
                    <div style={{flex:'1 1 300px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(67,206,162,0.08)', padding:'24px', width:'100%'}}>
                      <h3 style={{margin:'0 0 8px 0', color:'#43cea2'}}>Courses Enrolled</h3>
                      <div style={{fontSize:'1.05rem', color:'#1976d2'}}>
                        {access.includes('courses') && userId
                          ? `You are enrolled in ${chartData.length} course${chartData.length === 1 ? '' : 's'}.`
                          : 'No enrolled courses found.'}
                      </div>
                    </div>
                    {/* Earnings Card */}
                    <div style={{flex:'1 1 300px', background:'#fff', borderRadius:'16px', boxShadow:'0 2px 12px rgba(0,0,0,0.08)', padding:'24px', width:'100%'}}>
                      <div style={{fontWeight:'bold', fontSize:'1.15rem', color:'#43cea2', marginBottom:'8px'}}>Earning <span style={{fontWeight:'normal', color:'#888'}}> | {monthNames[selectedMonth]} {selectedYear}</span></div>
                      <div style={{fontSize:'1.5rem', color:'#27ae60', fontWeight:'bold'}}>{earningValue.toFixed(2)} USD</div>
                      <div style={{fontSize:'1rem', color:'#888'}}>
                        {earnings.display_quantity !== undefined && earnings.display_unit ? `${earnings.display_quantity} ${earnings.display_unit.charAt(0).toUpperCase() + earnings.display_unit.slice(1)}` : ''}
                      </div>
                      {percentChange !== null && (
                        <div style={{fontSize:'0.95rem', color: percentChange > 0 ? '#27ae60' : percentChange < 0 ? '#e53935' : '#888'}}>
                          {percentChange > 0 ? '▲' : percentChange < 0 ? '▼' : ''} {Math.abs(percentChange).toFixed(1)}% {percentChange > 0 ? 'increase' : percentChange < 0 ? 'decrease' : 'no change'} from last month
                        </div>
                      )}
                    </div>
                  </div>
                  {/* Chart Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2 style={{textAlign:'center', color:'#1976d2', marginBottom:'18px'}}>Monthly Progress</h2>
                    <div style={{display:'flex', justifyContent:'center', gap:'8px', marginBottom:'18px', flexWrap:'wrap'}}>
                      {monthNames.map((m, idx) => (
                        <span key={m} style={{background: idx === selectedMonth ? '#ffa726' : '#90caf9', color:'#fff', borderRadius:'16px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', margin:'2px', cursor:'pointer'}} onClick={()=>{setSelectedMonth(idx); setSelectedYear(new Date().getFullYear());}}>{m}</span>
                      ))}
                    </div>
                    <div style={{width:'100%', height:'320px'}}>
                      <ResponsiveContainer width="100%" height={320}>
                        <BarChart data={chartData.filter(d=>d.year===selectedYear)} margin={{top: 20, right: 30, left: 0, bottom: 5}}>
                          <CartesianGrid strokeDasharray="3 3" />
                          <XAxis dataKey="name" />
                          <YAxis />
                          <Tooltip />
                          <Legend />
                          <Bar dataKey="minutes">
                            {chartData.filter(d=>d.year===selectedYear).map((entry, idx) => (
                              <Cell key={`cell-${idx}`} fill={entry.fill} />
                            ))}
                            <LabelList dataKey="minutes" position="top" />
                          </Bar>
                        </BarChart>
                      </ResponsiveContainer>
                    </div>
                  </div>
                </>
              )}
              {selectedMenu === 'courses' && (
                <>
                  {/* Courses Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2>Courses</h2>
                    {/* Pass userId and isAdmin to CourseDashboard */}
                    <CourseDashboard userId={userId} isAdmin={role === 'Administrator'} />
                  </div>
                </>
              )}
              {selectedMenu === 'tasks' && (
                <>
                  {/* Tasks Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2>Tasks</h2>
                    {/* Render tasks manager or tasks list here */}
                    {/* You can import and use TasksManager if available */}
                    {/* <TasksManager /> */}
                    <div>Tasks content goes here.</div>
                  </div>
                </>
              )}
              {selectedMenu === 'my_progress' && (
                <>
                  {/* My Progress Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2>My Progress</h2>
                    <div>Progress content goes here.</div>
                  </div>
                </>
              )}
              {selectedMenu === 'edit_user' && (
                <>
                  {/* Edit User Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2>Edit User</h2>
                    <div>Edit user content goes here.</div>
                  </div>
                </>
              )}
              {selectedMenu === 'admin_settings' && (
                <>
                  {/* Admin Settings Section */}
                  <div style={{marginBottom:'32px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.05)', padding:'24px'}}>
                    <h2>Admin Settings</h2>
                    <div>Admin settings content goes here.</div>
                  </div>
                </>
              )}
            </>
          )}
        </main>
        {/* Right Sidebar (Recent Activities, News & Blog) */}
        <aside className="sidebar-right">
          <div style={{marginBottom: '32px'}}>
            <h3 style={{color: '#1976d2', fontWeight: 700, marginBottom: '18px'}}>Recent Activities</h3>
            <div style={{background: '#fff', borderRadius: '12px', boxShadow: '0 2px 8px rgba(0,0,0,0.08)', padding: '18px', marginBottom: '18px', maxHeight: '220px', overflowY: 'auto'}}>
              {actionsLoading ? (
                <div style={{color:'#1976d2'}}>Loading recent actions...</div>
              ) : actionsError ? (
                <div style={{color:'#e53935'}}>{actionsError}</div>
              ) : recentActions.length === 0 ? (
                <div style={{color:'#888'}}>No recent actions found.</div>
              ) : (
                <ul style={{listStyle:'none', padding:0, margin:0}}>
                  {recentActions.map(action => {
                    // Show 'I' or 'You' if the action was performed by the current user
                    let desc = action.action_description;
                    const currentUserId = localStorage.getItem('userId');
                    if (String(action.actor_id) === String(currentUserId)) {
                      desc = desc.replace(/\bUser\b|\bYou\b|\bMe\b|\bI\b/gi, 'I');
                      if (!/\bI\b/i.test(desc)) {
                        desc = 'I ' + desc.charAt(0).toLowerCase() + desc.slice(1);
                      }
                    }
                    return (
                      <li key={action.id} style={{marginBottom:'12px', display:'flex', alignItems:'center', fontSize:'1rem'}}>
                        <span style={{marginRight:'10px'}}>
                          <i className={`bi bi-${action.action_type === 'login' ? 'box-arrow-in-right text-success' : action.action_type === 'edit_interpreter_chart' ? 'pencil-square text-warning' : action.action_type === 'print_interpreter_chart' ? 'printer text-primary' : 'activity text-secondary'}`}></i>
                        </span>
                        <div style={{flex:1}}>
                          <div style={{fontWeight:'bold', color:'#1976d2'}}>{action.action_type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}</div>
                          <div style={{color:'#888', fontSize:'0.95rem'}}>{desc}</div>
                        </div>
                        <span style={{fontSize:'0.92rem', minWidth: 80, textAlign:'right', color:'#888', marginLeft:'8px'}}>{action.time_ago}</span>
                      </li>
                    );
                  })}
                </ul>
              )}
            </div>
          </div>
          <div>
            <h3 style={{color: '#1976d2', fontWeight: 700, marginBottom: '12px'}}>Latest News & Blog</h3>
            <div style={{background: '#fff', borderRadius: '12px', boxShadow: '0 2px 8px rgba(0,0,0,0.08)', padding: '12px', maxHeight: 320, overflowY: 'auto', display: 'flex', flexDirection: 'column', gap: '12px'}}>
              {newsBlogLoading ? (
                <div style={{color:'#888'}}>Loading news and blog posts...</div>
              ) : newsBlogError ? (
                <div style={{color:'#e53935'}}>{newsBlogError}</div>
              ) : newsBlog.length === 0 ? (
                <div style={{color:'#888'}}>No news or blog posts found.</div>
              ) : (
                newsBlog.slice(0, 5).map(item => {
                  // Helper to strip HTML tags for clean text
                  const stripHtml = (html) => {
                    if (!html) return '';
                    const div = document.createElement('div');
                    div.innerHTML = html;
                    return div.textContent || div.innerText || '';
                  };
                  const title = stripHtml(item.title);
                  const date = item.date || item.created_at || '';
                  const imageName = item.image || item.picture || '';
                  const imageUrl = imageName ? `/backend/public/images/news/${imageName}` : 'https://via.placeholder.com/44x44?text=News';
                  const type = (item.type || '').toLowerCase();
                  const typeLabel = type === 'blog' ? 'Blog' : 'News';
                  const typeColor = type === 'blog' ? '#ffb347' : '#1976d2';
                  return (
                    <div key={item.id} className="card border-0 shadow-sm" style={{
                      borderRadius: '12px',
                      background: '#fafdff',
                      minWidth: 0,
                      width: '100%',
                      maxWidth: '100%',
                      margin: 0,
                      boxSizing: 'border-box',
                      display: 'flex',
                      flexDirection: 'row',
                      alignItems: 'center',
                      gap: '10px',
                    }}>
                      <img
                        src={imageUrl}
                        alt={typeLabel}
                        style={{
                          width: 44,
                          height: 44,
                          objectFit: 'cover',
                          borderRadius: '8px',
                          background: '#e0e0e0',
                          flexShrink: 0,
                          marginLeft: 8,
                          marginRight: 0
                        }}
                        onError={e => { e.target.onerror = null; e.target.src = 'https://via.placeholder.com/44x44?text=News'; }}
                      />
                      <div className="card-body p-2 d-flex flex-column" style={{minWidth:0, width:'100%'}}>
                        <div style={{display:'flex', alignItems:'center', gap:6, marginBottom:2}}>
                          <span className="fw-bold text-primary" style={{fontSize:'1.01rem', wordBreak:'break-word', lineHeight:'1.2', width:'100%'}} title={title}>{title}</span>
                        </div>
                        <div className="text-muted mt-1" style={{fontSize:'0.93rem', textAlign:'right', width:'100%'}}>{date}</div>
                      </div>
                    </div>
                  );
                })
              )}
            </div>
          </div>
        </aside>
        {/* Announcement Popup Modal */}
        {popupAnnouncements.length > 0 && (
          <Dialog open={announcementPopupOpen} onClose={() => setAnnouncementPopupOpen(false)} maxWidth="md" fullWidth>
            <DialogTitle>Announcements</DialogTitle>
            <DialogContent>
              {popupAnnouncements.map(a => {
                const expInfo = getExpirationSeverity(a.expiration_date);
                return (
                  <div key={a.id} style={{marginBottom:'24px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.08)', padding:'24px', marginBottom:'8px'}}>
                    <div style={{marginBottom:'8px'}}>
                      <strong style={{fontSize:'1.25rem', marginBottom:'8px', display:'inline-block'}}>{a.title}</strong>
                      {a.announcement_type && (
                        <span style={{
                          background: announcementTypeColors[a.announcement_type] || '#1976d2',
                          color: '#fff',
                          borderRadius: '6px',
                          padding: '2px 10px',
                          marginLeft: '12px',
                          fontSize: '0.95rem',
                          fontWeight: 'bold',
                          verticalAlign: 'middle',
                          display: 'inline-block',
                        }}>{a.announcement_type}</span>
                      )}
                    </div>
                    <div dangerouslySetInnerHTML={{ __html: a.content }} style={{ marginBottom: 8 }} />
                    <div style={{marginBottom:'8px'}}>
                      {a.target_roles && a.target_roles.split(',').map(role => (
                        <span key={role} style={{display:'inline-block', background:'#e3f2fd', color:'#1976d2', borderRadius:'8px', padding:'4px 12px', marginRight:'8px', fontSize:'0.95rem', fontWeight:'bold'}}>{role}</span>
                      ))}
                      {a.target_users && a.target_users.split(',').map(user => (
                        <span key={user} style={{display:'inline-block', background:'#b2ebf2', color:'#1976d2', borderRadius:'8px', padding:'4px 12px', marginRight:'8px', fontSize:'0.95rem', fontWeight:'bold'}}>{user}</span>
                      ))}
                    </div>
                    {a.expiration_date && (
                      expInfo.message ? (
                        <div style={{ color: expirationSeverityColors[expInfo.severity], fontSize: '0.95rem', fontWeight:'bold' }}>{expInfo.message} ({a.expiration_date})</div>
                      ) : (
                        <div style={{ color: expirationSeverityColors.normal, fontSize: '0.95rem', fontWeight:'bold' }}>Expires: {a.expiration_date}</div>
                      )
                    )}
                  </div>
                );
              })}
            </DialogContent>
            <DialogActions>
              <Button onClick={() => setAnnouncementPopupOpen(false)} variant="contained">Close</Button>
            </DialogActions>
          </Dialog>
        )}
        <style>{`
          .sidebar-right {
            width: 320px;
            background: #eaf6ff;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
            border-radius: 16px;
            padding: 24px;
            position: fixed;
            top: 0;
            right: 0;
            height: 100vh;
            overflow-y: auto;
            z-index: 100;
          }
          @media (max-width: 900px) {
            .sidebar-left {
              display: none !important;
            }
            .main-content {
              margin-left: 0 !important;
              margin-right: 0 !important;
              width: 100vw !important;
              padding: 12px !important;
            }
            .sidebar-right {
              position: static !important;
              width: 100vw !important;
              border-radius: 0 !important;
              box-shadow: none !important;
              padding: 12px !important;
              height: auto !important;
              margin-bottom: 24px !important;
            }
            div[style*='display: flex'] {
              flex-direction: column !important;
            }
          }
        `}</style>
      </div>
    </>
  );
};

export default Index10000;
