import React, { useState } from 'react';
import axios from 'axios';

const CertificateVerify = () => {
  const [certType, setCertType] = useState('medical');
  const [certId, setCertId] = useState('');
  const [result, setResult] = useState(null);
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);

  const handleVerify = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError('');
    setResult(null);
    try {
      const res = await axios.get(`/api/verify/${certType}/${certId}`);
      setResult(res.data);
    } catch (err) {
      setError(err.response?.data?.error || 'Certificate not found');
    }
    setLoading(false);
  };

  return (
    <div className="container mt-5">
      <h2>Certificate Verification</h2>
      <form onSubmit={handleVerify} className="mb-4">
        <div className="row g-2 align-items-end">
          <div className="col-md-3">
            <label className="form-label">Certificate Type</label>
            <select className="form-select" value={certType} onChange={e => setCertType(e.target.value)}>
              <option value="medical">Medical Interpreter</option>
              <option value="hipaa">HIPAA Compliance</option>
              <option value="legal">Legal Interpreter</option>
            </select>
          </div>
          <div className="col-md-4">
            <label className="form-label">Certificate Number</label>
            <input type="text" className="form-control" value={certId} onChange={e => setCertId(e.target.value)} required placeholder="Enter Certificate Number (cerf_id)" />
          </div>
          <div className="col-md-2">
            <button type="submit" className="btn btn-primary" disabled={loading}>Verify</button>
          </div>
        </div>
      </form>
      {loading && <div>Loading...</div>}
      {error && <div className="alert alert-danger">{error}</div>}
      {result && (
        <div className="card mt-4">
          <div className="card-header" style={{background: getCertColor(result.certificate.certificate_type, 'primary'), color: '#fff'}}>
            <h4>{capitalize(result.certificate.certificate_type)} Certificate</h4>
            <span className={`badge ${result.is_expired ? 'bg-danger' : 'bg-success'}`}>{result.is_expired ? 'Expired' : 'Valid'}</span>
            {result.expiration_message && <div className="mt-2 text-warning">{result.expiration_message}</div>}
          </div>
          <div className="card-body">
            <div className="row mb-3">
              <div className="col-md-3 text-center">
                <img src={result.user.profile_image || '/default-avatar.png'} alt="Profile" className="rounded-circle border border-3" style={{width: '100px', height: '100px', objectFit: 'cover'}} />
                <div className="mt-2">
                  <span className="badge bg-light text-dark mx-1">User: {result.user.id}</span>
                  <span className="badge bg-light text-dark mx-1">Cert: {result.certificate.cerf_id}</span>
                </div>
              </div>
              <div className="col-md-9">
                <h5 className="mb-1">{result.user.nom} {result.user.prenom}</h5>
                <p className="mb-1"><strong>Email:</strong> {result.user.email}</p>
                <p className="mb-1"><strong>Languages:</strong> {result.certificate.language_from} ↔ {result.certificate.language_to}</p>
                <p className="mb-1"><strong>Date Validity:</strong> {formatDate(result.certificate.date_validity)}</p>
                <p className="mb-1"><strong>Text:</strong> {result.certificate.certificate_text}</p>
              </div>
            </div>
            {/* Expiration/Expiring Soon Alerts */}
            {result.is_expired && (
              <div className="alert alert-danger">
                <strong>⚠️ Certificate Expired</strong><br />
                {result.expiration_message}<br />
                <span>Action Required: Please renew your {capitalize(result.certificate.certificate_type)} certification. <a href="mailto:training@makeitservices.org">Contact training@makeitservices.org</a></span>
              </div>
            )}
            {!result.is_expired && result.days_until_expiry <= 90 && result.days_until_expiry > 0 && (
              <div className="alert alert-warning">
                <strong>📅 Certificate Expiring Soon</strong><br />
                {result.expiration_message}<br />
                <span>Renewal Recommended: Please consider renewing soon. <a href="mailto:training@makeitservices.org">Contact training@makeitservices.org</a></span>
              </div>
            )}
            {/* Certificate Type Specific Block */}
            {!result.is_expired && (
              <div className="mt-3 p-3 border rounded bg-light">
                {result.certificate.certificate_type === 'medical' && (
                  <>
                    <h5 style={{color: getCertColor('medical', 'primary')}}>Medical Interpreter Certificate</h5>
                    <p>Successfully completed the 60-hour Medical Interpreter Training course and demonstrated proficiency in professional medical interpretation.</p>
                    <ul className="list-inline text-center mb-3">
                      <li className="list-inline-item">LPT</li>
                      <li className="list-inline-item">MTC</li>
                      <li className="list-inline-item">MOC</li>
                      <li className="list-inline-item">HIPAA</li>
                    </ul>
                    <div className="mb-2">
                      <h6>Medical Subjects</h6>
                      <ul>
                        <li>Health Care</li>
                        <li>OBGYN</li>
                        <li>Neurology</li>
                        <li>Cardiology</li>
                        <li>Dental</li>
                        <li>Emergency</li>
                        <li>Gynecology</li>
                        <li>Ophtalmology</li>
                        <li>Orthopedic</li>
                        <li>Importance to take Notes</li>
                        <li>Key Words, Key Phrases, Symbols</li>
                        <li>Online portal Setting, Dress code</li>
                      </ul>
                    </div>
                  </>
                )}
                {result.certificate.certificate_type === 'hipaa' && (
                  <>
                    <h5 style={{color: getCertColor('hipaa', 'primary')}}>HIPAA Privacy & Security</h5>
                    <p>Completed HIPAA compliance training and understands requirements for protecting patient health information.</p>
                    <ul className="list-inline text-center mb-3">
                      <li className="list-inline-item">Privacy Rule</li>
                      <li className="list-inline-item">Security Rule</li>
                      <li className="list-inline-item">Breach Notification</li>
                      <li className="list-inline-item">PHI Protection</li>
                    </ul>
                    <div className="mb-2">
                      <h6>HIPAA Subjects</h6>
                      <ul>
                        <li>Protected Health Information (PHI)</li>
                        <li>Minimum Necessary Standard</li>
                        <li>Individual Rights</li>
                        <li>Administrative, Physical, Technical Safeguards</li>
                        <li>Breach Assessment, Notification Procedures</li>
                        <li>BAA Compliance, Subcontractor Management</li>
                        <li>OCR Investigations, Civil Monetary Penalties</li>
                      </ul>
                    </div>
                  </>
                )}
                {result.certificate.certificate_type === 'legal' && (
                  <>
                    <h5 style={{color: getCertColor('legal', 'primary')}}>Legal Interpreter Certificate</h5>
                    <p>Completed the 40-hour Legal Interpreter certification program and is qualified to provide legal interpretation services.</p>
                    <ul className="list-inline text-center mb-3">
                      <li className="list-inline-item">Legal Terminology</li>
                      <li className="list-inline-item">Court Procedures</li>
                      <li className="list-inline-item">Ethics</li>
                      <li className="list-inline-item">Professional Standards</li>
                    </ul>
                    <div className="mb-2">
                      <h6>Legal Subjects</h6>
                      <ul>
                        <li>Court Structure, Legal Procedures, Due Process</li>
                        <li>Criminal, Civil, Immigration Law Terms</li>
                        <li>Arraignment, Trial, Sentencing</li>
                        <li>Interpreter Code of Ethics, Confidentiality, Impartiality</li>
                        <li>Consecutive, Simultaneous, Sight Translation</li>
                      </ul>
                    </div>
                  </>
                )}
                {/* Award Date */}
                <div className="text-center py-3 border-top">
                  <p className="text-success fw-bold mb-2">Awarded on {formatDate(result.certificate.date_created)}</p>
                </div>
                {/* Contact Info */}
                <div className="text-center pt-3 border-top">
                  <a href="https://makeitservices.org/" className="btn btn-outline-primary btn-sm mx-2"><i className="fas fa-globe me-1"></i>Website</a>
                  <a href="mailto:training@makeitservices.org" className="btn btn-outline-primary btn-sm mx-2"><i className="fas fa-envelope me-1"></i>Contact</a>
                </div>
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default CertificateVerify;

// Helper functions for color and formatting
function getCertColor(type, key) {
  const certColors = {
    medical: { primary: '#c90000', secondary: '#000000', accent: '#28a745' },
    hipaa: { primary: '#0056b3', secondary: '#495057', accent: '#17a2b8' },
    legal: { primary: '#8B4513', secondary: '#2c3e50', accent: '#dc3545' }
  };
  return certColors[type]?.[key] || certColors['medical'][key];
}
function formatDate(dateStr) {
  if (!dateStr) return '';
  const d = new Date(dateStr);
  if (isNaN(d)) return dateStr;
  return d.toLocaleDateString(undefined, { year: 'numeric', month: 'long', day: 'numeric' });
}
function capitalize(str) {
  if (!str) return '';
  return str.charAt(0).toUpperCase() + str.slice(1);
}
