// src/utils/actionLogger.js
import axios from 'axios';

/**
 * Log a user action to the backend action_logs table.
 * @param {Object} params - Log parameters
 * @param {number|string} params.actor_id - The ID of the user performing the action
 * @param {string} params.actor_role - The role of the user (e.g., Administrator, 10000)
 * @param {string} params.actor_email - The email of the user
 * @param {string} params.action_type - The type of action (e.g., 'delete', 'view', 'update', 'login')
 * @param {string} [params.action_description] - Optional description of the action
 * @param {number|string} [params.target_id] - The ID of the target user/entity
 * @param {string} [params.target_role] - The role of the target
 * @param {string} [params.target_email] - The email of the target
 * @param {string} [params.page] - The page or route where the action occurred
 */
export async function logAction({
  actor_id,
  actor_role,
  actor_email,
  action_type,
  action_description = '',
  target_id = null,
  target_role = null,
  target_email = null,
  page = window.location.pathname,
}) {
  // Safety check: skip logging if required fields are missing
  if (!actor_id || !actor_role || !actor_email || !action_type) return;
  try {
    await axios.post('/api/log_action', {
      actor_id,
      actor_role,
      actor_email,
      action_type,
      action_description,
      target_id,
      target_role,
      target_email,
      page,
    });
  } catch (err) {
    // Optionally handle/log error
    // console.error('Failed to log action', err);
  }
}
