

import React, { useEffect, useState } from 'react';
import Header from './Header';
import About from './About';
import Courses from './Courses';
import Teachers from './Teachers';
import Portfolio from './Portfolio';
import Contact from './Contact';
import CourseCard from './CourseCard';
import 'bootstrap/dist/css/bootstrap.min.css';
import { BASE_API } from '../utils/config';

export default function ProfessionalIndex() {
  const [activeTab, setActiveTab] = useState('Home');
  const [courses, setCourses] = useState([]);
  const [carouselIndex, setCarouselIndex] = useState(0);
  const [carouselImages, setCarouselImages] = useState([]);
  const [enrolledCounts, setEnrolledCounts] = useState({});

  useEffect(() => {
    // Fetch top public courses and enrolled counts from backend
    Promise.all([
      fetch(`${BASE_API}/api/top_courses`).then(res => res.json()),
      fetch(`${BASE_API}/api/enrolled_course_count`).then(res => res.json()),
      fetch(`${BASE_API}/api/blog_news_school`).then(res => res.json())
    ]).then(([coursesData, enrolledData, blogs]) => {
      setCourses(coursesData.courses);
      // Map enrolled_counts to object by course_id
      const countsMap = {};
      if (enrolledData.enrolled_counts && Array.isArray(enrolledData.enrolled_counts)) {
        enrolledData.enrolled_counts.forEach(item => {
          countsMap[item.course_id] = item.num_users;
        });
      }
      setEnrolledCounts(countsMap);
      setCarouselImages(
        blogs.map(blog => ({
          url: blog.image_url || 'https://images.unsplash.com/photo-1513258496099-48168024aec0?auto=format&fit=crop&w=800&q=80',
          alt: blog.title,
          caption: blog.title,
          sub: blog.summary || '',
        }))
      );
    });
  }, []);

  // Carousel auto-advance
  useEffect(() => {
    const timer = setInterval(() => {
      setCarouselIndex(i => (i + 1) % carouselImages.length);
    }, 5000);
    return () => clearInterval(timer);
  }, [carouselImages.length]);

  // Only show public courses
  const publicCourses = courses.filter(c => c.public === true || c.public === 1);

  return (
    <>
      <Header activeTab={activeTab} setActiveTab={setActiveTab} />
      {activeTab === 'Home' && (
        <>
          {/* Bootstrap Carousel Section */}
          <div id="mainCarousel" className="carousel slide" data-bs-ride="carousel" style={{ width: '100%', minHeight: 540, maxHeight: 700, overflow: 'hidden', background: '#222' }}>
            <div className="carousel-indicators" style={{ left: 20, top: '50%', transform: 'translateY(-50%)', flexDirection: 'column', position: 'absolute', zIndex: 10 }}>
              {carouselImages.map((_, idx) => (
                <button
                  key={idx}
                  type="button"
                  data-bs-target="#mainCarousel"
                  data-bs-slide-to={idx}
                  className={idx === carouselIndex ? 'active' : ''}
                  aria-current={idx === carouselIndex ? 'true' : undefined}
                  aria-label={`Slide ${idx + 1}`}
                  style={{ width: 18, height: 18, borderRadius: '50%', border: '2px solid #fff', background: idx === carouselIndex ? '#f75c7a' : 'rgba(255,255,255,0.5)', marginBottom: 8 }}
                  onClick={() => setCarouselIndex(idx)}
                />
              ))}
            </div>
            <div className="carousel-inner" style={{ width: '100%', minHeight: 540, maxHeight: 700 }}>
              {carouselImages.map((img, idx) => (
                <div
                  className={`carousel-item${idx === carouselIndex ? ' active' : ''}`}
                  key={idx}
                  style={{ width: '100%', minHeight: 540, maxHeight: 700, background: `url(${img.url}) center/cover no-repeat`, position: 'relative' }}
                >
                  <div style={{
                    width: '100%',
                    height: '100%',
                    background: 'rgba(0,0,0,0.45)',
                    display: 'flex',
                    flexDirection: 'column',
                    alignItems: 'center',
                    justifyContent: 'center',
                    position: 'absolute',
                    top: 0,
                    left: 0,
                    zIndex: 3,
                  }}>
                    <div
                      style={{ color: '#fff', fontWeight: 600, letterSpacing: 1, marginBottom: 8, fontSize: 18 }}
                      dangerouslySetInnerHTML={{ __html: img.sub }}
                    />
                    <div
                      style={{ color: '#fff', fontWeight: 900, marginBottom: 24, textAlign: 'center', fontSize: '2.5rem', lineHeight: 1.1 }}
                      dangerouslySetInnerHTML={{ __html: img.caption }}
                    />
                    <button className="btn btn-danger btn-lg fw-bold rounded-2">Learn More</button>
                  </div>
                </div>
              ))}
            </div>
          </div>
          {/* Top Courses Section */}
          <div className="container my-5">
            <h2 className="fw-bold mb-4 text-center" style={{ letterSpacing: 1 }}>Top Courses</h2>
            <div className="row justify-content-center">
              {publicCourses.length === 0 && (
                <div className="col-12">
                  <h5 className="text-secondary text-center">No public courses available.</h5>
                </div>
              )}
              {publicCourses.map(course => (
                <div className="col-12 col-sm-6 col-md-4 mb-4" key={course.course_id || course.id}>
                  <CourseCard course={course} enrolledCount={enrolledCounts[course.course_id || course.id] || 0} />
                </div>
              ))}
            </div>
          </div>
        </>
      )}
      {activeTab === 'About' && <About />}
      {activeTab === 'Courses' && <Courses />}
      {activeTab === 'Teachers' && <Teachers />}
      {activeTab === 'Portfolio' && <Portfolio />}
      {activeTab === 'Contact' && <Contact />}
    </>
  );
}
