import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { BASE_API } from './utils/config';

const formStyle = {
  maxWidth: '400px',
  margin: '60px auto',
  padding: '32px',
  borderRadius: '16px',
  boxShadow: '0 4px 24px rgba(0,0,0,0.12)',
  background: '#fff',
  display: 'flex',
  flexDirection: 'column',
  gap: '18px',
  animation: 'fadeIn 1s',
};

const inputStyle = {
  padding: '12px',
  borderRadius: '8px',
  border: '1px solid #ccc',
  fontSize: '1rem',
};

const buttonStyle = {
  padding: '14px',
  borderRadius: '8px',
  border: 'none',
  background: 'linear-gradient(90deg, #a259e6 0%, #f75c7a 100%)',
  color: '#fff',
  fontWeight: 'bold',
  fontSize: '1.1rem',
  cursor: 'pointer',
  marginTop: '12px',
  transition: 'background 0.2s',
};

const errorStyle = {
  color: '#f75c7a',
  fontWeight: 'bold',
  textAlign: 'center',
};

function Register() {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [role] = useState('Default'); // Always Default
  const [sexe, setSexe] = useState('');
  const [language, setLanguage] = useState('');
  const [languages, setLanguages] = useState([]);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');
  const navigate = useNavigate();

  useEffect(() => {
    // Fetch languages from WorldWise API (using restcountries)
  fetch('https://restcountries.com/v3.1/all?fields=languages')
    .then(res => res.json())
    .then(data => {
      const langSet = new Set();
      if (Array.isArray(data)) {
        data.forEach(country => {
          if (country.languages) {
            Object.values(country.languages).forEach(lang => {
              if (lang !== 'English') langSet.add(lang);
            });
          }
        });
        setLanguages(Array.from(langSet).sort());
      } else {
        setLanguages([]);
      }
    })
    .catch(() => setLanguages([]));
  }, []);

  const validatePassword = (pwd) => {
    const minLength = pwd.length >= 8;
    const hasUpper = /[A-Z]/.test(pwd);
    const hasLower = /[a-z]/.test(pwd);
    const hasNumber = /[0-9]/.test(pwd);
    return minLength && hasUpper && hasLower && hasNumber;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError('');
    setSuccess('');
    if (!email || !password || !confirmPassword || !sexe || !language) {
      setError('Please fill in all fields.');
      return;
    }
    if (!validatePassword(password)) {
      setError('Password must be at least 8 characters, include uppercase, lowercase, and a number.');
      return;
    }
    if (password !== confirmPassword) {
      setError('Passwords do not match.');
      return;
    }
    try {
      const res = await fetch(`${BASE_API}/register`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          email,
          password,
          role: 'Default',
          code: 'new',
          sexe,
          language
        })
      });
      const data = await res.json();
      if (res.ok) {
        setSuccess(`Registration successful! Your user ID is ${data.id_user}`);
        setTimeout(() => navigate('/login'), 2000);
      } else {
        setError(data.error || 'Registration failed');
      }
    } catch (err) {
      setError('Server error');
    }
  };

  return (
    <>
      <div style={{textAlign:'center', marginBottom:'8px'}}>
        <span style={{color:'#a259e6', cursor:'pointer', fontWeight:'bold'}} onClick={() => window.history.back()}>← Back to site</span>
      </div>
      <form style={formStyle} onSubmit={handleSubmit}>
        <h2 style={{textAlign:'center', color:'#a259e6'}}>Register</h2>
  <input style={inputStyle} type="email" placeholder="Email" value={email} onChange={e => setEmail(e.target.value)} required />
      <input style={inputStyle} type="password" placeholder="Password" value={password} onChange={e => setPassword(e.target.value)} required />
      <input style={inputStyle} type="password" placeholder="Confirm Password" value={confirmPassword} onChange={e => setConfirmPassword(e.target.value)} required />
      <select style={inputStyle} value={sexe} onChange={e => setSexe(e.target.value)} required>
        <option value="">Select Sexe</option>
        <option value="Male">Male</option>
        <option value="Female">Female</option>
        <option value="Other">Other</option>
      </select>
      <select style={inputStyle} value={language} onChange={e => setLanguage(e.target.value)} required>
        <option value="">Select Language</option>
        {languages.map(lang => (
          <option key={lang} value={lang}>{lang}</option>
        ))}
      </select>
      <button style={buttonStyle} type="submit">Sign Up</button>
      {error && <div style={errorStyle}>{error}</div>}
      {success && <div style={{color:'#27ae60', fontWeight:'bold', textAlign:'center'}}>{success}</div>}
      <div style={{textAlign:'center', marginTop:'8px'}}>
        Already have an account? <span style={{color:'#a259e6', cursor:'pointer'}} onClick={()=>navigate('/login')}>Login</span>
      </div>
      </form>
    </>
  );
}

export default Register;
