import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { Tabs, Table, Button, Modal, Form, Input, Select, Spin, message } from 'antd';
import { Editor } from '@tinymce/tinymce-react';

const { TabPane } = Tabs;
const { Option } = Select;

// Language options (excluding English)
function MedicalDocumentGlossary({ documentId, userId }) {
  // Add state for selected language filter
  const [selectedLanguage, setSelectedLanguage] = useState(null);
  // State for multi-select in Delete tab
  const [selectedRowKeys, setSelectedRowKeys] = useState([]);
  const [activeTab, setActiveTab] = useState('Glossary');
  const [words, setWords] = useState([]);
  const [filteredWords, setFilteredWords] = useState([]);
  const [loading, setLoading] = useState(false);
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(20);
  const [total, setTotal] = useState(0);
  const [search, setSearch] = useState('');
  const [addModalVisible, setAddModalVisible] = useState(false);
  const [editModalVisible, setEditModalVisible] = useState(false);
  const [pendingWords, setPendingWords] = useState([]);
  const [pendingEdits, setPendingEdits] = useState([]);
  const [selectedWord, setSelectedWord] = useState(null);
  const [translations, setTranslations] = useState([]);
  const [translationModalVisible, setTranslationModalVisible] = useState(false);
  const [languageOptions, setLanguageOptions] = useState([]);
  const [descValue, setDescValue] = useState('');
  // Role logic for status field
  const [role, setRole] = useState('');
  const [userEmail, setUserEmail] = useState('');
  const [editSourceWordModalVisible, setEditSourceWordModalVisible] = useState(false);
  const [selectedSourceWord, setSelectedSourceWord] = useState(null);
  const [sourceWordInput, setSourceWordInput] = useState('');
  const [editSourceWordForm] = Form.useForm();
  // State to track if user has access to Glossary tab (access_key = 119)
  const [hasGlossaryAccess, setHasGlossaryAccess] = useState(false);
  // State to track if user has access to Edit tab (access_key = 120)
  const [hasEditAccess, setHasEditAccess] = useState(false);
  // State to track if user has access to Pending tab (access_key = 122)
  const [hasPendingAccess, setHasPendingAccess] = useState(false);
  // State to track if user has access to Approved tab (access_key = 123)
  const [hasApprovedAccess, setHasApprovedAccess] = useState(false);
  // State to track if user has access to Delete tab (access_key = 124)
  const [hasDeleteAccess, setHasDeleteAccess] = useState(false);
  // State to track if user has access to Add Word button (access_key = 121)
  const [hasAddWordAccess, setHasAddWordAccess] = useState(false);
  // State to track if user has access to Edit Source Word button (access_key = 125)
  const [hasEditSourceWordAccess, setHasEditSourceWordAccess] = useState(false);

  useEffect(() => {
    // Fetch user role and email from API
    async function fetchRole() {
      try {
        const res = await axios.get(`/roles/${userId}`);
        setRole(res.data.role);
        setUserEmail(res.data.email || '');
      } catch (e) {
        setRole('');
        setUserEmail('');
      }
    }
    if (userId) fetchRole();
  }, [userId]);

  // Check if user has access to Glossary tab (access_key = 119)
  useEffect(() => {
    async function checkGlossaryAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/119`);
        setHasGlossaryAccess(res.data.has_access);
      } catch (e) {
        setHasGlossaryAccess(false);
      }
    }
    if (userId) checkGlossaryAccess();
  }, [userId]);

  // Check if user has access to Edit tab (access_key = 120)
  useEffect(() => {
    async function checkEditAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/120`);
        setHasEditAccess(res.data.has_access);
      } catch (e) {
        setHasEditAccess(false);
      }
    }
    if (userId) checkEditAccess();
  }, [userId]);

  // Check if user has access to Pending tab (access_key = 122)
  useEffect(() => {
    async function checkPendingAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/122`);
        setHasPendingAccess(res.data.has_access);
      } catch (e) {
        setHasPendingAccess(false);
      }
    }
    if (userId) checkPendingAccess();
  }, [userId]);

  // Check if user has access to Approved tab (access_key = 123)
  useEffect(() => {
    async function checkApprovedAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/123`);
        setHasApprovedAccess(res.data.has_access);
      } catch (e) {
        setHasApprovedAccess(false);
      }
    }
    if (userId) checkApprovedAccess();
  }, [userId]);

  // Check if user has access to Delete tab (access_key = 124)
  useEffect(() => {
    async function checkDeleteAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/124`);
        setHasDeleteAccess(res.data.has_access);
      } catch (e) {
        setHasDeleteAccess(false);
      }
    }
    if (userId) checkDeleteAccess();
  }, [userId]);

  // Check if user has access to Add Word button (access_key = 121)
  useEffect(() => {
    async function checkAddWordAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/121`);
        setHasAddWordAccess(res.data.has_access);
      } catch (e) {
        setHasAddWordAccess(false);
      }
    }
    if (userId) checkAddWordAccess();
  }, [userId]);

  // Check if user has access to Edit Source Word button (access_key = 125)
  useEffect(() => {
    async function checkEditSourceWordAccess() {
      try {
        const res = await axios.get(`/api/user_access/check/${userId}/125`);
        setHasEditSourceWordAccess(res.data.has_access);
      } catch (e) {
        setHasEditSourceWordAccess(false);
      }
    }
    if (userId) checkEditSourceWordAccess();
  }, [userId]);

  // Set default tab based on access
  useEffect(() => {
    if (hasGlossaryAccess) {
      setActiveTab('Glossary');
    } else if (hasEditAccess) {
      setActiveTab('Edit');
    } else if (hasPendingAccess) {
      setActiveTab('Pending');
    } else if (hasApprovedAccess) {
      setActiveTab('Approved');
    } else if (hasDeleteAccess) {
      setActiveTab('Delete');
    } else {
      setActiveTab('Edit');
    }
    // Only run when access changes
  }, [hasGlossaryAccess, hasEditAccess, hasPendingAccess, hasApprovedAccess, hasDeleteAccess]);

  // Fix: track if modal is open to reset TinyMCE value
  useEffect(() => {
    if (!addModalVisible) setDescValue('');
  }, [addModalVisible]);
    // TinyMCE CDN URL for version 6 (official)
    const tinymceCDN = "https://cdn.tiny.cloud/1/2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65/tinymce/6/tinymce.min.js";
  const [addForm] = Form.useForm();
  const [editForm] = Form.useForm();

  // Fetch approved translations directly from medical_word_translation
  const fetchWords = async () => {
    setLoading(true);
    try {
      const res = await axios.get('/api/medical_word_translations?status=approved');
      // Transform translations to word-like objects for table display
      const translations = res.data.translations || [];
      // Group by word_id, fetch source_word/source_language via joined data if available
      const grouped = {};
      translations.forEach(t => {
        if (!grouped[t.word_id]) {
          grouped[t.word_id] = {
            id: t.word_id,
            source_word: t.source_word || '',
            source_language: t.source_language || 'English',
            last_update: t.last_update,
            translations: []
          };
        }
        grouped[t.word_id].translations.push({
          id: t.id,
          language: t.language,
          target_word: t.target_word,
          description: t.description,
          status: t.status,
          last_update: t.last_update
        });
      });
      setWords(Object.values(grouped));
      setTotal(translations.length);
    } catch (e) {
      message.error('Failed to fetch approved translations');
    }
    setLoading(false);
  };

  // Fetch ALL words with all translations (for Edit tab) - includes words without translations
  const fetchAllWords = async () => {
    setLoading(true);
    try {
      const res = await axios.get('/api/medical_words_with_all_translations');
      const words = res.data.words || [];
      
      // Keep all rows as-is, no grouping - show each translation as a separate row
      setWords(words);
      setTotal(words.length);
    } catch (e) {
      message.error('Failed to fetch all medical words');
    }
    setLoading(false);
  };

  // Fetch pending words
  const fetchPendingWords = async () => {
    try {
      const res = await axios.get('/api/medical_words/pending');
      setPendingWords(res.data.pending_words);
    } catch (e) {}
  };

  // Fetch pending edits
  const fetchPendingEdits = async () => {
    try {
      const res = await axios.get('/api/medical_words/pending_edits');
      // Transform pending edits to match the words table structure
      const pendingEditsData = res.data.pending_edits || [];
      const transformed = pendingEditsData.map(edit => ({
        id: edit.word_id, // Use word_id as id for matching with main table
        edit_request_id: edit.id, // Keep the actual edit request id for approval
        source_word: edit.new_source_word || edit.source_word,
        source_language: 'English',
        last_update: edit.created_at,
        translations: [
          {
            id: 1,
            language: edit.new_language || edit.language,
            target_word: edit.new_target_word || edit.target_word,
            description: edit.new_translation_description || edit.description,
            status: 'pending',
            last_update: edit.created_at
          }
        ]
      }));
      setPendingEdits(transformed);
    } catch (e) {}
  };

  // Fetch language options from API
  useEffect(() => {
    async function fetchLanguages() {
      try {
        const res = await fetch('https://restcountries.com/v3.1/all?fields=languages');
        const data = await res.json();
        const langSet = new Set();
        data.forEach(country => {
          if (country.languages) {
            Object.entries(country.languages).forEach(([code, name]) => {
              if (name !== 'English') langSet.add(JSON.stringify({ value: code.toLowerCase(), label: name }));
            });
          }
        });
        const langs = Array.from(langSet).map(str => JSON.parse(str)).sort((a, b) => a.label.localeCompare(b.label));
        setLanguageOptions(langs);
      } catch (e) {
        setLanguageOptions([]);
      }
    }
    fetchLanguages();
  }, []);

  useEffect(() => {
    if (activeTab === 'Edit') {
      // For Edit tab, fetch ALL words with all translations
      fetchAllWords();
    } else {
      // For other tabs (Glossary, Approved), fetch only approved translations
      fetchWords();
    }
    fetchPendingWords();
    fetchPendingEdits();
  }, [documentId, activeTab]);

  // Filtering logic
  useEffect(() => {
    let filtered = words;
    if (search) {
      const lower = search.toLowerCase();
      filtered = words.filter(word => {
        const sourceMatch = word.source_word && word.source_word.toLowerCase().includes(lower);
        const descMatch = word.description && word.description.toLowerCase().includes(lower);
        const translationMatch = word.translations && word.translations.some(t => {
          const targetMatch = t.target_word && t.target_word.toLowerCase().includes(lower);
          const langMatch = t.language && t.language.toLowerCase().includes(lower);
          return targetMatch || langMatch;
        });
        return sourceMatch || descMatch || translationMatch;
      });
    }
    setFilteredWords(filtered);
    setTotal(filtered.length);
    setPage(1);
  }, [search, words]);
  // Table columns: Source Word, Source Language, and dynamic language columns
  const languageSet = new Set();
  (words || []).forEach(word => {
    (word.translations || []).forEach(t => {
      languageSet.add(t.language);
    });
  });
  const languageColumns = Array.from(languageSet).map(lang => ({
    title: lang,
    children: [
      {
        title: 'Target Word',
        dataIndex: ['translations', lang, 'target_word'],
        key: `${lang}_target_word`,
        render: (_, record) => {
          const t = record.translations.find(tr => tr.language === lang);
          return t ? t.target_word : '';
        }
      },
      {
        title: 'Description',
        dataIndex: ['translations', lang, 'description'],
        key: `${lang}_description`,
        render: (_, record) => {
          const t = record.translations.find(tr => tr.language === lang);
          return t ? t.description : '';
        }
      }
    ]
  }));

  // Highlight helper
  const highlightText = (text, search) => {
    if (!search) return text;
    const regex = new RegExp(`(${search.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')})`, 'gi');
    return (
      <span>
        {String(text).split(regex).map((part, i) =>
          regex.test(part) ? <span key={i} style={{ background: '#ffe58f', fontWeight: 600 }}>{part}</span> : part
        )}
      </span>
    );
  };

  const columns = [
    {
      title: '#',
      key: 'index',
      render: (_, __, index) => index + 1
    },
    {
      title: (
        activeTab === 'Delete' ? (
          <input
            type="checkbox"
            checked={(() => {
              // Only check if all visible rows are selected
              const visibleIds = words
                .slice((page - 1) * pageSize, page * pageSize)
                .map(w => w.id);
              return visibleIds.length > 0 && visibleIds.every(id => selectedRowKeys.includes(id));
            })()}
            indeterminate={(() => {
              const visibleIds = words
                .slice((page - 1) * pageSize, page * pageSize)
                .map(w => w.id);
              const selectedVisible = visibleIds.filter(id => selectedRowKeys.includes(id));
              return selectedVisible.length > 0 && selectedVisible.length < visibleIds.length;
            })()}
            onChange={e => {
              const visibleIds = words
                .slice((page - 1) * pageSize, page * pageSize)
                .map(w => w.id);
              if (e.target.checked) {
                // Add all visible ids to selectedRowKeys
                setSelectedRowKeys(Array.from(new Set([...selectedRowKeys, ...visibleIds])));
              } else {
                // Remove all visible ids from selectedRowKeys
                setSelectedRowKeys(selectedRowKeys.filter(id => !visibleIds.includes(id)));
              }
            }}
          />
        ) : null
      ),
      key: 'select',
      render: (_, record) => activeTab === 'Delete' ? (
        <input
          type="checkbox"
          checked={selectedRowKeys.includes(record.id)}
          onChange={e => {
            if (e.target.checked) {
              setSelectedRowKeys([...selectedRowKeys, record.id]);
            } else {
              setSelectedRowKeys(selectedRowKeys.filter(id => id !== record.id));
            }
          }}
        />
      ) : null
    },
    {
      title: 'Source Language (En)',
      dataIndex: 'source_word',
      key: 'source_word',
      render: (text, record) => highlightText(text, search)
    },
    {
      title: 'Target Language',
      key: 'target_language',
      render: (_, record) => {
        const translations = Array.isArray(record.translations) ? record.translations : [];
        return translations.map(t => t.language ? highlightText(t.language, search) : '').reduce((acc, curr, idx) => acc === '' ? curr : <>{acc}, {curr}</>, '');
      }
    },
    {
      title: 'Target Word',
      key: 'target_word',
      render: (_, record) => {
        const translations = Array.isArray(record.translations) ? record.translations : [];
        return translations.map(t => t.target_word ? highlightText(t.target_word, search) : '').reduce((acc, curr, idx) => acc === '' ? curr : <>{acc}, {curr}</>, '');
      }
    },
    {
      title: 'Description',
      key: 'description',
      render: (_, record) => {
        const translations = Array.isArray(record.translations) ? record.translations : [];
        return translations.length > 0
          ? translations.map(t => t.description ? highlightText(t.description, search) : '').reduce((acc, curr, idx) => acc === '' ? curr : <>{acc}, {curr}</>, '')
          : highlightText(record.description, search);
      }
    },
    {
      title: 'Pending Edit',
      key: 'pending_edit',
      render: (_, record) => {
        // Find pending edit for this word
        const pendingEdit = pendingEdits.find(e => e.word_id === record.id);
        if (!pendingEdit) return null;
        return (
          <div style={{ color: '#ffa726' }}>
            <div><b>New Source:</b> {pendingEdit.new_source_word}</div>
            <div><b>New Description:</b> {pendingEdit.new_description}</div>
            {/* Add more fields if you store translation edits in pendingEdits */}
          </div>
        );
      }
    },
    {
      title: 'Last Update',
      key: 'last_update',
      dataIndex: 'last_update',
      render: (text) => text ? new Date(text).toLocaleString() : ''
    },
    {
      title: 'Status',
      key: 'status',
      render: (_, record) => {
        // Show status from first translation (or blank)
        let status = record.translations && record.translations[0]?.status;
        let color = '#1976d2';
        let label = '';
        if (status === 'approved') {
          color = '#27ae60';
          label = 'Approved';
        } else if (status === 'pending' || !status) {
          color = '#ffa726';
          label = 'Pending';
        } else if (status === 'rejected') {
          color = '#e53935';
          label = 'Rejected';
        } else {
          label = status ? status.charAt(0).toUpperCase() + status.slice(1) : 'Pending';
        }
        return (
          <span style={{ display: 'inline-block', minWidth: 80 }}>
            <span style={{ background: color, color: '#fff', borderRadius: 4, padding: '2px 12px', fontWeight: 600 }}>{label}</span>
          </span>
        );
      }
    },
    {
      title: 'Actions',
      key: 'actions',
      render: (_, record) => {
        // Show Approve and Delete buttons in Pending tab
        if (activeTab === 'Pending') {
          return (
            <div style={{ display: 'flex', gap: 8 }}>
              <Button type="primary" onClick={async () => {
                try {
                  await axios.post(`/api/medical_words/${record.edit_request_id}/approve_edit`, {
                    actor_id: userId,
                    actor_role: role,
                    actor_email: userEmail
                  });
                  message.success('Edit approved');
                  fetchWords();
                  fetchPendingEdits();
                } catch (e) {
                  message.error('Failed to approve edit');
                }
              }}>Approve</Button>
              <Button danger onClick={async () => {
                try {
                  await axios.delete(`/api/medical_words/${record.edit_request_id}/edit_request`, {
                    data: {
                      actor_id: userId,
                      actor_role: role,
                      actor_email: userEmail
                    }
                  });
                  message.success('Edit request deleted');
                  fetchPendingEdits();
                } catch (e) {
                  message.error('Failed to delete edit request');
                }
              }}>Delete</Button>
            </div>
          );
        }
        // Only show Delete button in Delete tab
        if (activeTab === 'Delete') {
          return (
            <Button danger onClick={async () => {
              // Delete all translations for this word
              try {
                await axios.delete(`/api/medical_word_translations/by_word/${record.id}`);
                await axios.delete(`/api/medical_words/${record.id}`, {
                  data: {
                    actor_id: userId,
                    actor_role: role,
                    actor_email: userEmail,
                    source_word: record.source_word
                  }
                });
                message.success('Word and all translations deleted');
                fetchWords();
                setSelectedRowKeys(selectedRowKeys.filter(id => id !== record.id));
              } catch (e) {
                message.error('Failed to delete word and translations');
              }
            }}>Delete</Button>
          );
        }
        // Default: Edit button
        return (
          <Button
            onClick={() => openEditModal(record)}
          >Edit</Button>
        );
      }
    }
  ];

  // Add source word and translation
  const handleAddWord = async (values) => {
    try {
      // 1. Add the source word
      const wordRes = await axios.post('/api/add_medical_word', {
        source_language: 'English',
        source_word: values.source_word,
        status: values.status,
        date_created: new Date().toISOString(),
        created_by: userId,
        update_by: userId,
        last_update: new Date().toISOString()
      });
      const wordId = wordRes.data.id;

      // 2. Add the translation
      await axios.post('/api/add_medical_word_translation', {
        word_id: wordId,
        language: values.language,
        target_word: values.target_word,
        description: values.description,
        date_created: new Date().toISOString(),
        created_by: userId,
        update_by: userId,
        last_update: new Date().toISOString()
      });

      setAddModalVisible(false);
      addForm.resetFields();
      fetchWords();
      message.success('Medical word and translation added');
    } catch (e) {
      message.error('Failed to add medical word and translation');
    }
  };

  // Fetch joined word and translation by ID for edit modal
  const fetchWordWithTranslationById = async (id) => {
    try {
      const res = await axios.get(`/api/medical_words_with_translations/${id}`);
      if (res.data && res.data.word) {
        return res.data.word;
      }
      return null;
    } catch (e) {
      message.error('Failed to fetch word and translation details');
      return null;
    }
  };

  // Helper: get first translation from word object
  function getFirstTranslation(word) {
    if (!word || !word.translations || word.translations.length === 0) return {};
    return word.translations[0];
  }

  // Edit modal
  const openEditModal = async (word) => {
    setEditModalVisible(true);
    // Fetch latest joined word/translation data from backend
    const latestWord = await fetchWordWithTranslationById(word.id);
    if (latestWord) {
      console.log('Fetched word from backend:', latestWord);
      setSelectedWord(latestWord);
    } else {
      // fallback to passed word if fetch fails
      setSelectedWord(word);
    }
  };

  // Set form values for edit modal when selectedWord and languageOptions are ready
  useEffect(() => {
    if (editModalVisible && selectedWord && languageOptions.length > 0) {
      let firstTranslation = getFirstTranslation(selectedWord);
      if (!firstTranslation) firstTranslation = {};
      // Show the actual translation values in the modal
      editForm.setFieldsValue({
        source_word: selectedWord.source_word ?? '',
        description: firstTranslation.description ?? '',
        target_language: firstTranslation.language ?? '',
        target_translation: firstTranslation.target_word ?? ''
      });
      setDescValue(firstTranslation.description ?? '');
    }
  }, [editModalVisible, selectedWord, languageOptions]);

  // Edit word (request)
  const handleEditWord = async (values) => {
    message.loading({ content: 'Saving...', key: 'editWord' });
    try {
      // Send edit request to backend with translation data
      await axios.post(`/api/medical_words/${selectedWord.id}/edit`, {
        source_word: values.source_word,
        description: values.description,
        requested_by: userId,
        translation: {
          language: values.target_language,
          target_word: values.target_translation,
          description: values.description
        }
      });
      setEditModalVisible(false);
      fetchWords();
      fetchPendingEdits();
      message.success({ content: 'Edit request submitted for approval', key: 'editWord' });
    } catch (e) {
      message.error({ content: 'Failed to submit edit request', key: 'editWord' });
    }
  };

  // Handle update source word
  const handleUpdateSourceWord = async (values) => {
    try {
      await axios.put(`/api/medical_words/${selectedSourceWord.id}`, {
        source_word: values.source_word,
        source_language: values.source_language,
        actor_id: userId,
        actor_role: role,
        actor_email: userEmail
      });
      setEditSourceWordModalVisible(false);
      setSelectedSourceWord(null);
      setSourceWordInput('');
      editSourceWordForm.resetFields();
      fetchWords();
      message.success('Source word updated successfully');
    } catch (e) {
      const errorMsg = e.response?.data?.error || 'Failed to update source word';
      message.error(errorMsg);
    }
  };

  // Approve word
  const handleApproveWord = async (id) => {
    try {
      await axios.post(`/api/medical_words/${id}/approve`);
      fetchWords();
      fetchPendingWords();
      message.success('Word approved');
    } catch (e) {
      message.error('Failed to approve word');
    }
  };

  // Approve edit
  const handleApproveEdit = async (id) => {
    try {
      await axios.post(`/api/medical_words/${id}/approve_edit`);
      fetchWords();
      fetchPendingEdits();
      message.success('Edit approved');
    } catch (e) {
      message.error('Failed to approve edit');
    }
  };

  // Translation modal logic
  const openTranslationModal = async (word) => {
    setSelectedWord(word);
    setTranslationModalVisible(true);
    setTranslations(word.translations || []);
  };

  // Add translation for selected word
  const handleAddTranslation = async (values) => {
    try {
      await axios.post('/api/medical_word_translations', {
        word_id: selectedWord.id,
        language: values.language,
        target_word: values.target_word,
        description: values.description,
        created_by: userId
      });
      fetchWords();
      openTranslationModal(selectedWord);
      message.success('Translation added');
    } catch (e) {
      message.error('Failed to add translation');
    }
  };

  return (
    <div>
      <Tabs activeKey={activeTab} onChange={setActiveTab}>
        <>
          {/* Glossary Tab - Only visible if user has access_key = 119 */}
          {hasGlossaryAccess && (
            <TabPane tab="Glossary" key="Glossary">
              {/* Glossary Tab Table (only approved, no Actions column, no Add Word button) */}
              <div
                className="d-flex flex-wrap align-items-center mb-3"
                style={{ gap: '12px' }}
              >
                <div className="flex-grow-1" style={{ minWidth: 220 }}>
                  <Input.Search
                    placeholder="Search by source word, target word, language, description..."
                    value={search}
                    onChange={e => setSearch(e.target.value)}
                    onSearch={val => setSearch(val)}
                    className="w-100"
                  />
                </div>
                <div style={{ minWidth: 160 }}>
                  <Select
                    placeholder="Filter by language"
                    allowClear
                    className="w-100"
                    value={selectedLanguage || undefined}
                    onChange={value => {
                      setSelectedLanguage(value);
                      setPage(1);
                    }}
                  >
                    {languageOptions.map(opt => (
                      <Select.Option key={opt.label} value={opt.label}>{opt.label}</Select.Option>
                    ))}
                  </Select>
                </div>
                <div className="d-flex align-items-center" style={{ minWidth: 120 }}>
                  <span className="me-2">Rows per page:</span>
                  <Select
                    value={pageSize}
                    className="w-100"
                    style={{ minWidth: 70 }}
                    onChange={value => {
                      if (value === 'all') {
                        setPageSize(filteredWords.length || 1);
                        setPage(1);
                      } else {
                        setPageSize(Number(value));
                        setPage(1);
                      }
                    }}
                  >
                    <Select.Option value={10}>10</Select.Option>
                    <Select.Option value={25}>25</Select.Option>
                    <Select.Option value={50}>50</Select.Option>
                    <Select.Option value={100}>100</Select.Option>
                    <Select.Option value={'all'}>All</Select.Option>
                  </Select>
                </div>
              </div>
              <Spin spinning={loading}>
                <Table
                  columns={columns.filter(col => col.key !== 'actions' && col.key !== 'status')}
                  dataSource={filteredWords
                    .filter(w =>
                      w.translations.some(t => t.status === 'approved') &&
                      (!selectedLanguage || w.translations.some(t => t.language === selectedLanguage))
                    )
                    .sort((a, b) => {
                      // Order by last_update DESC, then id DESC
                      const aTime = a.last_update ? new Date(a.last_update).getTime() : 0;
                      const bTime = b.last_update ? new Date(b.last_update).getTime() : 0;
                      if (bTime !== aTime) return bTime - aTime;
                      return b.id - a.id;
                    })
                    .slice((page - 1) * pageSize, page * pageSize)
                  }
                  rowKey="id"
                  pagination={{
                    current: page,
                    pageSize,
                    total,
                    onChange: setPage,
                    onShowSizeChange: (current, size) => setPageSize(size)
                  }}
                />
              </Spin>
            </TabPane>
          )}
          {hasEditAccess && (
            <TabPane tab="Edit" key="Edit">
              {/* Edit Tab Table (with Status column) */}
              <div style={{ marginBottom: 16, display: 'flex', alignItems: 'center', gap: 8 }}>
                <Input.Search
                  placeholder="Search by source word, target word, language, description..."
                  value={search}
                  onChange={e => setSearch(e.target.value)}
                  onSearch={val => setSearch(val)}
                  style={{ width: 300 }}
                />
                {hasAddWordAccess && (
                  <Button type="primary" onClick={() => setAddModalVisible(true)}>
                    Add Word
                  </Button>
                )}
                {hasEditSourceWordAccess && (
                  <Button onClick={() => setEditSourceWordModalVisible(true)}>
                    Edit Source Word
                  </Button>
                )}
                <span style={{ marginLeft: 'auto', display: 'flex', alignItems: 'center' }}>
                  <span style={{ marginRight: 8 }}>Rows per page:</span>
                  <Select
                    value={pageSize}
                    style={{ width: 100 }}
                    onChange={value => {
                      if (value === 'all') {
                        setPageSize(filteredWords.length || 1);
                        setPage(1);
                      } else {
                        setPageSize(Number(value));
                        setPage(1);
                      }
                    }}
                  >
                    <Select.Option value={10}>10</Select.Option>
                    <Select.Option value={25}>25</Select.Option>
                    <Select.Option value={50}>50</Select.Option>
                    <Select.Option value={100}>100</Select.Option>
                    <Select.Option value={'all'}>All</Select.Option>
                  </Select>
                </span>
              </div>
              <Spin spinning={loading}>
                <Table
                  columns={columns}
                  dataSource={filteredWords
                    .slice()
                    .sort((a, b) => {
                      // Order by last_update DESC, then id DESC
                      const aTime = a.last_update ? new Date(a.last_update).getTime() : 0;
                      const bTime = b.last_update ? new Date(b.last_update).getTime() : 0;
                      if (bTime !== aTime) return bTime - aTime;
                      return b.id - a.id;
                    })
                    .slice((page - 1) * pageSize, page * pageSize)
                  }
                  rowKey="id"
                  pagination={{
                    current: page,
                    pageSize,
                    total,
                    onChange: setPage,
                    onShowSizeChange: (current, size) => setPageSize(size)
                  }}
                />
              </Spin>
            </TabPane>
          )}
          {hasPendingAccess && (
            <TabPane tab="Pending" key="Pending">
              {/* Pending Tab Table: Display all rows from medical_word_edit_request with Approve and Delete buttons */}
              <div style={{ marginBottom: 16, display: 'flex', alignItems: 'center' }}>
                <Input.Search
                  placeholder="Search by source word, target word, language, description..."
                  value={search}
                  onChange={e => setSearch(e.target.value)}
                  onSearch={val => setSearch(val)}
                  style={{ width: 300 }}
                />
                <span style={{ marginLeft: 'auto', display: 'flex', alignItems: 'center' }}>
                  <span style={{ marginRight: 8 }}>Rows per page:</span>
                  <Select
                    value={pageSize}
                    style={{ width: 100 }}
                    onChange={value => {
                      if (value === 'all') {
                        setPageSize(pendingEdits.length || 1);
                        setPage(1);
                      } else {
                        setPageSize(Number(value));
                        setPage(1);
                      }
                    }}
                  >
                    <Select.Option value={10}>10</Select.Option>
                    <Select.Option value={25}>25</Select.Option>
                    <Select.Option value={50}>50</Select.Option>
                    <Select.Option value={100}>100</Select.Option>
                    <Select.Option value={'all'}>All</Select.Option>
                  </Select>
                </span>
              </div>
              <Spin spinning={loading}>
                <Table
                  columns={columns}
                  dataSource={pendingEdits
                    .sort((a, b) => {
                      // Order by last_update DESC
                      const aTime = a.last_update ? new Date(a.last_update).getTime() : 0;
                      const bTime = b.last_update ? new Date(b.last_update).getTime() : 0;
                      return bTime - aTime;
                    })
                    .slice((page - 1) * pageSize, page * pageSize)
                  }
                  rowKey="edit_request_id"
                  pagination={{
                    current: page,
                    pageSize,
                    total: pendingEdits.length,
                    onChange: setPage,
                    onShowSizeChange: (current, size) => setPageSize(size)
                  }}
                />
              </Spin>
            </TabPane>
          )}
          {hasApprovedAccess && (
            <TabPane tab="Approved" key="Approved">
              {/* Approved Tab Table: show only translations with status === 'approved' */}
              <Spin spinning={loading}>
                <Table
                  columns={columns}
                  dataSource={words
                    .filter(w => w.translations.some(t => t.status === 'approved'))
                    .sort((a, b) => {
                      const aTime = a.last_update ? new Date(a.last_update).getTime() : 0;
                      const bTime = b.last_update ? new Date(b.last_update).getTime() : 0;
                      if (bTime !== aTime) return bTime - aTime;
                      return b.id - a.id;
                    })
                  }
                  rowKey="id"
                  pagination={false}
                />
              </Spin>
            </TabPane>
          )}
          {hasDeleteAccess && (
            <TabPane tab="Delete" key="Delete">
              {/* Multi-delete button for selected rows */}
              {selectedRowKeys.length > 0 && (
                <Button
                  danger
                  style={{ marginBottom: 16 }}
                  onClick={async () => {
                    for (const id of selectedRowKeys) {
                      try {
                        await axios.delete(`/api/medical_word_translations/by_word/${id}`);
                        await axios.delete(`/api/medical_words/${id}`);
                      } catch (e) {}
                    }
                    message.success('Selected words deleted');
                    setSelectedRowKeys([]);
                    fetchWords();
                  }}
                >Delete Selected ({selectedRowKeys.length})</Button>
              )}
              {/* Delete Tab Table (copied from Edit, fetches all words) */}
              <div style={{ marginBottom: 16, display: 'flex', alignItems: 'center' }}>
                <Input.Search
                  placeholder="Search by source word, target word, language, description..."
                  value={search}
                  onChange={e => setSearch(e.target.value)}
                  onSearch={val => setSearch(val)}
                  style={{ width: 300 }}
                />
                <span style={{ marginLeft: 'auto', display: 'flex', alignItems: 'center' }}>
                  <span style={{ marginRight: 8 }}>Rows per page:</span>
                  <Select
                    value={pageSize}
                    style={{ width: 100 }}
                    onChange={value => {
                      if (value === 'all') {
                        setPageSize(words.length || 1);
                        setPage(1);
                      } else {
                        setPageSize(Number(value));
                        setPage(1);
                      }
                    }}
                  >
                    <Select.Option value={10}>10</Select.Option>
                    <Select.Option value={25}>25</Select.Option>
                    <Select.Option value={50}>50</Select.Option>
                    <Select.Option value={100}>100</Select.Option>
                    <Select.Option value={'all'}>All</Select.Option>
                  </Select>
                </span>
              </div>
              <Spin spinning={loading}>
                <Table
                  columns={columns}
                  dataSource={words
                    .slice()
                    .sort((a, b) => {
                      // Order by last_update DESC, then id DESC
                      const aTime = a.last_update ? new Date(a.last_update).getTime() : 0;
                      const bTime = b.last_update ? new Date(b.last_update).getTime() : 0;
                      if (bTime !== aTime) return bTime - aTime;
                      return b.id - a.id;
                    })
                    .slice((page - 1) * pageSize, page * pageSize)
                  }
                  rowKey="id"
                  pagination={{
                    current: page,
                    pageSize,
                    total: words.length,
                    onChange: setPage,
                    onShowSizeChange: (current, size) => setPageSize(size)
                  }}
                />
              </Spin>
            </TabPane>
          )}
        </>
      </Tabs>
      <Modal
        title="Add Medical Word"
        open={addModalVisible}
        onCancel={() => setAddModalVisible(false)}
        footer={null}
      >
        <Form
          layout="vertical"
          form={addForm}
          onFinish={handleAddWord}
        >
          <Form.Item label="Source Language">
            <Input value="English" disabled />
          </Form.Item>
          <Form.Item name="source_word" label="Source Word (English)" rules={[{ required: true, message: "'source_word' is required" }]}>
            <Input value={addForm.getFieldValue('source_word') || ''} onChange={(e) => addForm.setFieldsValue({ source_word: e.target.value })} />
          </Form.Item>
          
          <Form.Item name="language" label="Target Language" rules={[{ required: true, message: 'Target language is required' }]}> 
            <Select showSearch placeholder="Select target language">
              {languageOptions.map(opt => (
                <Select.Option key={opt.label} value={opt.label}>{opt.label}</Select.Option>
              ))}
            </Select>
          </Form.Item>
          <Form.Item name="target_word" label="Target Word" rules={[{ required: true, message: 'Target word is required' }]}> 
            <Input />
          </Form.Item>

          <Form.Item
            name="description"
            label="Description"
            rules={[{ required: false, message: 'Description is required' }]}
            getValueFromEvent={e => (typeof e === 'string' ? e : descValue)}
          >
            <Editor
              apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
              value={descValue}
              onEditorChange={content => {
                setDescValue(content);
                addForm.setFieldsValue({ description: content });
              }}
              tinymceScriptSrc={tinymceCDN}
              init={{
                height: 120,
                menubar: false,
                plugins: 'lists link charmap preview anchor',
                toolbar: 'undo redo | formatselect | bold italic | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | removeformat'
              }}
            />
          </Form.Item>
          <Form.Item
            name="status"
            label="Status"
            rules={[{ required: true, message: 'Status is required' }]}
            initialValue={role === 'Administrator' ? 'approved' : 'pending'}
          >
            <Select
              disabled={role !== 'Administrator'}
            >
              <Select.Option value="pending">Pending</Select.Option>
              <Select.Option value="approved">Approved</Select.Option>
            </Select>
          </Form.Item>
          <Button type="primary" onClick={() => addForm.submit()}>Add</Button>
        </Form>
      </Modal>

      {/* Edit Word Modal - same as Add Word Modal */}
      <Modal
        title="Edit Medical Word"
        open={editModalVisible}
        onCancel={() => {
          setEditModalVisible(false);
          editForm.resetFields();
          setDescValue(selectedWord?.description || '');
        }}
        footer={null}
      >
        <Form
          layout="vertical"
          form={editForm}
          onFinish={handleEditWord}
        >
          <Form.Item label="Source Language">
            <Input value="English" disabled />
          </Form.Item>
          <Form.Item name="source_word" label="Source Word (English)" rules={[{ required: true, message: "'source_word' is required" }]}>
            <Input disabled />
          </Form.Item>
          <Form.Item name="description" label="Description" rules={[{ required: false, message: 'Description is required' }]} getValueFromEvent={e => (typeof e === 'string' ? e : descValue)}>
            <Editor
              apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
              value={descValue}
              onEditorChange={content => {
                setDescValue(content);
                editForm.setFieldsValue({ description: content });
              }}
              tinymceScriptSrc={tinymceCDN}
              init={{
                height: 120,
                menubar: false,
                plugins: 'lists link charmap preview anchor',
                toolbar: 'undo redo | formatselect | bold italic | alignleft aligncenter alignright alignjustify | bullist numlist outdent indent | removeformat'
              }}
            />
          </Form.Item>
          <Form.Item name="target_language" label="Target Language" rules={[{ required: true, message: "'target_language' is required" }]}>
            <Select showSearch placeholder="Select target language">
              {languageOptions.map(opt => (<Option key={opt.value} value={opt.label}>{opt.label}</Option>))}
            </Select>
          </Form.Item>
          <Form.Item name="target_translation" label="Target Translation" rules={[{ required: true, message: "'target_translation' is required" }]}>
            <Input />
          </Form.Item>
          <Button type="primary" onClick={() => editForm.submit()}>Save</Button>
        </Form>
      </Modal>
      {/* Translation Modal */}
      <Modal
  title={`Translations for ${selectedWord?.source_word}`}
  open={translationModalVisible}
  onCancel={() => setTranslationModalVisible(false)}
  footer={null}
      >
        <Table
          columns={[
            { title: 'Language', dataIndex: 'language', key: 'language' },
            { title: 'Translation', dataIndex: 'translation', key: 'translation' },
            { title: 'Description', dataIndex: 'description', key: 'description' }
          ]}
          dataSource={translations}
          rowKey="language"
          pagination={false}
        />
        <Form layout="vertical" onFinish={handleAddTranslation} style={{ marginTop: 16 }}>
          <Form.Item name="language" label="Target Language" rules={[{ required: true, message: 'Target language is required' }]}> <Input /> </Form.Item>
          <Form.Item name="target_word" label="Target Word" rules={[{ required: true, message: 'Target word is required' }]}> <Input /> </Form.Item>
          <Form.Item name="description" label="Description"> <Input.TextArea /> </Form.Item>
          <Button type="primary" htmlType="submit">Add Translation</Button>
        </Form>
      </Modal>

      {/* Edit Source Word Modal */}
      <Modal
        title="Edit Source Word"
        open={editSourceWordModalVisible}
        onCancel={() => {
          setEditSourceWordModalVisible(false);
          setSelectedSourceWord(null);
          setSourceWordInput('');
          editSourceWordForm.resetFields();
        }}
        footer={null}
      >
        <Form layout="vertical" form={editSourceWordForm} onFinish={handleUpdateSourceWord}>
          <Form.Item 
            label="Select Source Word" 
            rules={[{ required: true, message: 'Please select a source word' }]}
          >
            <Select
              placeholder="Select a source word to edit"
              allowClear
              showSearch
              optionFilterProp="children"
              filterOption={(input, option) =>
                (option?.label ?? '').toLowerCase().includes(input.toLowerCase())
              }
              onChange={(value) => {
                const selected = filteredWords.find(w => w.id === value);
                setSelectedSourceWord(selected);
                editSourceWordForm.setFieldsValue({ source_word: selected?.source_word || '' });
              }}
              options={filteredWords.map(w => ({
                label: w.source_word,
                value: w.id
              }))}
            />
          </Form.Item>
          
          {selectedSourceWord && (
            <Form.Item 
              name="source_word" 
              label="New Source Word" 
              rules={[{ required: true, message: 'New source word is required' }]}
            >
              <Input placeholder="Enter new source word" />
            </Form.Item>
          )}
          
          <Button type="primary" htmlType="submit" block>
            Update Source Word
          </Button>
        </Form>
      </Modal>
    </div>
  );
}

export default MedicalDocumentGlossary;
