import { Chart, ArcElement, Tooltip, Legend } from 'chart.js';
import ChartDataLabels from 'chartjs-plugin-datalabels';
Chart.register(ArcElement, Tooltip, Legend, ChartDataLabels);

import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { Doughnut } from 'react-chartjs-2';


const columns = [
  { key: 'id', label: 'ID' },
  { key: 'site_url', label: 'Site URL' },
  { key: 'visit_datetime', label: 'Date/Time' },
  { key: 'visitor_name', label: 'Name' },
  { key: 'visitor_email', label: 'Email' },
  { key: 'country', label: 'Country' },
  { key: 'browser', label: 'Browser' },
  { key: 'device', label: 'Device' },
  { key: 'referrer', label: 'Referrer' },
  { key: 'ip_address', label: 'IP' },
  { key: 'page_title', label: 'Page Title' },
];


export default function WebsiteTrafficPage() {
  const [rows, setRows] = useState([]);
  const [selectedRows, setSelectedRows] = useState([]);
  const [filters, setFilters] = useState({ country: '', device: '', criteria: 'today', customStart: '', customEnd: '' });
  const [chartData, setChartData] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  // Helper to get date range from criteria
  function getDateRange(criteria, customStart, customEnd) {
    const now = new Date();
    let start, end;
    switch (criteria) {
      case 'today':
        start = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
        break;
      case 'this_week': {
        // Monday as first day of week
        const day = now.getDay() === 0 ? 6 : now.getDay() - 1;
        start = new Date(now.getFullYear(), now.getMonth(), now.getDate() - day);
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
        break;
      }
      case 'last_week': {
        // Monday as first day of week
        const day = now.getDay() === 0 ? 6 : now.getDay() - 1;
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() - day);
        start = new Date(end.getFullYear(), end.getMonth(), end.getDate() - 7);
        break;
      }
      case 'this_month':
        start = new Date(now.getFullYear(), now.getMonth(), 1);
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
        break;
      case 'last_month': {
        const lastMonth = new Date(now.getFullYear(), now.getMonth() - 1, 1);
        start = lastMonth;
        end = new Date(now.getFullYear(), now.getMonth(), 1);
        break;
      }
      case 'last_3_months': {
        start = new Date(now.getFullYear(), now.getMonth() - 3, 1);
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
        break;
      }
      case 'last_year':
        start = new Date(now.getFullYear() - 1, 0, 1);
        end = new Date(now.getFullYear(), 0, 1);
        break;
      case 'custom':
        start = customStart ? new Date(customStart) : new Date(now.getFullYear(), now.getMonth(), now.getDate());
        end = customEnd ? new Date(customEnd) : new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
        if (customEnd) {
          // Add 1 day to end to make it inclusive
          end.setDate(end.getDate() + 1);
        }
        break;
      default:
        start = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        end = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1);
    }
    return {
      start: start.toISOString().slice(0, 10),
      end: end.toISOString().slice(0, 10),
    };
  }


  useEffect(() => {
    fetchData();
    // eslint-disable-next-line
  }, [filters]);

  const fetchData = async () => {
    setLoading(true);
    setError(null);
    try {
      // Build params with date range from criteria
      const { start, end } = getDateRange(filters.criteria, filters.customStart, filters.customEnd);
      const params = {
        country: filters.country,
        device: filters.device,
        start_date: start,
        end_date: end,
      };
      const res = await axios.get('/api/site-visits', { params });
      // Filter rows by visit_datetime in frontend to ensure strict match
      const allRows = res.data.site_visits || [];
      const filteredRows = allRows.filter(row => {
        if (!row.visit_datetime) return false;
        const dt = row.visit_datetime.split(' ')[0];
        return dt >= start && dt < end;
      });
      setRows(filteredRows);
      // Prepare chart data (visits per site_url/category)
      const visitsBySite = {};
      filteredRows.forEach(v => {
        let label = v.site_url;
        if (label.includes('shop')) label = 'SHOP';
        else if (label.includes('school')) label = 'School';
        else if (label.includes('alolaba')) label = 'Alolaba';
        else if (label.includes('make-it')) label = 'MAKE-IT';
        visitsBySite[label] = (visitsBySite[label] || 0) + 1;
      });
      const colorMap = {
        'MAKE-IT': '#3b6ef6',
        'School': '#6cc24a',
        'SHOP': '#ffd166',
        'Alolaba': '#ef476f',
      };
      const labels = Object.keys(visitsBySite);
      const data = Object.values(visitsBySite);
      const backgroundColor = labels.map(l => colorMap[l] || '#8884d8');
      setChartData({
        labels,
        datasets: [{ data, backgroundColor }],
      });
    } catch (err) {
      setError('Failed to load data. Please check your API or network connection.');
      setRows([]);
      setChartData(null);
    } finally {
      setLoading(false);
    }
  };


  const handleDelete = async (ids) => {
    if (!Array.isArray(ids)) ids = [ids];
    await axios.post('/api/site-visits/delete', { ids });
    fetchData();
  };


  const handleFilterChange = (e) => {
    setFilters({ ...filters, [e.target.name]: e.target.value });
  };

  return (
    <div className="container py-3">
      <h3 className="mb-4">Website Traffic</h3>
      {loading && (
        <div className="alert alert-info">Loading data...</div>
      )}
      {error && (
        <div className="alert alert-danger">{error}</div>
      )}
      {!loading && chartData && chartData.datasets && chartData.datasets[0] && chartData.datasets[0].data && chartData.datasets[0].data.length > 0 && (
        <div className="mb-4 d-flex flex-column align-items-center" style={{ maxWidth: 400, margin: '0 auto' }}>
          <h5>Visits Overview</h5>
          <Doughnut
            data={chartData}
            options={{
              plugins: {
                legend: { display: true, position: 'top' },
                tooltip: { enabled: true },
                datalabels: {
                  color: '#222',
                  font: { weight: 'bold', size: 18 },
                  formatter: (value) => value,
                },
              },
              cutout: '70%',
              maintainAspectRatio: true,
              responsive: true,
            }}
          />
        </div>
      )}
      <div className="card mb-3 p-3">
        <form className="row gy-2 gx-3 align-items-end flex-wrap" onSubmit={e => e.preventDefault()}>
          <div className="col-12 col-md-3">
            <label className="form-label">Country</label>
            <input type="text" className="form-control" name="country" value={filters.country} onChange={handleFilterChange} />
          </div>
          <div className="col-12 col-md-3">
            <label className="form-label">Device</label>
            <select className="form-select" name="device" value={filters.device} onChange={handleFilterChange}>
              <option value="">All</option>
              <option value="desktop">Desktop</option>
              <option value="mobile">Mobile</option>
            </select>
          </div>
          <div className="col-12 col-md-3">
            <label className="form-label">Criteria</label>
            <select className="form-select" name="criteria" value={filters.criteria} onChange={handleFilterChange}>
              <option value="today">Today</option>
              <option value="this_week">This Week</option>
              <option value="last_week">Last Week</option>
              <option value="this_month">This Month</option>
              <option value="last_month">Last Month</option>
              <option value="last_3_months">Last 3 Months</option>
              <option value="last_year">Last Year</option>
              <option value="custom">Custom Range</option>
            </select>
          </div>
          {filters.criteria === 'custom' && (
            <>
              <div className="col-12 col-md-3">
                <label className="form-label">Start Date</label>
                <input type="date" className="form-control" name="customStart" value={filters.customStart} onChange={handleFilterChange} />
              </div>
              <div className="col-12 col-md-3">
                <label className="form-label">End Date</label>
                <input type="date" className="form-control" name="customEnd" value={filters.customEnd} onChange={handleFilterChange} />
              </div>
            </>
          )}
          <div className="col-12 col-md-3 d-flex gap-2 mt-2 mt-md-0">
            <button type="button" className="btn btn-danger" disabled={selectedRows.length === 0} onClick={() => handleDelete(selectedRows)}>
              <i className="bi bi-trash"></i> Delete Selected
            </button>
          </div>
        </form>
      </div>
      <div className="table-responsive" style={{ maxHeight: 500 }}>
        <table className="table table-bordered table-hover align-middle">
          <thead className="table-light">
            <tr>
              <th scope="col"><input type="checkbox" checked={selectedRows.length === rows.length && rows.length > 0} onChange={e => setSelectedRows(e.target.checked ? rows.map(r => r.id) : [])} /></th>
              {columns.map(col => (
                <th key={col.key} scope="col">{col.label}</th>
              ))}
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            {rows.map(row => (
              <tr key={row.id}>
                <td><input type="checkbox" checked={selectedRows.includes(row.id)} onChange={e => {
                  if (e.target.checked) setSelectedRows([...selectedRows, row.id]);
                  else setSelectedRows(selectedRows.filter(id => id !== row.id));
                }} /></td>
                {columns.map(col => (
                  <td key={col.key}>{row[col.key]}</td>
                ))}
                <td>
                  <button className="btn btn-sm btn-danger" onClick={() => handleDelete(row.id)} title="Delete"><i className="bi bi-trash"></i></button>
                </td>
              </tr>
            ))}
            {rows.length === 0 && (
              <tr><td colSpan={columns.length + 2} className="text-center">No data found.</td></tr>
            )}
          </tbody>
        </table>
      </div>
    </div>
  );
}
