import React, { useEffect, useState } from 'react';
import AddTeacherModal from './AddTeacherModal';
import { useRef } from 'react';

const Teachers = () => {
  const [teachers, setTeachers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [showAdd, setShowAdd] = useState(false);
  const [saving, setSaving] = useState(false);
  const [editTeacher, setEditTeacher] = useState(null);
  const [showEdit, setShowEdit] = useState(false);

  const fetchTeachers = () => {
    setLoading(true);
    fetch('/api/teachers')
      .then(res => res.json())
      .then(data => {
        setTeachers(data.teachers || []);
        setLoading(false);
      })
      .catch(err => {
        setError('Failed to fetch teachers');
        setLoading(false);
      });
  };

  useEffect(() => {
    fetchTeachers();
  }, []);

  const handleAddTeacher = (teacherData) => {
    setSaving(true);
    fetch('/api/teachers', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(teacherData)
    })
      .then(res => {
        setSaving(false);
        setShowAdd(false);
        if (res.ok) {
          fetchTeachers();
        } else {
          return res.json().then(data => { throw new Error(data.error || 'Failed to add teacher'); });
        }
      })
      .catch(err => {
        setError(err.message);
        setSaving(false);
      });
  };

  const handleEditTeacher = (teacher) => {
    setEditTeacher(teacher);
    setShowEdit(true);
  };

  const handleSaveEdit = (teacherData) => {
    setSaving(true);
    fetch(`/api/teachers/${editTeacher.id}`, {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(teacherData)
    })
      .then(res => {
        setSaving(false);
        setShowEdit(false);
        setEditTeacher(null);
        if (res.ok) {
          fetchTeachers();
        } else {
          return res.json().then(data => { throw new Error(data.error || 'Failed to update teacher'); });
        }
      })
      .catch(err => {
        setError(err.message);
        setSaving(false);
      });
  };

  const handleDeleteTeacher = (id) => {
    if (!window.confirm('Are you sure you want to delete this teacher?')) return;
    setSaving(true);
    fetch(`/api/teachers/${id}`, {
      method: 'DELETE'
    })
      .then(res => {
        setSaving(false);
        if (res.ok) {
          fetchTeachers();
        } else {
          return res.json().then(data => { throw new Error(data.error || 'Failed to delete teacher'); });
        }
      })
      .catch(err => {
        setError(err.message);
        setSaving(false);
      });
  };

  return (
    <div style={{padding: '32px', background: '#fff', borderRadius: '12px', boxShadow: '0 2px 12px rgba(0,0,0,0.08)', maxWidth: 1200, margin: 'auto', overflowX: 'auto'}}>
      <div style={{display:'flex', alignItems:'center', justifyContent:'space-between', marginBottom:'24px'}}>
        <h2 style={{margin: 0}}>Teachers</h2>
        <button
          style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}
          onClick={()=>setShowAdd(true)}
          disabled={saving}
        >
          Add Teacher
        </button>
      </div>
      <AddTeacherModal open={showAdd} onClose={()=>setShowAdd(false)} onSave={handleAddTeacher} />
      {showEdit && editTeacher && (
        <AddTeacherModal
          open={showEdit}
          onClose={() => { setShowEdit(false); setEditTeacher(null); }}
          onSave={handleSaveEdit}
          initialData={editTeacher}
        />
      )}
      {loading && <div>Loading...</div>}
      {error && <div style={{color: 'red'}}>{error}</div>}
      <table style={{minWidth: 1000, width: '100%', borderCollapse: 'collapse', marginBottom: '24px', overflowX: 'auto', display: 'block'}}>
        <thead>
          <tr style={{background: '#f5f5f5'}}>
            <th style={{padding: '8px', border: '1px solid #eee'}}>ID</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Name</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Email</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Subject</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Bio</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>School Type</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Created At</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Visible</th>
            <th style={{padding: '8px', border: '1px solid #eee'}}>Actions</th>
          </tr>
        </thead>
        <tbody>
          {teachers.map(t => (
            <tr key={t.id}>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.id}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.prenom} {t.nom}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.email}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.subject}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.bio}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.school_type === 'both' ? 'Both' : t.school_type}</td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>{t.created_at}</td>
              <td style={{padding: '8px', border: '1px solid #eee', textAlign: 'center'}}>
                <span style={{display: 'inline-block', minWidth: 40, fontWeight: 'bold', color: t.visible === 1 ? '#388e3c' : '#d32f2f'}}>
                  {t.visible === 1 ? 'Yes' : 'No'}
                </span>
              </td>
              <td style={{padding: '8px', border: '1px solid #eee'}}>
                <button style={{marginRight: 8, background:'#ffc107', color:'#222', border:'none', borderRadius:4, padding:'4px 12px', cursor:'pointer'}} onClick={()=>handleEditTeacher(t)}>Edit</button>
                <button style={{background:'#d32f2f', color:'#fff', border:'none', borderRadius:4, padding:'4px 12px', cursor:'pointer'}} onClick={()=>handleDeleteTeacher(t.id)}>Delete</button>
              </td>
            </tr>
          ))}
          {teachers.length === 0 && !loading && (
            <tr><td colSpan={9} style={{textAlign: 'center', padding: '16px'}}>No teachers found.</td></tr>
          )}
        </tbody>
      </table>
      {/* Add Teacher button moved to header */}
    </div>
  );
};

export default Teachers;
