import React, { useState, useEffect } from 'react';
import { BASE_API } from '../../utils/config';
import InvoicePage from './InvoicePage';
import CourseSidebar from './CourseSidebar';

// Invoice Modal Component
const InvoiceModal = ({ task, onClose, invoiceCriteria, setInvoiceCriteria, invoiceData, setInvoiceData }) => {
  const [users, setUsers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  useEffect(() => {
    fetch(`${BASE_API}/api/users`)
      .then(res => res.json())
      .then(data => setUsers(data.users || []));
  }, []);

  const handleFetchInvoice = async () => {
    setLoading(true);
    setError(null);
    try {
      // Fetch user rates
      const user = users.find(u => String(u.id) === String(invoiceCriteria.userId));
      // Fetch interpreter chart for month/minutes
      const chartRes = await fetch(`${BASE_API}/api/interpreter_chart/${invoiceCriteria.userId}`);
      const chartData = await chartRes.json();
      // Find month data
      let monthKey = `${invoiceCriteria.year}-${String(invoiceCriteria.month).padStart(2,'0')}`;
      let minutesWorked = chartData.chart && chartData.chart[monthKey] ? chartData.chart[monthKey] : 0;
      setInvoiceData({
        user,
        rate: user?.rate || 0,
        month: invoiceCriteria.month,
        year: invoiceCriteria.year,
        minutesWorked,
        task
      });
    } catch (err) {
      setError('Failed to fetch invoice data');
    }
    setLoading(false);
  };

  // Editable invoice preview
  return (
    <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={onClose}>
      <div style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'540px', maxWidth:'90vw', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}} onClick={e=>e.stopPropagation()}>
        <h3>Generate Invoice</h3>
        <div style={{display:'flex', gap:'16px', marginBottom:'18px'}}>
          <select value={invoiceCriteria.userId} onChange={e=>setInvoiceCriteria({...invoiceCriteria, userId: e.target.value})} style={{padding:'8px', borderRadius:'8px', border:'1px solid #ccc'}}>
            <option value="">Select User</option>
            {users.map(u => <option key={u.id} value={u.id}>{u.name || u.email}</option>)}
          </select>
          <select value={invoiceCriteria.month} onChange={e=>setInvoiceCriteria({...invoiceCriteria, month: e.target.value})} style={{padding:'8px', borderRadius:'8px', border:'1px solid #ccc'}}>
            <option value="">Month</option>
            {[...Array(12)].map((_,i)=>(<option key={i+1} value={i+1}>{new Date(2000,i).toLocaleString('default',{month:'long'})}</option>))}
          </select>
          <select value={invoiceCriteria.year} onChange={e=>setInvoiceCriteria({...invoiceCriteria, year: e.target.value})} style={{padding:'8px', borderRadius:'8px', border:'1px solid #ccc'}}>
            <option value="">Year</option>
            {[2024,2025,2026].map(y=>(<option key={y} value={y}>{y}</option>))}
          </select>
          <button style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 18px', fontWeight:'bold', cursor:'pointer'}} onClick={handleFetchInvoice} disabled={loading || !invoiceCriteria.userId || !invoiceCriteria.month || !invoiceCriteria.year}>Fetch</button>
        </div>
        {error && <div style={{color:'#f75c7a', marginBottom:'8px'}}>{error}</div>}
        {invoiceData && (
          <div style={{background:'#f7f8fa', borderRadius:'12px', padding:'24px', marginTop:'12px'}}>
            <h2 style={{textAlign:'center', letterSpacing:'2px', fontWeight:'bold', marginBottom:'18px'}}>INVOICE</h2>
            <div style={{display:'flex', justifyContent:'space-between', marginBottom:'18px'}}>
              <div>
                <div><b>To :</b> {invoiceData.user?.name || invoiceData.user?.email}</div>
                <div><b>Month :</b> {new Date(2000,invoiceData.month-1).toLocaleString('default',{month:'long'})} {invoiceData.year}</div>
                <div><b>Email :</b> {invoiceData.user?.email}</div>
              </div>
              <div>
                <div><b>Invoice # :</b> {Math.floor(Math.random()*1000000000)}</div>
                <div><b>Date :</b> {new Date().toISOString().slice(0,10)}</div>
                <div><b>Amount Due :</b> $ {(invoiceData.minutesWorked * invoiceData.rate).toFixed(2)}</div>
              </div>
            </div>
            <table style={{width:'100%', borderCollapse:'collapse', marginBottom:'18px'}}>
              <thead>
                <tr>
                  <th style={{border:'1px solid #eee', padding:'8px'}}>Item</th>
                  <th style={{border:'1px solid #eee', padding:'8px'}}>Description</th>
                  <th style={{border:'1px solid #eee', padding:'8px'}}>Rate</th>
                  <th style={{border:'1px solid #eee', padding:'8px'}}>Quantity</th>
                  <th style={{border:'1px solid #eee', padding:'8px'}}>Total Price</th>
                </tr>
              </thead>
              <tbody>
                <tr>
                  <td style={{border:'1px solid #eee', padding:'8px'}}>Interpretation</td>
                  <td style={{border:'1px solid #eee', padding:'8px'}}>
                    <input type="text" value={invoiceData.description || 'Service'} onChange={e=>setInvoiceData({...invoiceData, description: e.target.value})} style={{width:'100%', padding:'6px', borderRadius:'6px', border:'1px solid #ccc'}} />
                  </td>
                  <td style={{border:'1px solid #eee', padding:'8px'}}>
                    <input type="number" value={invoiceData.rate} min="0" step="0.01" onChange={e=>setInvoiceData({...invoiceData, rate: parseFloat(e.target.value)})} style={{width:'80px', padding:'6px', borderRadius:'6px', border:'1px solid #ccc'}} />
                  </td>
                  <td style={{border:'1px solid #eee', padding:'8px'}}>
                    <input type="number" value={invoiceData.minutesWorked} min="0" step="1" onChange={e=>setInvoiceData({...invoiceData, minutesWorked: parseInt(e.target.value)})} style={{width:'80px', padding:'6px', borderRadius:'6px', border:'1px solid #ccc'}} />
                  </td>
                  <td style={{border:'1px solid #eee', padding:'8px'}}>${(invoiceData.minutesWorked * invoiceData.rate).toFixed(2)}</td>
                </tr>
              </tbody>
            </table>
            <div style={{display:'flex', gap:'24px', marginBottom:'12px'}}>
              <div><b>Total:</b> ${(invoiceData.minutesWorked * invoiceData.rate).toFixed(2)}</div>
              <div><b>Amount Paid:</b> $0</div>
              <div><b>Balance Due:</b> ${(invoiceData.minutesWorked * invoiceData.rate).toFixed(2)}</div>
            </div>
            <div style={{marginTop:'18px', fontStyle:'italic', color:'#888'}}>This invoice will be executing within 15 following days</div>
          </div>
        )}
        <button onClick={onClose} style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer', marginTop:'18px'}}>Close</button>
      </div>
    </div>
  );
};


const TasksManager = () => {
  const [showRolePopup, setShowRolePopup] = useState(false);
  const [showTaskPopup, setShowTaskPopup] = useState(false);
  const [editTask, setEditTask] = useState(null);
  const [overlayTask, setOverlayTask] = useState(null);
  const [tasks, setTasks] = useState([]);
  const [filter, setFilter] = useState('');
  const [loading, setLoading] = useState(false);
  const [showInvoiceModal, setShowInvoiceModal] = useState(false);
  const [invoiceCriteria, setInvoiceCriteria] = useState({ userId: '', month: '', year: '' });
  const [invoiceData, setInvoiceData] = useState(null);
  const [selectedMenu, setSelectedMenu] = useState('');

  useEffect(() => {
    fetch(`${BASE_API}/api/tasks`)
      .then(res => res.json())
      .then(setTasks);
  }, []);

  const handleTaskCreated = (newTask) => {
    setTasks([...tasks, newTask]);
    setShowTaskPopup(false);
  };

  const handleTaskUpdated = (updatedTask) => {
    setTasks(tasks.map(t => t.id === updatedTask.id ? updatedTask : t));
    setEditTask(null);
  };

  const handleDeleteTask = (taskId) => {
    if (!window.confirm('Are you sure you want to delete this task?')) return;
    setLoading(true);
    fetch(`${BASE_API}/api/tasks/${taskId}`, { method: 'DELETE' })
      .then(res => {
        if (!res.ok) throw new Error('Failed to delete task');
        setTasks(tasks => tasks.filter(t => t.id !== taskId));
        setLoading(false);
      })
      .catch(() => setLoading(false));
  };

  const filteredTasks = tasks.filter(task =>
    task.name.toLowerCase().includes(filter.toLowerCase()) ||
    (Array.isArray(task.access) ? task.access.join(', ').toLowerCase().includes(filter.toLowerCase()) : String(task.access).toLowerCase().includes(filter.toLowerCase()))
  );

  return (
    <div style={{padding:'32px'}}>
      {/* Removed CourseSidebar from main content; dropdowns now only in sidebar */}
      {selectedMenu === 'invoicing' ? (
        <InvoicePage />
      ) : (
        <>
          <h2>Tasks Manager</h2>
          <div style={{display:'flex', gap:'16px', marginBottom:'24px'}}>
            <button style={{background:'#a259e6', color:'#fff', border:'none', borderRadius:'8px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={()=>setShowRolePopup(true)}>
              Add Role
            </button>
            <button style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={()=>setShowTaskPopup(true)}>
              Add Task
            </button>
            <input type="text" placeholder="Filter tasks..." value={filter} onChange={e=>setFilter(e.target.value)} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc', flex:'1', minWidth:'180px'}} />
          </div>
          {showRolePopup && <RolePopup onClose={()=>setShowRolePopup(false)} />}
          {showTaskPopup && <TaskPopup onClose={()=>setShowTaskPopup(false)} onCreated={handleTaskCreated} />}
          {editTask && <TaskPopup task={editTask} onClose={()=>setEditTask(null)} onUpdated={handleTaskUpdated} />}
          <div style={{marginTop:'32px'}}>
            <h3>Available Tasks</h3>
          </div>
          <div style={{marginTop:'32px'}}>
            <h3>Available Tasks</h3>
            <table style={{width:'100%', borderCollapse:'collapse', marginTop:'12px'}}>
              <thead>
                <tr>
                  <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Task</th>
                  <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Who Can Access</th>
                  <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Details</th>
                  <th style={{border:'1px solid #eee', padding:'10px', background:'#f7f8fa'}}>Actions</th>
                </tr>
              </thead>
              <tbody>
                {filteredTasks.map((task, idx) => (
                  <tr key={task.id || task.name}>
                    <td style={{border:'1px solid #eee', padding:'10px'}}>{task.name}</td>
                    <td style={{border:'1px solid #eee', padding:'10px'}}>{Array.isArray(task.access) ? task.access.join(', ') : task.access}</td>
                    <td style={{border:'1px solid #eee', padding:'10px'}}>
                      <button style={{background:'#eaf6ff', color:'#1565c0', border:'none', borderRadius:'6px', padding:'6px 16px', cursor:'pointer'}} onClick={()=>setOverlayTask(task)}>
                        View Access
                      </button>
                    </td>
                    <td style={{border:'1px solid #eee', padding:'10px', display:'flex', gap:'8px', alignItems:'center'}}>
                      <button style={{background:'#43cea2', color:'#fff', border:'none', borderRadius:'6px', padding:'6px 16px', cursor:'pointer'}} onClick={()=>alert(`Special Invoice action for task: ${task.name}`)}>
                        Invoice
                      </button>
                      <button style={{background:'#f7f8fa', color:'#1565c0', border:'1px solid #a259e6', borderRadius:'6px', padding:'6px 16px', cursor:'pointer'}} onClick={()=>setEditTask(task)}>
                        Edit
                      </button>
                      <button style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'6px', padding:'6px 16px', cursor:'pointer'}} onClick={()=>handleDeleteTask(task.id)}>
                        Delete
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          {overlayTask && <TaskAccessOverlay task={overlayTask} onClose={()=>setOverlayTask(null)} />}
          {showInvoiceModal && (
            <InvoiceModal
              task={showInvoiceModal}
              onClose={()=>{setShowInvoiceModal(false); setInvoiceData(null);}}
              invoiceCriteria={invoiceCriteria}
              setInvoiceCriteria={setInvoiceCriteria}
              invoiceData={invoiceData}
              setInvoiceData={setInvoiceData}
            />
          )}
        </>
      )}
    </div>
  );
};

// removed duplicate import of React, useState
const RolePopup = ({ onClose }) => {
  const [roleName, setRoleName] = useState('');
  const [redirectPath, setRedirectPath] = useState('');
  const [description, setDescription] = useState('');
  const [permissions, setPermissions] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [success, setSuccess] = useState('');

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError('');
    setSuccess('');
    try {
      const res = await fetch('/api/roles', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          role_name: roleName,
          redirect_path: redirectPath,
          description,
          permissions
        })
      });
      const data = await res.json();
      if (res.ok) {
        setSuccess('Role added successfully!');
        setRoleName('');
        setRedirectPath('');
        setDescription('');
        setPermissions('');
      } else {
        setError(data.error || 'Failed to add role');
      }
    } catch (err) {
      setError('Network error');
    }
    setLoading(false);
  };

  return (
    <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={onClose}>
      <div style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'340px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}} onClick={e=>e.stopPropagation()}>
        <h3>Add Role</h3>
        <form onSubmit={handleSubmit}>
          <div style={{marginBottom:'12px'}}>
            <label style={{fontWeight:'bold'}}>Role Name</label><br/>
            <input value={roleName} onChange={e=>setRoleName(e.target.value)} required style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} />
          </div>
          <div style={{marginBottom:'12px'}}>
            <label>Redirect Path</label><br/>
            <input value={redirectPath} onChange={e=>setRedirectPath(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} />
          </div>
          <div style={{marginBottom:'12px'}}>
            <label>Description</label><br/>
            <textarea value={description} onChange={e=>setDescription(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} />
          </div>
          <div style={{marginBottom:'12px'}}>
            <label>Permissions (comma separated)</label><br/>
            <input value={permissions} onChange={e=>setPermissions(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} />
          </div>
          {error && <div style={{color:'#f75c7a', marginBottom:'10px'}}>{error}</div>}
          {success && <div style={{color:'#27ae60', marginBottom:'10px'}}>{success}</div>}
          <button type="submit" disabled={loading} style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer', marginRight:'8px'}}>
            {loading ? 'Adding...' : 'Add Role'}
          </button>
          <button type="button" onClick={onClose} style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Close</button>
        </form>
      </div>
    </div>
  );
};

const TaskAccessOverlay = ({ task, onClose }) => {
 return (
   <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={onClose}>
     <div style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'340px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}} onClick={e=>e.stopPropagation()}>
       <h3>Task Access Details</h3>
       <p><strong>Task:</strong> {task.name}</p>
       <p><strong>Access:</strong> {Array.isArray(task.access) ? task.access.join(', ') : task.access}</p>
       <p><strong>Description:</strong> {task.description}</p>
       <button onClick={onClose} style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}}>Close</button>
     </div>
   </div>
 );
};

const TaskPopup = ({ onClose, onCreated, onUpdated, task }) => {
  const [name, setName] = useState(task?.name || '');
  const [access, setAccess] = useState(task?.access ? task.access.join(', ') : '');
  const [description, setDescription] = useState(task?.description || '');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const isEdit = !!task;

  const handleSubmit = e => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    const payload = {
      name,
      access: access.split(',').map(a => a.trim()),
      description
    };
    const url = isEdit ? `${BASE_API}/api/tasks/${task.id}` : `${BASE_API}/api/tasks`;
    const method = isEdit ? 'PUT' : 'POST';
    fetch(url, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload)
    })
      .then(res => {
        if (!res.ok) throw new Error(isEdit ? 'Failed to update task' : 'Failed to add task');
        return res.json();
      })
      .then(data => {
        setLoading(false);
        if (isEdit && onUpdated) onUpdated(data);
        if (!isEdit && onCreated) onCreated(data);
        onClose();
      })
      .catch(err => {
        setError(err.message);
        setLoading(false);
      });
  };

  return (
    <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={onClose}>
      <div style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'340px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}} onClick={e=>e.stopPropagation()}>
        <h3>{isEdit ? 'Edit Task' : 'Add Task'}</h3>
        <form onSubmit={handleSubmit} style={{display:'flex', flexDirection:'column', gap:'16px'}}>
          <input type="text" placeholder="Task Name" value={name} onChange={e=>setName(e.target.value)} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} required />
          <input type="text" placeholder="Who Can Access (comma separated)" value={access} onChange={e=>setAccess(e.target.value)} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} required />
          <textarea placeholder="Task Description" value={description} onChange={e=>setDescription(e.target.value)} style={{padding:'10px', borderRadius:'8px', border:'1px solid #ccc'}} />
          {error && <div style={{color:'#f75c7a'}}>{error}</div>}
          <div style={{display:'flex', gap:'12px', marginTop:'18px'}}>
            <button type="submit" style={{background:'#1565c0', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} disabled={loading}>{loading ? (isEdit ? 'Updating...' : 'Adding...') : (isEdit ? 'Update Task' : 'Add Task')}</button>
            <button type="button" style={{background:'#f75c7a', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={onClose} disabled={loading}>Cancel</button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default TasksManager;
