import React, { useState, useEffect } from 'react';
// import TinyMCE or your preferred rich text editor
// import user and video data from backend

const ManageTutorial = () => {

  // State for tutorials
  const [tutorials, setTutorials] = useState([]);
  // State for users
  const [users, setUsers] = useState([]);
  // State for videos
  const [videos, setVideos] = useState([]);

  // Fetch tutorials, users, and videos from API
  useEffect(() => {
    fetch('/api/tutorials')
      .then(res => res.json())
      .then(data => setTutorials(data.tutorials || []));
    fetch('/api/users')
      .then(res => res.json())
      .then(data => setUsers(data.map(u => ({ id: u.Id, name: u.name }))));
    fetch('/api/videos')
      .then(res => res.json())
      .then(data => setVideos((data.videos || []).map(v => ({ url: v.video_url, title: v.video_name }))));
  }, []);

  // Form state
  const [form, setForm] = useState({
    id: null,
    title: '',
    video_url: '',
    tutorial_insight: '',
    assigned_users: []
  });
  const [editing, setEditing] = useState(false);
  const [showModal, setShowModal] = useState(false);

  // TinyMCE integration with API key, re-initialize on modal open
  useEffect(() => {
    // Load TinyMCE script if not loaded
    if (!window.tinymce) {
      const script = document.createElement('script');
      script.src = 'https://cdn.tiny.cloud/1/2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65/tinymce/6/tinymce.min.js';
      script.referrerPolicy = 'origin';
      script.onload = () => {
        // No-op, will init below
      };
      document.body.appendChild(script);
    }
  }, []);

  useEffect(() => {
    if (showModal) {
      // Remove previous instance if exists
      if (window.tinymce && window.tinymce.get('tutorialInsight')) {
        window.tinymce.get('tutorialInsight').remove();
      }
      if (window.tinymce) {
        window.tinymce.init({
          selector: '#tutorialInsight',
          height: 180,
          menubar: false,
          setup: (editor) => {
            editor.on('Change KeyUp', () => {
              setForm(f => ({ ...f, tutorial_insight: editor.getContent() }));
            });
          }
        });
        if (form.tutorial_insight) {
          setTimeout(() => {
            if (window.tinymce && window.tinymce.get('tutorialInsight')) {
              window.tinymce.get('tutorialInsight').setContent(form.tutorial_insight);
            }
          }, 200);
        }
      }
    }
  }, [showModal]);

  // Handle form input
  const handleChange = e => {
    const { name, value } = e.target;
    setForm(f => ({ ...f, [name]: value }));
  };
  const handleUserChange = e => {
    const options = Array.from(e.target.selectedOptions).map(o => Number(o.value));
    setForm(f => ({ ...f, assigned_users: options }));
  };

  // Handle submit
  const handleSubmit = async e => {
    e.preventDefault();
    const payload = {
      title: form.title,
      video_url: form.video_url,
      tutorial_insight: form.tutorial_insight,
      assigned_users: form.assigned_users.map(Number)
    };
    if (editing && form.id) {
      await fetch(`/api/tutorials/${form.id}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });
    } else {
      await fetch('/api/tutorials', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });
    }
    // Refresh tutorials
    fetch('/api/tutorials')
      .then(res => res.json())
      .then(data => setTutorials(data.tutorials || []));
    setForm({ id: null, title: '', video_url: '', tutorial_insight: '', assigned_users: [] });
    setEditing(false);
    setShowModal(false);
    if (window.tinymce && window.tinymce.get('tutorialInsight')) window.tinymce.get('tutorialInsight').setContent('');
  };

  // Edit tutorial
  const handleEdit = tut => {
    setForm({
      id: tut.id,
      title: tut.title,
      video_url: tut.video_url,
      tutorial_insight: tut.tutorial_insight,
      assigned_users: (tut.assigned_users || []).map(u => u.id)
    });
    setEditing(true);
    setShowModal(true);
    setTimeout(() => {
      if (window.tinymce && window.tinymce.get('tutorialInsight')) window.tinymce.get('tutorialInsight').setContent(tut.tutorial_insight || '');
    }, 100);
  };

  // Delete tutorial
  const handleDelete = async (id) => {
    if (!window.confirm('Are you sure you want to delete this tutorial?')) return;
    await fetch(`/api/tutorials/${id}`, {
      method: 'DELETE'
    });
    // Refresh tutorials
    fetch('/api/tutorials')
      .then(res => res.json())
      .then(data => setTutorials(data.tutorials || []));
  };

  return (
    <div>
      <h2>Manage Tutorials</h2>
      <button onClick={() => { setShowModal(true); setEditing(false); setForm({ id: null, title: '', video_url: '', tutorial_insight: '', assigned_users: [] }); if (window.tinymce && window.tinymce.get('tutorialInsight')) window.tinymce.get('tutorialInsight').setContent(''); }} style={{ marginBottom: '18px', padding: '8px 24px', borderRadius: '8px', border: 'none', background: '#1976d2', color: '#fff', fontWeight: 'bold', cursor: 'pointer' }}>Add Tutorial</button>
      <table style={{ width: '100%', borderCollapse: 'collapse', marginBottom: '32px' }}>
        <thead>
          <tr style={{ background: '#f7f8fa' }}>
            <th style={{ padding: '8px', border: '1px solid #e0e0e0' }}>Title</th>
            <th style={{ padding: '8px', border: '1px solid #e0e0e0' }}>Video</th>
            <th style={{ padding: '8px', border: '1px solid #e0e0e0' }}>Insight</th>
            <th style={{ padding: '8px', border: '1px solid #e0e0e0' }}>Assigned Users</th>
            <th style={{ padding: '8px', border: '1px solid #e0e0e0' }}>Actions</th>
          </tr>
        </thead>
        <tbody>
          {tutorials.map(tut => (
            <tr key={tut.id}>
              <td style={{ padding: '8px', border: '1px solid #e0e0e0' }}>{tut.title}</td>
              <td style={{ padding: '8px', border: '1px solid #e0e0e0' }}>
                {tut.video_url && tut.video_url.startsWith('/videos/') ? (
                  <video width="180" height="100" controls>
                    <source src={tut.video_url} type="video/mp4" />
                    Your browser does not support the video tag.
                  </video>
                ) : (
                  <iframe width="180" height="100" src={tut.video_url} title={tut.title} frameBorder="0" allowFullScreen></iframe>
                )}
              </td>
              <td style={{ padding: '8px', border: '1px solid #e0e0e0' }}>
                <div dangerouslySetInnerHTML={{ __html: tut.tutorial_insight }} />
              </td>
              <td style={{ padding: '8px', border: '1px solid #e0e0e0' }}>
                {(tut.assigned_users || []).map(u => (
                  <span key={u.id} style={{ marginRight: 8, background: '#e3f2fd', padding: '4px 10px', borderRadius: '8px' }}>{u.name}</span>
                ))}
              </td>
              <td style={{ padding: '8px', border: '1px solid #e0e0e0' }}>
                <button onClick={() => handleEdit(tut)} style={{ marginRight: 8, padding: '4px 12px', borderRadius: '6px', border: 'none', background: '#1976d2', color: '#fff', cursor: 'pointer' }}>Edit</button>
                <button onClick={() => handleDelete(tut.id)} style={{ padding: '4px 12px', borderRadius: '6px', border: 'none', background: '#e53935', color: '#fff', cursor: 'pointer' }}>Delete</button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>

      {/* Modal for Add/Edit Tutorial */}
      {showModal && (
        <div style={{ position: 'fixed', top: 0, left: 0, width: '100vw', height: '100vh', background: 'rgba(0,0,0,0.18)', zIndex: 9999, display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
          <div style={{ background: '#fff', borderRadius: '16px', boxShadow: '0 2px 24px rgba(0,0,0,0.18)', minWidth: '420px', maxWidth: '90vw', maxHeight: '90vh', overflowY: 'auto', padding: '32px', position: 'relative' }}>
            <h3>{editing ? 'Edit Tutorial' : 'Add New Tutorial'}</h3>
            <form onSubmit={handleSubmit} style={{ marginBottom: '12px' }}>
              <div style={{ marginBottom: '12px' }}>
                <label>Title:<br />
                  <input type="text" name="title" value={form.title} onChange={handleChange} required style={{ width: '100%', padding: '8px', borderRadius: '6px', border: '1px solid #e0e0e0' }} />
                </label>
              </div>
              <div style={{ marginBottom: '12px' }}>
                <label>Video:<br />
                  <select name="video_url" value={form.video_url} onChange={handleChange} required style={{ width: '100%', padding: '8px', borderRadius: '6px', border: '1px solid #e0e0e0' }}>
                    <option value="">Select a video</option>
                    {videos.map(v => (
                      <option key={v.url} value={v.url}>{v.title}</option>
                    ))}
                  </select>
                </label>
              </div>
              <div style={{ marginBottom: '12px' }}>
                <label>Insight:<br />
                  <textarea id="tutorialInsight" name="tutorial_insight" defaultValue={form.tutorial_insight} />
                </label>
              </div>
              <div style={{ marginBottom: '12px' }}>
                <label>Assign Users:<br />
                  <select multiple name="assigned_users" value={form.assigned_users.map(String)} onChange={handleUserChange} style={{ width: '100%', padding: '8px', borderRadius: '6px', border: '1px solid #e0e0e0', minHeight: '60px' }}>
                    {users.map(u => (
                      <option key={u.id} value={u.id}>{u.name}</option>
                    ))}
                  </select>
                </label>
              </div>
              <button type="submit" style={{ padding: '8px 24px', borderRadius: '8px', border: 'none', background: '#1976d2', color: '#fff', fontWeight: 'bold', cursor: 'pointer' }}>{editing ? 'Update Tutorial' : 'Add Tutorial'}</button>
              <button type="button" onClick={() => { setShowModal(false); setEditing(false); setForm({ id: null, title: '', video_url: '', tutorial_insight: '', assigned_users: [] }); if (window.tinymce && window.tinymce.get('tutorialInsight')) window.tinymce.get('tutorialInsight').setContent(''); }} style={{ marginLeft: 12, padding: '8px 24px', borderRadius: '8px', border: 'none', background: '#aaa', color: '#fff', fontWeight: 'bold', cursor: 'pointer' }}>Cancel</button>
            </form>
            <button style={{ position: 'absolute', top: 12, right: 18, background: 'none', border: 'none', fontSize: '1.5rem', cursor: 'pointer', color: '#888' }} onClick={() => { setShowModal(false); setEditing(false); setForm({ id: null, title: '', video_url: '', tutorial_insight: '', assigned_users: [] }); if (window.tinymce && window.tinymce.get('tutorialInsight')) window.tinymce.get('tutorialInsight').setContent(''); }}>&times;</button>
          </div>
        </div>
      )}
    </div>
  );
};

export default ManageTutorial;
