import React, { useState, useEffect } from 'react';
import { BASE_API } from '../../utils/config';

const MEDIA_TABS = [
  { key: 'audios', label: 'Audios' },
  { key: 'videos', label: 'Videos' },
  { key: 'images', label: 'Images' },
  { key: 'documents', label: 'Documents' }
];

const ManageMedia = () => {
  const [activeTab, setActiveTab] = useState('audios');
  const [showModal, setShowModal] = useState(false);
  const [mediaName, setMediaName] = useState('');
  const [mediaFile, setMediaFile] = useState(null);
  const [uploading, setUploading] = useState(false);

  const [mediaList, setMediaList] = useState([]);
  const API_MAP = {
    audios: `${BASE_API}/api/audios`,
    videos: `${BASE_API}/api/videos`,
    images: `${BASE_API}/api/images`,
    documents: `${BASE_API}/api/documents_url`
  };

  useEffect(() => {
    fetchMediaList(activeTab);
    // eslint-disable-next-line
  }, [activeTab]);

  const fetchMediaList = async (tab) => {
    try {
      const res = await fetch(API_MAP[tab]);
      const data = await res.json();
      if (tab === 'documents') {
        setMediaList(data.documents || []);
      } else {
        setMediaList(data[tab] || []);
      }
    } catch (err) {
      setMediaList([]);
    }
  };

  const handleAddMedia = async (e) => {
    e.preventDefault();
    if (!mediaFile) return;
    setUploading(true);
    const formData = new FormData();
    formData.append('name', mediaName);
    formData.append('file', mediaFile);
    formData.append('type', activeTab);
    // Upload to backend
    try {
      let res;
      if (activeTab === 'documents') {
        // For documents, use /api/documents_url/upload endpoint
        res = await fetch(`${BASE_API}/api/documents_url/upload`, {
          method: 'POST',
          body: formData
        });
      } else {
        res = await fetch(`${BASE_API}/api/${activeTab}/upload`, {
          method: 'POST',
          body: formData
        });
      }
      if (res.ok) {
        fetchMediaList(activeTab);
        setShowModal(false);
        setMediaName('');
        setMediaFile(null);
      }
    } catch (err) {}
    setUploading(false);
  };

  return (
    <div style={{maxWidth:'900px', margin:'32px auto', background:'#fff', borderRadius:'16px', boxShadow:'0 2px 12px rgba(0,0,0,0.08)', padding:'32px'}}>
      <h2>Manage Media</h2>
      <div style={{display:'flex', gap:'16px', marginBottom:'24px'}}>
        {MEDIA_TABS.map(tab => (
          <button
            key={tab.key}
            onClick={() => setActiveTab(tab.key)}
            style={{
              background: activeTab === tab.key ? '#1976d2' : '#f1f3f6',
              color: activeTab === tab.key ? '#fff' : '#222',
              border: 'none',
              borderRadius: '6px',
              padding: '10px 24px',
              fontWeight: 'bold',
              fontSize: '1rem',
              cursor: 'pointer'
            }}
          >
            {tab.label}
          </button>
        ))}
        <button
          style={{marginLeft:'auto', background:'#27ae60', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}
          onClick={() => setShowModal(true)}
        >
          + Add Media
        </button>
      </div>
      <div style={{marginBottom:'24px'}}>
        <h3>{MEDIA_TABS.find(tab => tab.key === activeTab).label} List</h3>
        {mediaList.length === 0 ? (
          <div style={{color:'#888'}}>No media found.</div>
        ) : (
          <table style={{width:'100%', borderCollapse:'collapse', marginTop:'12px'}}>
            <thead>
              <tr style={{background:'#f7f7f7'}}>
                <th>Name</th>
                <th>Type</th>
                <th>Description</th>
                <th>Created By</th>
                <th>Created At</th>
                <th>Actions</th>
              </tr>
            </thead>
            <tbody>
              {mediaList.map(item => (
                <tr key={item.id}>
                  <td>{item.audio_name || item.video_name || item.image_name || item.document_name || item.name}</td>
                  <td>{item.type || (activeTab === 'documents' ? 'document' : '')}</td>
                  <td>{item.description || ''}</td>
                  <td>{item.create_by || ''}</td>
                  <td>{item.created_at || ''}</td>
                  <td>
                    {item.audio_url && (
                      <a href={item.audio_url} target="_blank" rel="noopener noreferrer" style={{marginRight:'8px'}}>View</a>
                    )}
                    {item.video_url && (
                      <a href={item.video_url} target="_blank" rel="noopener noreferrer" style={{marginRight:'8px'}}>View</a>
                    )}
                    {item.image_url && (
                      <a href={item.image_url} target="_blank" rel="noopener noreferrer" style={{marginRight:'8px'}}>View</a>
                    )}
                    {item.document_url && (
                      <a href={item.document_url} target="_blank" rel="noopener noreferrer" style={{marginRight:'8px'}}>View</a>
                    )}
                    <button style={{background:'#e74c3c', color:'#fff', border:'none', borderRadius:'4px', padding:'6px 12px', cursor:'pointer'}} onClick={async () => {
                      if (activeTab === 'documents') {
                        // Document delete logic (implement as needed)
                        alert('Delete document not implemented');
                      } else {
                        await fetch(`${BASE_API}/api/${activeTab}/delete/${item.id}`, { method: 'DELETE' });
                        fetchMediaList(activeTab);
                      }
                    }}>Delete</button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        )}
      </div>
      {showModal && (
        <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.2)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}}>
          <form onSubmit={handleAddMedia} style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'320px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}}>
            <h3>Add Media</h3>
            <label>Media Name:</label>
            <input name="mediaName" value={mediaName} onChange={e => setMediaName(e.target.value)} required style={{width:'100%', marginBottom:'10px'}} />
            <label>File:</label>
            <input type="file" accept={activeTab === 'images' ? 'image/*' : activeTab === 'videos' ? 'video/*' : 'audio/*'} onChange={e => setMediaFile(e.target.files[0])} required style={{width:'100%', marginBottom:'10px'}} />
            <button type="submit" style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} disabled={uploading}>{uploading ? 'Uploading...' : 'Add'}</button>
            <button type="button" onClick={() => setShowModal(false)} style={{marginLeft:'12px', background:'#ccc', color:'#222', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Cancel</button>
          </form>
        </div>
      )}
    </div>
  );
};

export default ManageMedia;
