import React, { useEffect, useState } from 'react';
import Modal from 'react-modal';
import { Editor } from '@tinymce/tinymce-react';
import { BASE_API } from '../../utils/config';

const defaultForm = { title: '', content: '', document_url: '' };

const ManageInsight = () => {
  const [imageOptions, setImageOptions] = useState([]);
  const [insights, setInsights] = useState([]);
  const [form, setForm] = useState(defaultForm);
  const [editingId, setEditingId] = useState(null);
  const [loading, setLoading] = useState(false);
  const [showForm, setShowForm] = useState(false);
  useEffect(() => { Modal.setAppElement('body'); }, []);
  const [documentOptions, setDocumentOptions] = useState([]);
  const [users, setUsers] = useState([]);
  const [assignedUsers, setAssignedUsers] = useState([]);

  const fetchInsights = () => {
    setLoading(true);
    fetch(`${BASE_API.replace('5000', '3000')}/api/insights`)
      .then(res => res.json())
      .then(data => { setInsights(data.insights || []); setLoading(false); });
  };

  const fetchDocumentOptions = () => {
    fetch(`${BASE_API}/api/documents_url`)
      .then(res => res.json())
      .then(data => {
        setDocumentOptions(data.documents || []);
      });
  };

  useEffect(() => {
    fetchInsights();
    fetchDocumentOptions();
    fetch(`${BASE_API}/api/users`)
      .then(res => res.json())
      .then(data => setUsers(data));
    fetch(`${BASE_API}/api/images`)
      .then(res => res.json())
      .then(data => setImageOptions(data.images || []));
  }, []);

  const handleChange = e => {
    const { name, value } = e.target;
    // If filling document_url, clear content
    if (name === 'document_url' && value) {
      setForm({ ...form, document_url: value, content: '' });
    } else if (name === 'content' && value) {
      setForm({ ...form, content: value, document_url: '' });
    } else {
      setForm({ ...form, [name]: value });
    }
  };

  const handleEditorChange = content => {
  setForm({ ...form, content, document_url: '' });
  };

  const handleSubmit = e => {
    e.preventDefault();
    setLoading(true);
    const method = editingId ? 'PUT' : 'POST';
    const url = editingId ? `/api/insights/${editingId}` : '/api/insights';
    fetch(`${BASE_API}${url.startsWith('/') ? url : '/' + url}`, {
      method,
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(form)
    })
      .then(res => res.json())
      .then((result) => {
        if (result.error) {
          alert(result.error);
          setLoading(false);
          return;
        }
        setForm(defaultForm);
        setEditingId(null);
        setShowForm(false);
        fetchInsights();
      });
  };

  const handleEdit = insight => {
  setForm({ title: insight.title, content: insight.content, document_url: insight.document_url || '', image_url: insight.image_url || '' });
    setEditingId(insight.id);
    // Fetch assigned users for this insight
    fetch(`${BASE_API}/api/insight_assign/${insight.id}`)
      .then(res => res.json())
      .then(data => setAssignedUsers(data.assigned_users || []));
    setShowForm(true);
  };

  const handleDelete = id => {
    if (!window.confirm('Delete this insight?')) return;
    setLoading(true);
    fetch(`${BASE_API}/api/insights/${id}`, { method: 'DELETE' })
      .then(res => res.json())
      .then(() => fetchInsights());
  };

  return (
    <div style={{ padding: '32px', maxWidth: '900px', margin: '0 auto' }}>
      <h2 style={{ color: '#1976d2', marginBottom: '24px' }}>Manage Insights</h2>
      {!showForm && (
        <button style={{ background: '#1565c0', color: '#fff', border: 'none', borderRadius: '8px', padding: '10px 24px', fontWeight: 'bold', cursor: 'pointer', marginBottom: '24px' }} onClick={() => { setShowForm(true); setForm(defaultForm); setEditingId(null); setAssignedUsers([]); }}>Add Insight</button>
      )}
      <Modal isOpen={showForm} onRequestClose={() => { setForm(defaultForm); setEditingId(null); setShowForm(false); setAssignedUsers([]); }} style={{ content: { maxWidth: '600px', margin: 'auto', borderRadius: '12px', padding: '24px' } }}>
        <div style={{position:'relative', minHeight:'100vh'}}>
          <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(25,118,210,0.08)', zIndex:9998}}></div>
          <form onSubmit={async e => {
            e.preventDefault();
            setLoading(true);
            const method = editingId ? 'PUT' : 'POST';
            const url = editingId ? `/api/insights/${editingId}` : '/api/insights';
            const payload = { ...form };
            // Ensure image_url is included
            if (!payload.image_url) payload.image_url = '';
            const res = await fetch(url, {
              method,
              headers: { 'Content-Type': 'application/json' },
              body: JSON.stringify(payload)
            });
            const result = await res.json();
            if (result.error) {
              alert(result.error);
              setLoading(false);
              return;
            }
            // Assign users after insight is created/edited
            const insightId = editingId || result.id;
            if (assignedUsers.length > 0) {
              await fetch('/api/insight_assign', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ insight_id: insightId, user_ids: assignedUsers.map(u => u.Id) })
              });
            }
            setForm(defaultForm);
            setEditingId(null);
            setShowForm(false);
            setAssignedUsers([]);
            fetchInsights();
          }} style={{ background: '#fff', borderRadius: '22px', boxShadow: '0 12px 48px rgba(25,118,210,0.18)', padding: '40px 36px', maxWidth: '540px', margin: '60px auto', display: 'flex', flexDirection: 'column', gap: '22px', zIndex:9999, position:'relative' }}>
            <h1 style={{ color: '#1565c0', marginBottom: '0', fontWeight: 'bold', textAlign: 'center', letterSpacing: '0.02em', fontSize:'2rem' }}>{editingId ? 'Edit Insight' : 'Add Insight'}</h1>
            <div style={{borderBottom:'1.5px solid #e3f2fd', margin:'0 0 18px 0'}}></div>
            <div>
              <label style={{ fontWeight: 'bold', color: '#1976d2', marginBottom: '8px', display: 'block', fontSize:'1.08rem' }}>Cover Image</label>
              <select
                name="image_url"
                value={form.image_url || ''}
                onChange={handleChange}
                style={{ width: '100%', padding: '12px', borderRadius: '10px', border: '1.5px solid #e0e0e0', marginTop: '4px', fontSize: '1.08rem', background: '#f7f8fa', transition:'border 0.2s' }}
              >
                <option value="">Select a cover image...</option>
                {imageOptions.map(img => (
                  <option key={img.id} value={img.image_url}>
                    {img.image_name || img.image_url}
                  </option>
                ))}
              </select>
            </div>
            <div>
              <label style={{ fontWeight: 'bold', color: '#1976d2', marginBottom: '8px', display: 'block', fontSize:'1.08rem' }}>Title</label>
              <input name="title" value={form.title} onChange={handleChange} required style={{ width: '100%', padding: '12px', borderRadius: '10px', border: '1.5px solid #e0e0e0', marginTop: '4px', fontSize: '1.08rem', background: '#f7f8fa', transition:'border 0.2s' }} />
            </div>
            <div>
              <label style={{ fontWeight: 'bold', color: '#1976d2', marginBottom: '8px', display: 'block', fontSize:'1.08rem' }}>Document URL</label>
              <select
                name="document_url"
                value={form.document_url}
                onChange={handleChange}
                disabled={!!form.content}
                style={{ width: '100%', padding: '12px', borderRadius: '10px', border: '1.5px solid #e0e0e0', marginTop: '4px', fontSize: '1.08rem', background: form.content ? '#f0f0f0' : '#f7f8fa', transition:'border 0.2s' }}
              >
                <option value="">Select a document...</option>
                {documentOptions.map(doc => (
                  <option key={doc.id} value={doc.document_url}>{doc.document_name || doc.document_url}</option>
                ))}
              </select>
            </div>
            <div>
              <label style={{ fontWeight: 'bold', color: '#1976d2', marginBottom: '8px', display: 'block', fontSize:'1.08rem' }}>Content</label>
              <Editor
                apiKey="2ujb8js1jflsddb96gzk4pzh89gxa8v4hn3f1tze484trd65"
                value={form.content}
                init={{ height: 220, menubar: false, plugins: 'link lists code', toolbar: 'undo redo | bold italic | alignleft aligncenter alignright | bullist numlist | link code' }}
                onEditorChange={handleEditorChange}
                disabled={!!form.document_url}
              />
            </div>
            <div>
              <label style={{ fontWeight: 'bold', color: '#1976d2', marginBottom: '8px', display: 'block', fontSize:'1.08rem' }}>Assign Users</label>
              <select
                multiple
                value={assignedUsers.map(u => u.Id)}
                onChange={e => {
                  const selectedIds = Array.from(e.target.selectedOptions, opt => Number(opt.value));
                  setAssignedUsers(users.filter(u => selectedIds.includes(u.Id)));
                }}
                style={{ width: '100%', padding: '12px', borderRadius: '10px', border: '1.5px solid #e0e0e0', marginTop: '4px', minHeight: '80px', fontSize: '1.08rem', background: '#f7f8fa', transition:'border 0.2s' }}
              >
                {users.map(u => (
                  <option key={u.Id} value={u.Id}>{u.name} ({u.email})</option>
                ))}
              </select>
            </div>
            <div style={{borderTop:'1.5px solid #e3f2fd', margin:'18px 0 0 0'}}></div>
            <div style={{ display: 'flex', justifyContent: 'center', gap: '18px', marginTop: '10px' }}>
              <button type="submit" style={{ background: '#1976d2', color: '#fff', border: 'none', borderRadius: '10px', padding: '14px 38px', fontWeight: 'bold', fontSize: '1.12rem', cursor: 'pointer', boxShadow: '0 2px 8px rgba(25,118,210,0.10)', transition:'background 0.2s' }} onMouseOver={e=>e.target.style.background='#1565c0'} onMouseOut={e=>e.target.style.background='#1976d2'}>{editingId ? 'Update Insight' : 'Add Insight'}</button>
              <button type="button" style={{ background: '#888', color: '#fff', border: 'none', borderRadius: '10px', padding: '14px 38px', fontWeight: 'bold', fontSize: '1.12rem', cursor: 'pointer', boxShadow: '0 2px 8px rgba(0,0,0,0.10)', transition:'background 0.2s' }} onClick={() => { setForm(defaultForm); setEditingId(null); setShowForm(false); setAssignedUsers([]); }} onMouseOver={e=>e.target.style.background='#555'} onMouseOut={e=>e.target.style.background='#888'}>Cancel</button>
            </div>
          </form>
        </div>
      </Modal>
      )}
      <h3 style={{ color: '#1565c0', marginBottom: '16px' }}>All Insights</h3>
      {loading ? <div>Loading...</div> : (
        <table style={{ width: '100%', borderCollapse: 'collapse', background: '#fff', borderRadius: '12px', boxShadow: '0 2px 8px rgba(0,0,0,0.04)' }}>
          <thead>
            <tr style={{ background: '#f7f8fa' }}>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Cover Image</th>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Title</th>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Document URL</th>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Content</th>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Assigned Users</th>
              <th style={{ padding: '10px', border: '1px solid #e0e0e0' }}>Actions</th>
            </tr>
          </thead>
          <tbody>
            {insights.map(insight => (
              <tr key={insight.id}>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0', textAlign: 'center' }}>
                  {insight.image_url ? (
                    <img src={insight.image_url} alt="cover" style={{ width: '60px', height: '60px', objectFit: 'cover', borderRadius: '8px', boxShadow: '0 2px 8px rgba(25,118,210,0.08)' }} />
                  ) : <span style={{ color: '#888' }}>No image</span>}
                </td>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0', fontWeight: 'bold', color: '#1976d2' }}>{insight.title}</td>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0' }}>{insight.document_url ? <a href={insight.document_url} target="_blank" rel="noopener noreferrer">{insight.document_url}</a> : '-'}</td>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0', maxWidth: '260px', whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' }}>
                  {insight.content
                    ? (
                        <span>
                          {(() => {
                            // Strip HTML tags and limit to 120 chars
                            const tmp = document.createElement('div');
                            tmp.innerHTML = insight.content;
                            const text = tmp.textContent || tmp.innerText || '';
                            return text.length > 120 ? text.slice(0, 120) + '...' : text;
                          })()}
                        </span>
                      )
                    : '-'}
                </td>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0' }}>
                  <AssignedUsersCell insightId={insight.id} />
                </td>
                <td style={{ padding: '10px', border: '1px solid #e0e0e0' }}>
                  <button style={{ background: '#1565c0', color: '#fff', border: 'none', borderRadius: '6px', padding: '6px 16px', fontWeight: 'bold', cursor: 'pointer', marginRight: '8px' }} onClick={() => handleEdit(insight)}>Edit</button>
                  <button style={{ background: '#e53935', color: '#fff', border: 'none', borderRadius: '6px', padding: '6px 16px', fontWeight: 'bold', cursor: 'pointer' }} onClick={() => handleDelete(insight.id)}>Delete</button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      )}
    </div>
  );

  // Assigned users cell component
  function AssignedUsersCell({ insightId }) {
    const [assigned, setAssigned] = useState([]);
    useEffect(() => {
      fetch(`${BASE_API}/api/insight_assign/${insightId}`)
        .then(res => res.json())
        .then(data => setAssigned(data.assigned_users || []));
    }, [insightId]);
    return (
      <div>
        {assigned.length === 0 ? <span>-</span> : assigned.map(u => <span key={u.id}>{u.name} ({u.email})<br /></span>)}
      </div>
    );
  }
};

export default ManageInsight;
