import React, { useState, useEffect } from 'react';
import { Tab, Tabs, Card, Button, Form, Table, Modal, Alert, Row, Col } from 'react-bootstrap';
import DataTable from 'react-data-table-component';
import QRCode from 'qrcode.react';
import axios from 'axios';
import LegalCertificateModal from './LegalCertificateModal';

const CERT_TYPES = [
  { key: 'medical', label: 'Medical Interpreter', color: 'success', description: '60 Hours Medical Interpreter Certification', validity: 'Valid for 2 years' },
  { key: 'hipaa', label: 'HIPAA Compliance', color: 'primary', description: 'Health Insurance Portability and Accountability Act', validity: 'Valid for 1 year' },
  { key: 'legal', label: 'Legal Interpreter', color: 'warning', description: '40 Hours Legal Interpreter Program', validity: 'No expiration date' },
];

function ManageCertificate() {
  // State
  const [tab, setTab] = useState('add');
  const [users, setUsers] = useState([]);
  const [certificates, setCertificates] = useState([]);
  const [languages, setLanguages] = useState([]);
  const [fields, setFields] = useState({});
  const [form, setForm] = useState({});
  const [selectedCertTypes, setSelectedCertTypes] = useState([]);
  const [showAlert, setShowAlert] = useState(null);
  const [showModal, setShowModal] = useState(false);
  const [modalContent, setModalContent] = useState(null);
  const [previewCerfId, setPreviewCerfId] = useState(null);
  const [previewHtml, setPreviewHtml] = useState(null);

  // Effects: fetch users, certificates, languages, fields
  useEffect(() => {
    // Fetch users from backend
    axios.get('/api/users').then(res => {
      setUsers(res.data);
      console.log('Fetched users:', res.data); // Debug log
    });
    // Fetch languages from public countries API
    fetch('https://restcountries.com/v3.1/all?fields=languages')
      .then(res => res.json())
      .then(data => {
        // Extract unique language names
        const langSet = new Set();
        data.forEach(country => {
          if (country.languages) {
            Object.values(country.languages).forEach(lang => langSet.add(lang));
          }
        });
        const langArr = Array.from(langSet).map((lang, idx) => ({ id: idx, langue_name: lang }));
        setLanguages(langArr);
        console.log('Fetched languages:', langArr); // Debug log
      });
    // Initial fetch of certificates
    axios.get('/api/certificates').then(res => {
      setCertificates(res.data);
    });
    // TODO: Fetch fields from backend as needed
  }, []);

  // Fetch certificates when switching to List tab
  useEffect(() => {
    if (tab === 'list') {
      axios.get('/api/certificates').then(res => {
        setCertificates(res.data);
      });
    }
  }, [tab]);

  // Handlers
  const handleTabSelect = (key) => setTab(key);
  const handleCertTypeChange = (type) => {
    setSelectedCertTypes((prev) =>
      prev.includes(type) ? prev.filter((t) => t !== type) : [...prev, type]
    );
  };
  const handleFormChange = (e) => {
    const { name, value, type: inputType, checked } = e.target;
    setForm((prev) => ({ ...prev, [name]: inputType === 'checkbox' ? checked : value }));
  };
  const handleAddCertificate = async (e) => {
    e.preventDefault();
    // Always send cert_types as array, even for single selection
    const certTypesToSend = selectedCertTypes.length > 0 ? selectedCertTypes : (form.certificate_type ? [form.certificate_type] : []);
    if (certTypesToSend.length === 0) {
      setShowAlert('Please select at least one certificate type.');
      return;
    }
    try {
      const cerf_id = form.cerf_id || Math.floor(Math.random()*1e12);
      const payload = {
        ...form,
        cerf_id,
        cert_types: certTypesToSend,
        ...Object.fromEntries(Array.from({ length: 11 }, (_, i) => [`field_${i+1}`, !!form[`v${i+1}`]]) )
      };
      // Remove HTML generation and extra POST
      await axios.post('/api/certificates', payload);
      setShowAlert('Certificate(s) generated and saved successfully!');
      setForm({});
      setSelectedCertTypes([]);
    } catch (err) {
      setShowAlert('Error saving certificate: ' + (err.response?.data?.error || err.message));
    }
  };
  // ...other handlers for edit, delete, field update

  // UI
  return (
    <div className="container-fluid p-4">
      <h2 className="mb-4">Manage Certificates</h2>
      {showAlert && <Alert variant="info" onClose={() => setShowAlert(null)} dismissible>{showAlert}</Alert>}
      <Tabs activeKey={tab} onSelect={handleTabSelect} className="mb-3">
        <Tab eventKey="add" title="Add">
          <Card className="mb-3">
            <Card.Body>
              {/* Certificate Types to Generate */}
              <div style={{ marginBottom: '32px' }}>
                <h5 style={{ color: '#0057ff' }}>Certificate Type to Generate</h5>
                <div style={{ marginBottom: '8px' }}>Select one certificate type to generate (required)</div>
                <Form.Group>
                  <Form.Label>Certificate Type</Form.Label>
                  <Form.Control
                    as="select"
                    name="certificate_type"
                    value={form.certificate_type || ''}
                    onChange={e => setForm(prev => ({ ...prev, certificate_type: e.target.value }))}
                    required
                  >
                    <option value="">Choose type...</option>
                    <option value="medical">Medical Interpreter</option>
                    <option value="hipaa">HIPAA Compliance</option>
                    <option value="legal">Legal Interpreter</option>
                  </Form.Control>
                </Form.Group>
              </div>
              <Form onSubmit={handleAddCertificate}>
                <Row>
                  <Col md={4}>
                    <Form.Group className="mb-3">
                      <Form.Label>Use Profile Picture</Form.Label>
                      <Form.Check
                        type="switch"
                        name="use_profile_picture"
                        checked={!!form.use_profile_picture}
                        onChange={handleFormChange}
                        label="Use profile picture"
                      />
                    </Form.Group>
                  </Col>
                  <Col md={4}>
                    <Form.Group className="mb-3">
                      <Form.Label>Full Name (ID)</Form.Label>
                      <Form.Control as="select" name="fullname" value={form.fullname || ''} onChange={handleFormChange} required>
                        <option value="">Choose...</option>
                        {users.map((u) => (
                          <option key={u.Id} value={u.Id}>
                            {u.nom && u.prenom ? `${u.nom} ${u.prenom}` : (u.name && u.surname ? `${u.name} ${u.surname}` : (u.fullname || u.Id))}
                          </option>
                        ))}
                      </Form.Control>
                    </Form.Group>
                  </Col>
                  <Col md={3}>
                    <Form.Group className="mb-3">
                      <Form.Label>Certificate Number</Form.Label>
                      <Form.Control type="number" name="cerf_id" value={form.cerf_id || Math.floor(Math.random()*1e12)} readOnly required />
                    </Form.Group>
                  </Col>
                </Row>
                {/* Validation fields v1-v11 */}
                <div style={{ display: 'flex', alignItems: 'center', marginBottom: '8px' }}>
                  <Button variant="outline-primary" size="sm" onClick={() => {
                    const newFields = {};
                    for (let i = 1; i <= 11; i++) newFields[`v${i}`] = true;
                    setForm(prev => ({ ...prev, ...newFields }));
                  }}>Activate All Fields</Button>
                  <Button variant="outline-secondary" size="sm" style={{ marginLeft: '8px' }} onClick={() => {
                    const newFields = {};
                    for (let i = 1; i <= 11; i++) newFields[`v${i}`] = false;
                    setForm(prev => ({ ...prev, ...newFields }));
                  }}>Deactivate All Fields</Button>
                </div>
                <Row>
                  {Array.from({ length: 11 }, (_, i) => (
                    <Col md={4} key={i} className="mb-3">
                      <Form.Group>
                        <Form.Label>{fields[`v${i+1}_text`] || `Field ${i+1}`}</Form.Label>
                        <Form.Check type="switch" name={`v${i+1}`} checked={!!form[`v${i+1}`]} onChange={handleFormChange} label="Yes" />
                      </Form.Group>
                    </Col>
                  ))}
                </Row>
                <Row>
                  <Col md={5}>
                    <Form.Group className="mb-3">
                      <Form.Label>Language From</Form.Label>
                      <Form.Control as="select" name="language_from" value={form.language_from || ''} onChange={handleFormChange}>
                        <option value="">Select first language</option>
                        {languages.map((l) => (
                          <option key={l.id} value={l.langue_name} disabled={form.language_to === l.langue_name}>{l.langue_name}</option>
                        ))}
                      </Form.Control>
                    </Form.Group>
                  </Col>
                  <Col md={5}>
                    <Form.Group className="mb-3">
                      <Form.Label>Language To</Form.Label>
                      <Form.Control as="select" name="language_to" value={form.language_to || ''} onChange={handleFormChange}>
                        <option value="">Select second language</option>
                        {languages.map((l) => (
                          <option key={l.id} value={l.langue_name} disabled={form.language_from === l.langue_name}>{l.langue_name}</option>
                        ))}
                      </Form.Control>
                    </Form.Group>
                  </Col>
                </Row>
                <Row className="mb-3">
                  <Col md={4}>
                    <Form.Group>
                      <Form.Label>Date validity</Form.Label>
                      <Form.Control
                        type="datetime-local"
                        name="date_validity"
                        value={form.date_validity || ''}
                        onChange={handleFormChange}
                      />
                    </Form.Group>
                  </Col>
                  <Col md={4}>
                    <Form.Group>
                      <Form.Label>Text</Form.Label>
                      <Form.Control
                        as="textarea"
                        name="certificate_text"
                        rows={3}
                        value={form.certificate_text || ''}
                        onChange={handleFormChange}
                      />
                    </Form.Group>
                  </Col>
                  <Col md={4} className="d-flex align-items-center">
                    <Form.Group>
                      <Form.Label>Is active</Form.Label>
                      <div>
                        <Form.Check
                          type="switch"
                          name="is_active"
                          checked={!!form.is_active}
                          onChange={handleFormChange}
                          label={form.is_active ? 'Yes' : 'No'}
                        />
                      </div>
                    </Form.Group>
                  </Col>
                </Row>
                <Button type="submit" variant="primary">Save</Button>
              </Form>
            </Card.Body>
          </Card>
        </Tab>
        <Tab eventKey="edit" title="Edit/Delete">
          {/* TODO: Edit/Delete UI and logic */}
        </Tab>
        <Tab eventKey="field" title="Fields">
          {/* TODO: Field edit UI and logic */}
        </Tab>
        <Tab eventKey="list" title="List">
          <Card className="mb-3">
            <Card.Body>
              <h4>Certificates List</h4>
              <Table striped bordered hover responsive>
                <thead>
                  <tr>
                    <th>Full Name</th>
                    <th>id_user</th>
                    <th>Certificate #</th>
                    <th>Type</th>
                    <th>Created At</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {certificates.map(cert => (
                    <tr key={cert.cerf_id || cert.id || Math.random()}>
                      <td>{(() => {
                        const user = users.find(u => u.Id === cert.user_id);
                        return user ? `${user.prenom || ''} ${user.nom || ''}`.trim() : cert.user_id;
                      })()}</td>
                      <td>{cert.user_id}</td>
                      <td>{cert.cerf_id}</td>
                      <td>{cert.certificate_type}</td>
                      <td>{cert.created_at}</td>
                      <td>
                        <Button
                          size="sm"
                          variant="info"
                          onClick={() => {
                            let url = '';
                            if (cert.certificate_type === 'medical') {
                              url = `/Company/Administrator/MedicalCertificatePrintView?cerf_id=${cert.cerf_id}`;
                            } else if (cert.certificate_type === 'legal') {
                              url = `/Company/Administrator/LegalCertificatePrintView?cerf_id=${cert.cerf_id}`;
                            } else if (cert.certificate_type === 'hipaa') {
                              url = `/Company/Administrator/HipaaCertificatePrintView?cerf_id=${cert.cerf_id}`;
                            } else {
                              url = `/Company/Administrator/certificate-print?cerf_id=${cert.cerf_id}`;
                            }
                            window.open(url, '_blank');
                          }}
                        >
                          View
                        </Button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </Table>
            </Card.Body>
          </Card>
          {/* Modal for preview */}
          <Modal show={showModal && !!previewCerfId} onHide={() => { setShowModal(false); setPreviewCerfId(null); }} size="lg">
            <Modal.Header closeButton><Modal.Title>Certificate Preview</Modal.Title></Modal.Header>
            <Modal.Body>
              {previewCerfId && (
                <LegalCertificateModal cerf_id={previewCerfId} onClose={() => { setShowModal(false); setPreviewCerfId(null); }} />
              )}
            </Modal.Body>
          </Modal>
          {/* TODO: Certificate list UI and print functionality */}
        </Tab>
      </Tabs>
      {/* Modal for QR code, certificate preview, etc. */}
      <Modal show={showModal} onHide={() => setShowModal(false)}>
        <Modal.Body>{modalContent}</Modal.Body>
      </Modal>
    </div>
  );
}

export default ManageCertificate;
