import React, { useEffect, useState } from 'react';
import { logAction } from '../../utils/actionLogger';
import Modal from 'react-modal';
import { Bar } from 'react-chartjs-2';
import { Chart, CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend } from 'chart.js';
import { BASE_API } from '../../utils/config';
Chart.register(CategoryScale, LinearScale, BarElement, Title, Tooltip, Legend);

const InterpreterChart = () => {
  const [chart, setChart] = useState({});
  const [year, setYear] = useState(new Date().getFullYear());
  const [loading, setLoading] = useState(false);
  const [editMonth, setEditMonth] = useState(null);
  const [editMinutes, setEditMinutes] = useState(0);
  const [editInterpreter, setEditInterpreter] = useState(null);
  const [selectedMonth, setSelectedMonth] = useState(new Date().getMonth());
  const [showAddModal, setShowAddModal] = useState(false);
  const [interpreterList, setInterpreterList] = useState([]);
  const [selectedInterpreter, setSelectedInterpreter] = useState('');
  const [addMinutes, setAddMinutes] = useState('');
  const [addMonth, setAddMonth] = useState(selectedMonth);
  // Fetch interpreters with accept_invoice=true
  useEffect(() => {
    fetch(`${BASE_API}/api/users`)
      .then(res => res.json())
      .then(data => {
        const interpreters = data.filter(u => {
          const invoiceTrue = u.invoice === true || u.invoice === 1 || u.invoice === '1' || u.invoice === 'true';
          return u.role === 'Interpreter' && invoiceTrue;
        });
        interpreters.forEach(u => {
          // Always show prenom, nom, and id_user if available
          const firstname = u.prenom || '';
          const lastname = u.nom || '';
          const id = u.id_user;
          u.displayName = `${firstname} ${lastname}`.trim();
        });
        setInterpreterList(interpreters);
      });
  }, []);

  useEffect(() => {
    setLoading(true);
    fetch(`${BASE_API}/api/interpreter_chart?year=${year}`)
      .then(res => res.json())
      .then(data => {
        setChart(data.chart || {});
        setLoading(false);
      })
      .catch(() => setLoading(false));
  }, [year]);

  const handleEdit = (interpreterId, monthIdx, currentMinutes) => {
    setEditInterpreter(interpreterId);
    setEditMonth(monthIdx);
    setEditMinutes(currentMinutes);
  };

  const handleSave = () => {
    fetch(`${BASE_API}/api/interpreter_chart/update_minutes`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        year,
        month: editMonth + 1,
        minutes: editMinutes,
        interpreter_id: editInterpreter
      })
    })
      .then(res => res.json())
      .then(() => {
        setEditInterpreter(null);
        setEditMonth(null);
        setEditMinutes(0);
        // Refresh chart
        fetch(`${BASE_API}/api/interpreter_chart?year=${year}`)
          .then(res => res.json())
          .then(data => setChart(data.chart || {}));
      });
  };

  // Prepare chart data for selected month
  const monthNames = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
  const interpreters = Object.values(chart);
  const labels = interpreters.map(i => {
    // Prefer name if available, fallback to email
    return i.name ? i.name : i.interpreter_email;
  });
  const dataForMonth = interpreters.map(i => i.monthly_minutes[selectedMonth] || 0);
  const totalMinutes = dataForMonth.reduce((a,b)=>a+b,0);
  const totalHours = (totalMinutes/60).toFixed(2);

  // Color palette for bars
  // Generate up to 100 visually distinct colors with maximized hue separation
  const barColors = Array.from({length: 100}, (_, i) => {
    // Use golden angle to maximize color separation
    const goldenAngle = 137.508;
    const hue = Math.round((i * goldenAngle) % 360);
    return `hsl(${hue}, 70%, 55%)`;
  });

  const chartData = {
    labels,
    datasets: [
      {
        label: monthNames[selectedMonth] + ' Minutes',
        data: dataForMonth,
        backgroundColor: labels.map((_,i)=>barColors[i%barColors.length]),
        borderRadius: 6,
      }
    ]
  };

  const chartOptions = {
    responsive: true,
    plugins: {
      legend: { display: false },
      title: { display: false },
      tooltip: { enabled: true }
    },
    scales: {
      x: { grid: { display: false }, ticks: { color: '#333', font: { size: 12 } } },
      y: { grid: { color: '#eee' }, beginAtZero: true, ticks: { color: '#333', font: { size: 12 } } }
    }
  };

  // Add minutes handler
  const handleAddMinutes = () => {
    if (!selectedInterpreter || !addMinutes || addMonth === null) return;
    setLoading(true);
    // Log edit action
    const actor_id = localStorage.getItem('userId');
    const actor_role = localStorage.getItem('role');
    const actor_email = localStorage.getItem('email');
    logAction({
      actor_id,
      actor_role,
      actor_email,
      action_type: 'edit_interpreter_chart',
      action_description: `Edited interpreter chart: interpreter_id=${selectedInterpreter}, month=${addMonth + 1}, minutes=${addMinutes}`,
      target_id: selectedInterpreter,
      target_role: 'Interpreter',
      page: window.location.pathname,
    });
    fetch(`${BASE_API}/api/interpreter_chart/update_minutes`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        year,
        month: addMonth + 1,
        minutes: addMinutes,
        interpreter_id: selectedInterpreter
      })
    })
      .then(res => res.json())
      .then(() => {
        // Update localStorage to mark this interpreter/month as added for today
        const today = new Date().toISOString().slice(0, 10); // YYYY-MM-DD
        const localKey = `interpreter_minutes_added_${today}`;
        let todayAdded = [];
        try {
          todayAdded = JSON.parse(localStorage.getItem(localKey)) || [];
        } catch (e) {
          todayAdded = [];
        }
        const entry = `${selectedInterpreter}_${addMonth}`;
        if (!todayAdded.includes(entry)) {
          todayAdded.push(entry);
          localStorage.setItem(localKey, JSON.stringify(todayAdded));
        }
        setShowAddModal(false);
        setSelectedInterpreter('');
        setAddMinutes('');
        setAddMonth(selectedMonth);
        setLoading(false);
        // Refresh chart
        fetch(`${BASE_API}/api/interpreter_chart?year=${year}`)
          .then(res => res.json())
          .then(data => setChart(data.chart || {}));
      });
  };
  return (
    <div style={{marginTop:'32px', background:'#fff', borderRadius:'16px', boxShadow:'0 2px 12px rgba(0,0,0,0.08)', padding:'24px', maxWidth:'100%'}}>
      <div style={{display:'flex', justifyContent:'space-between', alignItems:'center', marginBottom:'12px'}}>
        <div>
          <button style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', marginRight:'8px', cursor:'pointer'}} onClick={()=>setShowAddModal(true)}>Edit</button>
          <button style={{background:'#43a047', color:'#fff', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={() => {
            // Log print action
            const actor_id = localStorage.getItem('userId');
            const actor_role = localStorage.getItem('role');
            const actor_email = localStorage.getItem('email');
            logAction({
              actor_id,
              actor_role,
              actor_email,
              action_type: 'print_interpreter_chart',
              action_description: 'Printed interpreter chart',
              page: window.location.pathname,
            });
            // You can add actual print logic here if needed
          }}>Print Chart</button>
        </div>
        <div>
          <label style={{fontWeight:'bold', marginRight:'8px'}}>Year:</label>
          <input type="number" value={year} onChange={e=>setYear(Number(e.target.value))} style={{padding:'6px', borderRadius:'6px', border:'1px solid #ccc', width:'80px'}} />
        </div>
      </div>
      {/* Add/Edit Modal */}
      <Modal
        isOpen={showAddModal}
        onRequestClose={()=>setShowAddModal(false)}
        style={{overlay:{zIndex:9999, background:'rgba(0,0,0,0.18)'}, content:{borderRadius:'18px', padding:'32px', maxWidth:'500px', margin:'auto', boxShadow:'0 4px 24px rgba(0,0,0,0.18)'}}}
        ariaHideApp={false}
      >
        <h2 style={{marginBottom:'18px'}}>Add or Edit</h2>
        <div style={{marginBottom:'12px', fontWeight:'bold'}}>Select Interpreter ID</div>
        <select value={selectedInterpreter} onChange={e=>setSelectedInterpreter(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc', marginBottom:'12px'}}>
          <option value="">Select Interpreter ID</option>
          {interpreterList.map(u => {
            // Check if this interpreter already has minutes for the selected month TODAY (using localStorage)
            const today = new Date().toISOString().slice(0, 10); // YYYY-MM-DD
            const localKey = `interpreter_minutes_added_${today}`;
            let todayAdded = [];
            try {
              todayAdded = JSON.parse(localStorage.getItem(localKey)) || [];
            } catch (e) {
              todayAdded = [];
            }
            const id = u.id_user || u.Id;
            const alreadyAdded = todayAdded.includes(`${id}_${addMonth}`);
            return (
              <option
                key={id}
                value={id}
                style={alreadyAdded ? {opacity:0.5, pointerEvents:'auto'} : {}}
                disabled={false} // keep clickable
              >
                {u.displayName} ({id}){alreadyAdded ? ' (already added)' : ''}
              </option>
            );
          })}
        </select>
        <div style={{marginBottom:'12px', fontWeight:'bold'}}>Minutes quantity</div>
        <input type="number" value={addMinutes} onChange={e=>setAddMinutes(e.target.value)} placeholder="Minutes" style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc', marginBottom:'12px'}} />
        <div style={{marginBottom:'12px', fontWeight:'bold'}}>Month</div>
        <select value={addMonth} onChange={e=>setAddMonth(Number(e.target.value))} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc', marginBottom:'12px'}}>
          {monthNames.map((m, idx) => (
            <option key={m} value={idx}>{m}</option>
          ))}
        </select>
        <div style={{display:'flex', gap:'16px', justifyContent:'flex-end'}}>
          <button style={{background:'#eaf6ff', color:'#1565c0', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={()=>setShowAddModal(false)}>Close</button>
          <button style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={handleAddMinutes}>Add</button>
        </div>
      </Modal>
      <div style={{display:'flex', gap:'8px', justifyContent:'center', marginBottom:'18px', flexWrap:'wrap'}}>
        {monthNames.map((m, idx) => (
          <button key={m} style={{background: selectedMonth===idx ? '#ffa726' : '#90caf9', color:'#fff', border:'none', borderRadius:'16px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} onClick={()=>setSelectedMonth(idx)}>{m}</button>
        ))}
      </div>
      <div style={{textAlign:'center', marginBottom:'8px', fontSize:'1.6rem', fontWeight:'bold'}}>{monthNames[selectedMonth]} Reports</div>
      <div style={{textAlign:'center', fontSize:'1.2rem', marginBottom:'4px'}}>{totalMinutes.toLocaleString()} Mns | {totalHours}Hrs</div>
      <div style={{textAlign:'center', color:'#43a047', fontWeight:'bold', fontSize:'1.1rem', marginBottom:'12px'}}>{new Date().toISOString().slice(0,10)} ({new Date().toLocaleString('default', { weekday: 'short' })}) {new Date().toLocaleTimeString()}</div>
      {loading ? <div>Loading...</div> : (
        <div style={{width:'100%', minHeight:'400px'}}>
          <Bar data={chartData} options={chartOptions} />
        </div>
      )}
    </div>
  );
};

export default InterpreterChart;
