import React, { useEffect, useState, useRef, useCallback } from 'react';
import axios from 'axios';
import { BASE_API } from '../../utils/config';
// For drag-and-drop
import { useRef as useDragRef } from 'react';
import { useNavigate } from 'react-router-dom';
import DashboardSidebar from './DashboardSidebar';
import { logAction } from '../../utils/actionLogger';
import WebsiteTrafficPage from './WebsiteTrafficPage';
import ManageInsight from './ManageInsight';
import MyTutorial from '../MyTutorial';
import InvoicePage from './InvoicePage';
import AnnouncementPage from '../AnnouncementPage';
import AnnouncementManager from './AnnouncementManager';
import CourseManager from './CourseManager';
import UsersManager from './UsersManager';
import SignatureManager from '../Document_signature/SignatureManager';
import TasksManager from './TasksManager';
import TracesManager from './TracesManager';
import InterpreterChart from './InterpreterChart';
import MyDocument from './MyDocument';
import ManageMedia from './ManageMedia';
import AddCourseContent from './AddCourseContent';
import CourseDashboard from './CourseDashboard';
import QuizManager from './QuizManager';
import MedicalDocumentGlossary from '../../MedicalDocumentGlossary';
import ManageCertificate from './ManageCertificate';
import ProfilePage from '../../Company/ProfilePage';
import UserAccessManager from './UserAccessManager';
import CourseAssignment from './CourseAssignment'; // Import CourseAssignment
import Teachers from './Teachers';


const cardStyle = {
  padding: '24px',
  marginBottom: '24px',
  background: '#fff',
  borderRadius: '16px',
  boxShadow: '0 2px 12px rgba(0,0,0,0.08)',
};

const statGrid = {
  display: 'flex',
  gap: '24px',
  marginBottom: '24px',
};

const announcementStyle = {
  ...cardStyle,
  borderLeft: '6px solid #a259e6',
};

const activitiesStyle = {
  ...cardStyle,
  background: '#eaf6ff',
};


// Right sidebar styles moved to CSS class for responsiveness

const AdminIndex = () => {
  // Avatar upload
  const [avatar, setAvatar] = useState('https://via.placeholder.com/36x36.png?text=Mi');
  // Avatar sync handler for ProfilePage
  const handleAvatarChange = useCallback((imgUrl) => {
    setAvatar(imgUrl || 'https://via.placeholder.com/36x36.png?text=Mi');
  }, []);

  // Fetch avatar on initial load
  useEffect(() => {
    const userId = localStorage.getItem('userId');
    if (!userId) return;
    (async () => {
      try {
        const res = await fetch(`/api/users/by-iduser/${userId}`);
        const data = await res.json();
        if (data.user && data.user.img) {
          const imgPath = data.user.img.startsWith('/profile/') ? BASE_API + data.user.img : data.user.img;
          setAvatar(imgPath);
        }
      } catch (e) {
        // ignore
      }
    })();
  }, []);
  // News & Blog state
  const [newsBlog, setNewsBlog] = useState([]);
  const [newsBlogLoading, setNewsBlogLoading] = useState(true);
  const [newsBlogError, setNewsBlogError] = useState('');
  // Users stats state
  const [users, setUsers] = useState([]);
  const [usersLoading, setUsersLoading] = useState(true);
  const [usersError, setUsersError] = useState('');
  // Roles state
  const [roles, setRoles] = useState([]);
  const [rolesLoading, setRolesLoading] = useState(true);
  const [rolesError, setRolesError] = useState('');
  // Fetch users and roles for stats
  useEffect(() => {
    setUsersLoading(true);
    setUsersError('');
    setRolesLoading(true);
    setRolesError('');
    // Fetch users
    fetch('/api/users')
      .then(res => res.json())
      .then(data => {
        setUsers(Array.isArray(data) ? data : []);
        setUsersLoading(false);
      })
      .catch(() => {
        setUsersError('Failed to load users.');
        setUsersLoading(false);
      });
    // Fetch roles
    fetch('/api/roles/list')
      .then(res => res.json())
      .then(data => {
        setRoles(Array.isArray(data.roles) ? data.roles : []);
        setRolesLoading(false);
      })
      .catch(() => {
        setRolesError('Failed to load roles.');
        setRolesLoading(false);
      });
  }, []);
  // Fetch latest news and blog posts
  useEffect(() => {
    setNewsBlogLoading(true);
    setNewsBlogError('');
    fetch('/api/news_blog')
      .then(res => res.json())
      .then(data => {
        setNewsBlog(Array.isArray(data) ? data : (data.news_blog || []));
        setNewsBlogLoading(false);
      })
      .catch(() => {
        setNewsBlogError('Failed to load news and blog posts.');
        setNewsBlogLoading(false);
      });
  }, []);
  // Announcement modal state
  // Recent actions state
  const [recentActions, setRecentActions] = useState([]);
  const [actionsLoading, setActionsLoading] = useState(true);
  const [actionsError, setActionsError] = useState('');
  // Fetch recent actions for right panel
  useEffect(() => {
    const userId = localStorage.getItem('userId');
    const role = localStorage.getItem('role');
    if (!userId || !role) return;
    setActionsLoading(true);
    axios.get('/api/recent-actions', { params: { user_id: userId, role } })
      .then(res => {
        setRecentActions(res.data.actions || []);
        setActionsLoading(false);
      })
      .catch(() => {
        setActionsError('Failed to load recent actions');
        setActionsLoading(false);
      });
  }, []);
  // Log dashboard access on mount
  useEffect(() => {
    const actor_id = localStorage.getItem('userId');
    const actor_role = localStorage.getItem('role');
    const actor_email = localStorage.getItem('email');
    if (actor_id && actor_role && actor_email) {
      logAction({
        actor_id,
        actor_role,
        actor_email,
        action_type: 'view_dashboard',
        action_description: 'Accessed Administrator dashboard',
        page: '/Company/Administrator/index',
      });
    }
  }, []);
  const [showAnnouncements, setShowAnnouncements] = useState(true);
  // Role state (assuming administrator for this component)
  const [role, setRole] = useState('administrator');
  // Sidebar menu selection state
  const [selectedMenu, setSelectedMenu] = useState('dashboard');
  // Profile page state
  const [showProfilePage, setShowProfilePage] = useState(false);
  // Get current user info for ProfilePage
  const currentUser = {
    id: localStorage.getItem('userId'),
    email: localStorage.getItem('email'),
    role: localStorage.getItem('role'),
  };

  // Theme state
  const [darkMode, setDarkMode] = useState(false);
  const [accentColor, setAccentColor] = useState('#1565c0');
  const [compactMode, setCompactMode] = useState(false);
  // Profile dropdown state
  const [profileOpen, setProfileOpen] = useState(false);
  // Notification dropdown state
  const [notifOpen, setNotifOpen] = useState(false);
  // Notification settings dropdown
  const [notifSettingsOpen, setNotifSettingsOpen] = useState(false);
  // Animated search state
  const [searchFocus, setSearchFocus] = useState(false);
  // Notification badge
  const [notifCount, setNotifCount] = useState(3);
  // Language switcher
  const [language, setLanguage] = useState('EN');
  // Help modal
  const [helpOpen, setHelpOpen] = useState(false);
  // Widget controls
  const [widgets, setWidgets] = useState([
    { id: 'announcements', name: 'Announcements', visible: true, size: 1 },
    { id: 'stats', name: 'Stats', visible: true, size: 1 },
  ]);
  // Drag-and-drop
  const [draggedWidget, setDraggedWidget] = useState(null);
  // Ref for closing dropdowns
  const profileRef = useRef();
  const notifRef = useRef();
  const [selectedCourseId, setSelectedCourseId] = useState('');
  const [visibleAnnouncements, setVisibleAnnouncements] = useState([]);
  function getExpirationSeverity(expirationDate) {
    if (!expirationDate) return { severity: 'normal', message: '' };
    const today = new Date();
    const expDate = new Date(expirationDate);
    // Remove time part for comparison
    today.setHours(0,0,0,0);
    expDate.setHours(0,0,0,0);
    const diffDays = Math.ceil((expDate - today) / (1000 * 60 * 60 * 24));
    if (diffDays < 0) {
      return { severity: 'expired', message: 'Expired' };
    } else if (diffDays === 0) {
      return { severity: 'expired', message: 'Expired today' };
    } else if (diffDays === 1) {
      return { severity: 'twoDays', message: 'Expires tomorrow' };
    } else if (diffDays === 2) {
      return { severity: 'twoDays', message: 'Expires in two days' };
    } else if (diffDays >= 3) {
      return { severity: 'almost', message: 'Almost expired' };
    } else {
      return { severity: 'normal', message: '' };
    }
  }

  // Close dropdowns on outside click
  useEffect(() => {
    function handleClick(e) {
      if (profileRef.current && !profileRef.current.contains(e.target)) setProfileOpen(false);
      if (notifRef.current && !notifRef.current.contains(e.target)) setNotifOpen(false);
    }
    document.addEventListener('mousedown', handleClick);
    return () => document.removeEventListener('mousedown', handleClick);
  }, []);

  // Fetch announcements and filter for Administrator
  useEffect(() => {
    async function fetchAnnouncements() {
      try {
        const res = await fetch('/api/announcements');
        const data = await res.json();
        const announcements = data.announcements || [];
        // Filtering logic for Administrator
        const adminId = 1;
        const adminEmail = 'admin@example.com';
        const adminRole = 'Administrator';
        const filtered = announcements.filter(a => {
          if (a.is_active !== 1) return false;
          if (!a.target_roles?.trim() && !a.target_users?.trim()) return true;
          if (a.target_roles && a.target_roles.split(',').map(r => r.trim()).includes(adminRole)) return true;
          if (a.target_users && (a.target_users.split(',').map(u => u.trim()).includes(String(adminId)) || a.target_users.split(',').map(u => u.trim()).includes(adminEmail))) return true;
          return false;
        });
        setVisibleAnnouncements(filtered);
      } catch (err) {
        setVisibleAnnouncements([]);
      }
    }
    fetchAnnouncements();
  }, []);

  // Theme styles
  const themeStyles = darkMode ? {
    background: '#181c24', color: '#fff', transition: 'background 0.2s', minHeight: '100vh'
  } : {
    background: '#f7f8fa', color: '#222', minHeight: '100vh', transition: 'background 0.2s'
  };
  themeStyles['--accent'] = accentColor;

  // Top bar style
  const topBarStyle = {
    width:'100vw', height:'64px', background: darkMode ? '#23283a' : '#fff', boxShadow:'0 2px 8px rgba(0,0,0,0.04)', display:'flex', alignItems:'center', justifyContent:'space-between', padding:'0 32px', position:'fixed', top:0, left:0, zIndex:10000
  };
  if (compactMode) topBarStyle.height = '48px';

  // Animated search style
  const searchInputStyle = {
    width: searchFocus ? '340px' : '180px',
    padding:'8px 16px', borderRadius:'8px', border:'1px solid #e0e0e0', fontSize:'1rem', marginRight:'16px', transition:'width 0.3s', outline: searchFocus ? '2px solid #1565c0' : 'none'
  };

  // Responsive design
  const isMobile = window.innerWidth < 700;
  
  // Responsive container style
  const responsiveContainerStyle = {
    display: 'flex',
    minHeight: '100vh',
    paddingLeft: isMobile ? '0' : '280px',
    paddingRight: isMobile ? '0' : '340px',
  };
  
  return (
    <>
      {/* Announcement Modal Popup on Login */}
      {showAnnouncements && visibleAnnouncements.length > 0 && (
        <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:9999, display:'flex', alignItems:'center', justifyContent:'center'}}>
          <div style={{background:'#fff', borderRadius:'16px', boxShadow:'0 2px 24px rgba(0,0,0,0.18)', minWidth:'420px', maxWidth:'90vw', maxHeight:'90vh', overflowY:'auto', padding:'32px', position:'relative'}}>
            <h2 style={{marginBottom:'18px', color:'#1976d2'}}>Announcements</h2>
            {visibleAnnouncements.map(a => {
              const expInfo = getExpirationSeverity(a.expiration_date);
              return (
                <div key={a.id} style={{marginBottom:'24px', background:'#fff', borderRadius:'12px', boxShadow:'0 2px 8px rgba(0,0,0,0.08)', padding:'24px', marginBottom:'8px'}}>
                  <div style={{marginBottom:'8px'}}>
                    <strong style={{fontSize:'1.25rem', marginBottom:'8px', display:'inline-block'}}>{a.title}</strong>
                    {a.announcement_type && (
                      <span style={{
                        background: announcementTypeColors[a.announcement_type] || '#1976d2',
                        color: '#fff',
                        borderRadius: '6px',
                        padding: '2px 10px',
                        marginLeft: '12px',
                        fontSize: '0.95rem',
                        fontWeight: 'bold',
                        verticalAlign: 'middle',
                        display: 'inline-block',
                      }}>{a.announcement_type}</span>
                    )}
                    <div style={{ color: expInfo.message ? expirationSeverityColors[expInfo.severity] : expirationSeverityColors.normal, fontSize: '0.95rem', fontWeight:'bold' }}>
                      {expInfo.message ? `${expInfo.message} (${a.expiration_date})` : `Expires: ${a.expiration_date}`}
                    </div>
                  </div>
                </div>
              );
            })}
            <button style={{marginTop:'18px', background:'#1976d2', color:'#fff', border:'none', borderRadius:'8px', padding:'8px 24px', fontWeight:'bold', cursor:'pointer'}} onClick={()=>setShowAnnouncements(false)}>Close</button>
          </div>
        </div>
      )}
      <div style={{...responsiveContainerStyle, ...themeStyles}}>
      {/* Top Bar */}
      <div style={topBarStyle}>
        <div style={{flex:1, display:'flex', alignItems:'center'}}>
          <input type="text" placeholder="Search users, courses, notifications..." style={searchInputStyle} onFocus={()=>setSearchFocus(true)} onBlur={()=>setSearchFocus(false)} />
          <button style={{background:'none', border:'none', cursor:'pointer'}}>
            <svg width="22" height="22" fill="none" viewBox="0 0 24 24"><circle cx="11" cy="11" r="8" stroke={accentColor} strokeWidth="2"/><path d="M21 21l-4.35-4.35" stroke={accentColor} strokeWidth="2" strokeLinecap="round"/></svg>
          </button>
          {/* Quick Actions */}
          <button style={{marginLeft:'16px', background:accentColor, color:'#fff', border:'none', borderRadius:'8px', padding:'6px 16px', cursor:'pointer', fontWeight:'bold'}}>+ Add Course</button>
          <button style={{marginLeft:'8px', background:'#27ae60', color:'#fff', border:'none', borderRadius:'8px', padding:'6px 16px', cursor:'pointer', fontWeight:'bold'}}>+ Invite User</button>
          {/* Language Switcher */}
          <select value={language} onChange={e=>setLanguage(e.target.value)} style={{marginLeft:'16px', borderRadius:'8px', padding:'6px', border:'1px solid #e0e0e0', fontWeight:'bold', color:accentColor}}>
            <option value="EN">EN</option>
            <option value="FR">FR</option>
            <option value="ES">ES</option>
          </select>
          {/* Help Button */}
          <button style={{marginLeft:'8px', background:'none', border:'none', cursor:'pointer'}} onClick={()=>setHelpOpen(true)} title="Help">
            <svg width="22" height="22" fill="none" viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke={accentColor} strokeWidth="2"/><text x="8" y="16" fontSize="12" fill={accentColor}>?</text></svg>
          </button>
        </div>
        <div style={{display:'flex', alignItems:'center', gap:'24px'}}>
          {/* Theme Switcher */}
          <button style={{background:'none', border:'none', cursor:'pointer'}} onClick={()=>setDarkMode(!darkMode)} title="Toggle theme">
            {darkMode ? (
              <svg width="24" height="24" fill="none" viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke="#fff" strokeWidth="2"/><path d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" fill="#fff"/></svg>
            ) : (
              <svg width="24" height="24" fill="none" viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke={accentColor} strokeWidth="2"/><path d="M12 7v5l4 2" stroke={accentColor} strokeWidth="2" strokeLinecap="round"/></svg>
            )}
          </button>
          {/* Accent Color Theme Picker */}
          <input type="color" value={accentColor} onChange={e=>setAccentColor(e.target.value)} style={{marginLeft:'8px', width:'32px', height:'32px', border:'none', background:'none', cursor:'pointer'}} title="Accent Color" />
          {/* Compact Mode Toggle */}
          <button style={{background:'none', border:'none', cursor:'pointer'}} onClick={()=>setCompactMode(!compactMode)} title="Compact Mode">
            <svg width="24" height="24" fill="none" viewBox="0 0 24 24"><rect x="4" y="8" width="16" height="8" rx="2" stroke={accentColor} strokeWidth="2"/></svg>
          </button>
          {/* Notification Icon with badge and dropdown */}
          <div style={{position:'relative'}} ref={notifRef}>
            <button style={{background:'none', border:'none', cursor:'pointer'}} onClick={()=>setNotifOpen(!notifOpen)}>
              <svg width="24" height="24" fill="none" viewBox="0 0 24 24"><path d="M12 22c1.1 0 2-.9 2-2h-4a2 2 0 002 2zm6-6V11c0-3.07-1.63-5.64-4.5-6.32V4a1.5 1.5 0 00-3 0v.68C7.63 5.36 6 7.92 6 11v5l-1.7 1.7a1 1 0 00.7 1.7h14a1 1 0 00.7-1.7L18 16z" stroke={darkMode ? '#fff' : accentColor} strokeWidth="2"/></svg>
              {notifCount > 0 && <span style={{position:'absolute', top:'-4px', right:'-4px', background:'#f75c7a', color:'#fff', borderRadius:'50%', fontSize:'0.8rem', padding:'2px 6px', fontWeight:'bold'}}>{notifCount}</span>}
            </button>
            {notifOpen && (
              <div style={{position:'absolute', top:'40px', right:0, background:darkMode ? '#23283a' : '#fff', boxShadow:'0 2px 8px rgba(0,0,0,0.12)', borderRadius:'12px', minWidth:'260px', zIndex:200, padding:'12px', transition:'all 0.2s'}}>
                <div style={{fontWeight:'bold', marginBottom:'8px'}}>Notifications</div>
                <div style={{marginBottom:'8px', color:'#27ae60'}}>You just Login <span style={{color:'#555', fontSize:'0.9rem'}}>1 second ago</span></div>
                <div style={{marginBottom:'8px', color:'#f75c7a'}}>System has been log you out <span style={{color:'#555', fontSize:'0.9rem'}}>1 day ago</span></div>
                <div style={{marginBottom:'8px', color:'#27ae60'}}>You just Login <span style={{color:'#555', fontSize:'0.9rem'}}>1 day ago</span></div>
                <button style={{marginTop:'8px', background:accentColor, color:'#fff', border:'none', borderRadius:'8px', padding:'6px 12px', cursor:'pointer', fontWeight:'bold'}}>View All</button>
                <button style={{marginTop:'8px', background:'none', color:accentColor, border:'none', borderRadius:'8px', padding:'6px 12px', cursor:'pointer', fontWeight:'bold'}} onClick={()=>setNotifSettingsOpen(!notifSettingsOpen)}>Settings</button>
                {notifSettingsOpen && (
                  <div style={{marginTop:'8px', background:darkMode ? '#23283a' : '#f7f8fa', borderRadius:'8px', padding:'8px', boxShadow:'0 2px 8px rgba(0,0,0,0.08)'}}>
                    <div style={{fontWeight:'bold', marginBottom:'4px'}}>Notification Settings</div>
                    <label style={{display:'block', marginBottom:'4px'}}><input type="checkbox" defaultChecked /> Login Alerts</label>
                    <label style={{display:'block', marginBottom:'4px'}}><input type="checkbox" defaultChecked /> System Alerts</label>
                    <label style={{display:'block', marginBottom:'4px'}}><input type="checkbox" /> Promotions</label>
                  </div>
                )}
              </div>
            )}
          </div>
          {/* Message Icon */}
          <button style={{background:'none', border:'none', cursor:'pointer'}}>
            <svg width="24" height="24" fill="none" viewBox="0 0 24 24"><rect x="4" y="8" width="16" height="10" rx="2" stroke={darkMode ? '#fff' : '#1565c0'} strokeWidth="2"/><path d="M4 8V6a2 2 0 012-2h12a2 2 0 012 2v2" stroke={darkMode ? '#fff' : '#1565c0'} strokeWidth="2"/></svg>
          </button>
          {/* Profile Avatar & Dropdown */}
          <div style={{position:'relative'}} ref={profileRef}>
            <div 
              style={{
                display:'flex', alignItems:'center', gap:'10px', cursor:'pointer',
                padding:'4px 12px', borderRadius:'24px',
                background: profileOpen ? '#f5f7fa' : 'transparent',
                boxShadow: profileOpen ? '0 2px 12px rgba(67,206,162,0.10)' : 'none',
                transition:'background 0.2s, box-shadow 0.2s',
                border: profileOpen ? `1.5px solid ${accentColor}` : '1.5px solid transparent',
                minHeight:'44px'
              }}
              onClick={()=>setProfileOpen(!profileOpen)}
              onMouseEnter={e=>e.currentTarget.style.background='#f5f7fa'}
              onMouseLeave={e=>!profileOpen && (e.currentTarget.style.background='transparent')}
            >
              <img src={avatar} alt="Profile" style={{width:'38px', height:'38px', borderRadius:'50%', border:`2.5px solid ${accentColor}`, boxShadow:'0 2px 8px rgba(67,206,162,0.10)', objectFit:'cover', background:'#fff'}} />
              <span style={{fontWeight:'bold', color:accentColor, fontSize:'1.08rem', letterSpacing:'0.01em'}}>Administrator</span>
              <span style={{width:'10px', height:'10px', background:'#27ae60', borderRadius:'50%', display:'inline-block', marginLeft:'4px', border:'2px solid #fff', boxShadow:'0 1px 4px rgba(39,174,96,0.10)'}}></span>
              <svg width="16" height="16" fill="none" viewBox="0 0 24 24"><path d="M7 10l5 5 5-5" stroke={accentColor} strokeWidth="2"/></svg>
            </div>
            {profileOpen && (
              <div style={{
                position:'absolute', top:'48px', right:0, background:darkMode ? '#23283a' : '#fff',
                boxShadow:'0 8px 32px rgba(67,206,162,0.18)', borderRadius:'14px', minWidth:'200px', zIndex:10002, padding:'16px', transition:'all 0.2s',
                border:`1.5px solid ${accentColor}`,
                display:'flex', flexDirection:'column', alignItems:'stretch',
              }}>
                <div style={{fontWeight:'bold', marginBottom:'10px', color:accentColor, fontSize:'1.08rem'}}>Profile</div>
                <div style={{marginBottom:'10px', cursor:'pointer', color:'#1976d2', fontWeight:500}}>Settings</div>
                <div
                  style={{marginBottom:'10px', cursor:'pointer', color:'#1976d2', fontWeight:500}}
                  onClick={() => {
                    setShowProfilePage(true);
                    setProfileOpen(false);
                  }}
                >My Account</div>
                <div
                  style={{marginBottom:'10px', cursor:'pointer', color:'#f75c7a', fontWeight:600}}
                  onClick={async () => {
                    // Set user enligne to 0 before logout
                    try {
                      const userId = localStorage.getItem('userId');
                      if (userId) {
                        await fetch(`/api/users/${userId}/enligne`, {
                          method: 'POST',
                          headers: { 'Content-Type': 'application/json' },
                          body: JSON.stringify({ enligne: 0 })
                        });
                      }
                    } catch (e) {}
                    // Log logout action
                    try {
                      const actor_id = localStorage.getItem('userId');
                      const actor_role = localStorage.getItem('role');
                      const actor_email = localStorage.getItem('email');
                      if (actor_id && actor_role && actor_email) {
                        logAction({
                          actor_id,
                          actor_role,
                          actor_email,
                          action_type: 'logout',
                          action_description: 'User logged out',
                          page: window.location.pathname,
                        });
                      }
                    } catch (e) {}
                    // Clear localStorage and redirect to /Login
                    localStorage.clear();
                    window.location.href = '/Login';
                  }}
                >
                  Logout
                </div>
              </div>
            )}
          </div>
      {/* Help Modal */}
      {helpOpen && (
        <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.2)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}} onClick={()=>setHelpOpen(false)}>
          <div style={{background:darkMode ? '#23283a' : '#fff', color:darkMode ? '#fff' : '#222', borderRadius:'16px', padding:'32px', minWidth:'320px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}} onClick={e=>e.stopPropagation()}>
            <h2 style={{marginBottom:'16px'}}>Help & Support</h2>
            <p>Welcome to the dashboard! Here you can manage users, courses, notifications, and more.<br/>For documentation or support, visit <a href="#" style={{color:accentColor}}>our help center</a>.</p>
            <button style={{marginTop:'16px', background:accentColor, color:'#fff', border:'none', borderRadius:'8px', padding:'6px 16px', cursor:'pointer', fontWeight:'bold'}} onClick={()=>setHelpOpen(false)}>Close</button>
          </div>
        </div>
      )}
        </div>
      </div>

      {/* Left Sidebar */}
      <div style={{marginTop:'64px'}}>
        <DashboardSidebar
          role={role}
          onSelectMenu={(menu) => {
            setSelectedMenu(menu);
            setShowProfilePage(false); // Hide profile page when switching menu
          }}
          selectedMenu={selectedMenu}
        />
      </div>

      {/* Main Content - Switch based on selectedMenu or profile page */}
      <main style={{marginTop:'64px', padding:'32px', flex:1}}>
        {showProfilePage ? (
          (() => {
            const id = localStorage.getItem('userId');
            const email = localStorage.getItem('email');
            const role = localStorage.getItem('role');
            const currentUser = { id, email, role };
            return <ProfilePage currentUser={currentUser} onAvatarChange={handleAvatarChange} />;
          })()
        ) : selectedMenu === 'medicalDocumentGlossary' ? (
          <MedicalDocumentGlossary documentId={1} userId={1} />
        ) : selectedMenu === 'manageInsight' ? (
          <ManageInsight />
        ) : selectedMenu === 'myTutorial' ? (
          <MyTutorial userId={localStorage.getItem('userId')} />
        ) : selectedMenu === 'manageTutorial' ? (
          require('./ManageTutorial').default ? React.createElement(require('./ManageTutorial').default) : null
        ) : selectedMenu === 'announcementsPage' ? (
          <AnnouncementPage currentUser={{ id: 1, email: 'admin@example.com', role: 'Administrator' }} />
        ) : selectedMenu === 'editCourse' ? (
          <CourseManager onSelectCourse={setSelectedCourseId} />
        ) : selectedMenu === 'addCourseContent' ? (
          <AddCourseContent courseId={selectedCourseId || null} />
        ) : selectedMenu === 'tasks' ? (
          <TasksManager />
        ) : selectedMenu === 'signature' ? (
          <SignatureManager />
        ) : selectedMenu === 'users' ? (
          <UsersManager />
        ) : selectedMenu === 'traces' ? (
          <TracesManager />
        ) : selectedMenu === 'manageMedia' ? (
          <ManageMedia />
        ) : selectedMenu === 'courses' ? (
          <CourseDashboard userId={1} isAdmin={true} />
        ) : selectedMenu === 'announcements' ? (
          <AnnouncementManager currentUser={{ id: 1, email: 'admin@example.com' }} />
        ) : selectedMenu === 'invoicing' ? (
          <InvoicePage />
        ) : selectedMenu === 'quizManagement' ? (
          <QuizManager />
        ) : selectedMenu === 'mydocument' ? (
          <MyDocument userId={1} role={role} />
        ) : selectedMenu === 'websiteTraffic' ? (
          <WebsiteTrafficPage />
        ) : selectedMenu === 'manageCertificate' ? (
          <ManageCertificate />
        ) : selectedMenu === 'userAccess' ? (
          <UserAccessManager />
        ) : selectedMenu === 'paymentUserMethod' ? (
          require('./PaymentUserMethod').default ? React.createElement(require('./PaymentUserMethod').default) : null
        ) : selectedMenu === 'assignCourse' ? (
          <CourseAssignment />
        ) : selectedMenu === 'addTeacher' ? (
          <Teachers />
        ) : (
          <>
            <div style={{display:'flex', gap:'16px', flexWrap:'wrap'}}>
              {widgets.map((widget, idx) => widget.visible && (
                <div
                  key={widget.id}
                  draggable
                  onDragStart={()=>setDraggedWidget(idx)}
                  onDragOver={e=>e.preventDefault()}
                  onDrop={()=>{
                    if (draggedWidget !== null && draggedWidget !== idx) {
                      const newWidgets = [...widgets];
                      const [moved] = newWidgets.splice(draggedWidget, 1);
                      newWidgets.splice(idx, 0, moved);
                      setWidgets(newWidgets);
                      setDraggedWidget(null);
                    }
                  }}
                  style={{flex:widget.size, minWidth:'260px', transition:'all 0.2s', position:'relative'}}
                >
                  {/* Widget Settings */}
                  <button style={{position:'absolute', top:'8px', right:'8px', background:'none', border:'none', cursor:'pointer', zIndex:2}} title="Widget Settings" onClick={()=>{
                    const newWidgets = widgets.map((w,i)=>i===idx?{...w, settingsOpen:!w.settingsOpen}:w);
                    setWidgets(newWidgets);
                  }}>
                    <svg width="18" height="18" fill="none" viewBox="0 0 24 24"><circle cx="12" cy="12" r="10" stroke={accentColor} strokeWidth="2"/><path d="M12 8v4l3 2" stroke={accentColor} strokeWidth="2" strokeLinecap="round"/></svg>
                  </button>
                  {widget.settingsOpen && (
                    <div style={{position:'absolute', top:'36px', right:'8px', background:'#fff', borderRadius:'8px', boxShadow:'0 2px 8px rgba(0,0,0,0.08)', padding:'8px', zIndex:3}}>
                      <label style={{display:'block', marginBottom:'4px'}}>
                        <input type="checkbox" checked={widget.visible} onChange={e=>{
                          const newWidgets = widgets.map((w,i)=>i===idx?{...w, visible:e.target.checked}:w);
                          setWidgets(newWidgets);
                        }} /> Show Widget
                      </label>
                      <label style={{display:'block', marginBottom:'4px'}}>
                        Size:
                        <input type="range" min={1} max={2} value={widget.size} onChange={e=>{
                          const newWidgets = widgets.map((w,i)=>i===idx?{...w, size:Number(e.target.value)}:w);
                          setWidgets(newWidgets);
                        }} />
                      </label>
                      <button style={{marginTop:'4px', background:'#f75c7a', color:'#fff', border:'none', borderRadius:'6px', padding:'4px 10px', cursor:'pointer'}} onClick={()=>{
                        const newWidgets = widgets.filter((_,i)=>i!==idx);
                        setWidgets(newWidgets);
                      }}>Remove</button>
                    </div>
                  )}
                  {/* Widget Content */}
                  {widget.id==='stats' && (
                    <div style={statGrid}>
                      <div style={cardStyle}>
                        <div style={{fontWeight:'bold', color:accentColor}}>Courses</div>
                        <div style={{fontSize:'2rem', fontWeight:'bold'}}>6</div>
                        <div style={{color:'#27ae60'}}>60% increase</div>
                      </div>
                      <div style={cardStyle}>
                        <div style={{fontWeight:'bold', color:accentColor}}>Users</div>
                        {usersLoading || rolesLoading ? (
                          <div style={{fontSize:'1.2rem'}}>Loading...</div>
                        ) : usersError ? (
                          <div style={{fontSize:'1.1rem', color:'#f75c7a'}}>{usersError}</div>
                        ) : rolesError ? (
                          <div style={{fontSize:'1.1rem', color:'#f75c7a'}}>{rolesError}</div>
                        ) : (
                          <>
                            <div style={{fontSize:'2rem', fontWeight:'bold'}}>{users.length}</div>
                            <div style={{fontSize:'0.9rem', color:'#333'}}>
                              {roles.length === 0 ? (
                                'No roles found.'
                              ) : (
                                (() => {
                                  // Default roles to always show
                                  const defaultRoles = ['interpreter', 'student', 'default', 'bpo', 'translator'];
                                  // Map roles to counts
                                  const roleCounts = roles.map(roleName => {
                                    const count = users.filter(u => (u.role || '').toLowerCase() === roleName.toLowerCase()).length;
                                    return { roleName, count };
                                  });
                                  // Inline: all defaultRoles (in order), then the rest in tooltip
                                  const inlineRoles = roleCounts.filter(({roleName}) =>
                                    defaultRoles.includes(roleName.toLowerCase())
                                  );
                                  const hiddenRoles = roleCounts.filter(({roleName}) =>
                                    !defaultRoles.includes(roleName.toLowerCase())
                                  );
                                  return (
                                    <>
                                      {inlineRoles.map(({roleName, count}) => (
                                        <span key={roleName} style={{marginRight:8}}>
                                          {count} {roleName}{count !== 1 ? 's' : ''}
                                        </span>
                                      ))}
                                      {hiddenRoles.length > 0 && (
                                        <span
                                          style={{
                                            marginLeft: 4,
                                            color: accentColor,
                                            cursor: 'pointer',
                                            textDecoration: 'underline',
                                            position: 'relative',
                                          }}
                                          onMouseEnter={e => {
                                            const tooltip = document.getElementById('role-tooltip');
                                            if (tooltip) {
                                              tooltip.style.display = 'block';
                                              const rect = e.target.getBoundingClientRect();
                                              tooltip.style.left = rect.left + 'px';
                                              tooltip.style.top = (rect.bottom + 6) + 'px';
                                            }
                                          }}
                                          onMouseLeave={() => {
                                            const tooltip = document.getElementById('role-tooltip');
                                            if (tooltip) tooltip.style.display = 'none';
                                          }}
                                        >
                                          +{hiddenRoles.length} more
                                          <span
                                            id="role-tooltip"
                                            style={{
                                              display: 'none',
                                              position: 'fixed',
                                              background: '#fff',
                                              color: '#222',
                                              border: '1px solid #e0e0e0',
                                              borderRadius: 8,
                                              boxShadow: '0 2px 12px rgba(0,0,0,0.12)',
                                              padding: '12px 16px',
                                              zIndex: 9999,
                                              minWidth: 180,
                                              fontSize: '0.97rem',
                                            }}
                                          >
                                            {hiddenRoles.map(({roleName, count}) => (
                                              <div key={roleName} style={{marginBottom: 4}}>
                                                {count} {roleName}{count !== 1 ? 's' : ''}
                                              </div>
                                            ))}
                                          </span>
                                        </span>
                                      )}
                                    </>
                                  );
                                })()
                              )}
                            </div>
                          </>
                        )}
                      </div>
                      <div style={cardStyle}>
                        <div style={{fontWeight:'bold', color:accentColor}}>Enrolled</div>
                        <div style={{fontSize:'2rem', fontWeight:'bold'}}>1</div>
                        <div style={{fontSize:'0.9rem', color:'#333'}}>Online Courses</div>
                      </div>
                      <div style={cardStyle}>
                        <div style={{fontWeight:'bold', color:accentColor}}>Earning</div>
                        <div style={{fontSize:'2rem', fontWeight:'bold'}}>0.00</div>
                        <div style={{color:'#27ae60'}}>0% increase</div>
                      </div>
                    </div>
                  )}
                </div>
              ))}
            </div>
            <InterpreterChart />
          </>
        )}
      </main>

      {/* Right Sidebar: Recent Activities & News */}
    </div> {/* close responsiveContainerStyle div before sidebar-right */}
    <aside className="sidebar-right">
      <style>{`
        .sidebar-right {
          width: 320px;
          max-width: 90vw;
          min-width: 220px;
          background: linear-gradient(135deg, #eaf6ff 80%, #fff 100%);
          box-shadow: 0 4px 24px rgba(0,0,0,0.10);
          border-radius: 24px 0 0 24px;
          padding: 32px 18px;
          position: fixed;
          top: 64px;
          right: 0;
          height: calc(100vh - 64px);
          overflow-y: auto;
          display: flex;
          flex-direction: column;
          align-items: flex-start;
          border-left: 3px solid #e0e0e0;
          z-index: 9998;
          box-sizing: border-box;
          pointer-events: auto;
        }
        @media (max-width: 900px) {
          .sidebar-right {
            position: static !important;
            width: 100vw !important;
            min-width: 0 !important;
            max-width: 100vw !important;
            border-radius: 0 !important;
            box-shadow: none !important;
            padding: 12px !important;
            height: auto !important;
            margin-bottom: 24px !important;
          }
          .main-content {
            margin-left: 0 !important;
            margin-right: 0 !important;
            width: 100vw !important;
            padding: 12px !important;
          }
          div[style*='display: flex'] {
            flex-direction: column !important;
          }
        }
      `}</style>
        <h3 className="mb-3 fw-bold text-primary" style={{fontSize:'1.15rem'}}>Recent Activities</h3>
        <div className="mb-4">
          {actionsLoading ? (
            <div className="alert alert-info">Loading recent actions...</div>
          ) : actionsError ? (
            <div className="alert alert-danger">{actionsError}</div>
          ) : recentActions.length === 0 ? (
            <div className="text-muted">No recent actions found.</div>
          ) : (
            <ul className="list-group list-group-flush" style={{maxHeight: 350, overflowY: 'auto'}}>
              {recentActions.map(action => {
                // Determine if the action was made by the current user (Administrator)
                const currentUserId = localStorage.getItem('userId');
                const isMe = action.actor_id && String(action.actor_id) === String(currentUserId);
                // Replace 'User' with 'I' or 'You' in action_description if it's the current user
                let description = action.action_description;
                if (isMe && description) {
                  description = description.replace(/\bUser\b/, 'I').replace(/\buser\b/, 'I').replace(/\bYou\b/, 'I').replace(/\byou\b/, 'I');
                  // Optionally, for more natural language, you could use 'You' instead of 'I' if preferred
                }
                return (
                  <li key={action.id} className="list-group-item d-flex align-items-center gap-2" style={{fontSize:'1rem'}}>
                    <span className="me-2">
                      <i className={`bi bi-${action.action_type === 'login' ? 'box-arrow-in-right text-success' : action.action_type === 'edit_interpreter_chart' ? 'pencil-square text-warning' : action.action_type === 'print_interpreter_chart' ? 'printer text-primary' : 'activity text-secondary'}`}></i>
                    </span>
                    <div style={{flex:1}}>
                      <div className="fw-bold" style={{fontSize:'1rem'}}>{action.action_type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}</div>
                      <div className="text-muted" style={{fontSize:'0.95rem'}}>{description}</div>
                    </div>
                    <span className="badge bg-light text-dark ms-auto" style={{fontSize:'0.92rem', minWidth: 80, textAlign:'right'}}>{action.time_ago}</span>
                  </li>
                );
              })}
            </ul>
          )}
        </div>
        {/* News/blog section */}
        <div className="mt-4">
          <h5 className="fw-bold mb-2 text-primary"><i className="bi bi-newspaper me-2"></i>Latest News & Blog</h5>
          {newsBlogLoading ? (
            <div className="text-muted">Loading news and blog posts...</div>
          ) : newsBlogError ? (
            <div className="text-danger">{newsBlogError}</div>
          ) : newsBlog.length === 0 ? (
            <div className="text-muted">Coming soon: news and blog posts will appear here.</div>
          ) : (
            <div style={{maxHeight: 320, overflowY: 'auto', paddingRight: 2, width: '100%', display: 'flex', flexDirection: 'column', gap: '12px'}}>
              {newsBlog.slice(0, 5).map(item => {
                // Helper to strip HTML tags for clean text
                const stripHtml = (html) => {
                  if (!html) return '';
                  const div = document.createElement('div');
                  div.innerHTML = html;
                  return div.textContent || div.innerText || '';
                };
                const title = stripHtml(item.title);
                const summary = stripHtml(item.summary || item.content || '');
                const date = item.date || item.created_at || '';
                // Use item.image or item.picture as the image filename
                const imageName = item.image || item.picture || '';
                // Use backend public images path
                const imageUrl = imageName ? `${BASE_API}/public/images/news/${imageName}` : '/images/default-news.png';
                return (
                  <div key={item.id} className="card border-0 shadow-sm" style={{
                    borderRadius: '12px',
                    background: '#fafdff',
                    minWidth: 0,
                    width: '100%',
                    maxWidth: '100%',
                    margin: 0,
                    boxSizing: 'border-box',
                    display: 'flex',
                    flexDirection: 'row',
                    alignItems: 'center',
                    gap: '10px',
                  }}>
                    <img
                      src={imageUrl}
                      alt="news"
                      style={{
                        width: 44,
                        height: 44,
                        objectFit: 'cover',
                        borderRadius: '8px',
                        background: '#e0e0e0',
                        flexShrink: 0,
                        marginLeft: 8,
                        marginRight: 0
                      }}
                      onError={e => {
                        if (!e.target.src.endsWith('/images/default-news.png')) {
                          e.target.onerror = null;
                          e.target.src = '/images/default-news.png';
                        }
                      }}
                    />
                    <div className="card-body p-2 d-flex flex-column" style={{minWidth:0, width:'100%'}}>
                      <div className="fw-bold text-primary mb-1" style={{fontSize:'1.04rem', wordBreak:'break-word', lineHeight:'1.2', width:'100%'}} title={title}>{title}</div>
                      {/* Optionally add summary here if you want */}
                      <div className="text-muted mt-1" style={{fontSize:'0.93rem', textAlign:'right', width:'100%'}}>{date}</div>
                    </div>
                  </div>
                );
              })}
            </div>
          )}
        </div>
      </aside>
  </>
  );
};

export default AdminIndex;
