import React, { useState, useEffect } from 'react';
import { BASE_API } from '../../utils/config';

const SCHOOLS_URL = `${BASE_API}/api/schools`;
const LANGUAGES_URL = `${BASE_API}/api/languages`;
const API_URL = `${BASE_API}/api/courses`;

const getSessionEmail = () => {
  // Replace with your actual session logic
  return localStorage.getItem('user_email') || '';
};

const CourseManager = ({ onSelectCourse }) => {
  const [courses, setCourses] = useState([]);
  const [schoolTypes, setSchoolTypes] = useState([]);
  const [languages, setLanguages] = useState([]);
  const [showModal, setShowModal] = useState(false);
  const [editModal, setEditModal] = useState(false);
  const [editCourse, setEditCourse] = useState(null);
  const [form, setForm] = useState({
    type: '',
    course_name: '',
    description: '',
    course_language: '',
    image_id: '',
  });
  const [images, setImages] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  useEffect(() => {
    fetchCourses();
    fetchSchoolTypes();
    // Fetch languages from WorldWise API (using restcountries)
    fetch('https://restcountries.com/v3.1/all?fields=languages')
      .then(res => res.json())
      .then(data => {
        const langSet = new Set();
        if (Array.isArray(data)) {
          data.forEach(country => {
            if (country.languages) {
              Object.values(country.languages).forEach(lang => langSet.add(lang));
            }
          });
          setLanguages(Array.from(langSet).sort());
        } else {
          setLanguages([]);
        }
      })
      .catch(() => setLanguages([]));
    // Fetch images from backend
    fetch(`${BASE_API}/api/images`)
      .then(res => res.json())
      .then(data => setImages(data.images || []));
  }, []);

  const fetchCourses = async () => {
    try {
      const res = await fetch(`${API_URL}`);
      const data = await res.json();
      setCourses(data.courses || []);
    } catch (err) {
      setError('Failed to fetch courses');
    }
  };

  const fetchSchoolTypes = async () => {
    try {
      const res = await fetch(`${SCHOOLS_URL}`);
      const data = await res.json();
      setSchoolTypes(data.schools || []);
    } catch (err) {
      setSchoolTypes([]);
    }
  };

  // Removed fetchLanguages, now using WorldWise API in useEffect

  const handleInputChange = (e) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleAddCourse = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError('');
    const payload = {
      ...form,
      create_by: getSessionEmail(),
    };
    try {
      const res = await fetch(`${API_URL}/add`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(payload)
      });
      const data = await res.json();
      if (res.ok) {
        setShowModal(false);
        setForm({ type: '', course_name: '', description: '', course_language: '' });
        fetchCourses();
      } else {
        setError(data.error || 'Failed to add course');
      }
    } catch (err) {
      setError('Network error');
    }
    setLoading(false);
  };

  const handleEditClick = (course) => {
    setEditCourse({ ...course, image_id: course.image_id || '' });
    setEditModal(true);
  };

  const handleEditInputChange = (e) => {
    setEditCourse({ ...editCourse, [e.target.name]: e.target.value });
  };

  const handleEditCourse = async (e) => {
    e.preventDefault();
    setLoading(true);
    setError('');
    try {
      const res = await fetch(`${API_URL}/edit/${editCourse.id}`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(editCourse)
      });
      const data = await res.json();
      if (res.ok) {
        setEditModal(false);
        fetchCourses();
      } else {
        setError(data.error || 'Failed to edit course');
      }
    } catch (err) {
      setError('Network error');
    }
    setLoading(false);
  };

  return (
    <div style={{padding:'20px'}}>
      <h2>Course Manager</h2>
      {/* Course selection dropdown for AddCourseContent */}
      {onSelectCourse && (
        <div style={{marginBottom:'16px'}}>
          <label style={{fontWeight:'bold', marginRight:'8px'}}>Select Course for Content:</label>
          <select onChange={e => onSelectCourse(e.target.value)} style={{padding:'8px', borderRadius:'6px', minWidth:'220px'}}>
            <option value="">Select a course</option>
            {courses.map(course => (
              <option key={course.id} value={course.id}>{course.course_name}</option>
            ))}
          </select>
        </div>
      )}
      <button onClick={() => setShowModal(true)} style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer', marginBottom:'20px'}}>Add Course</button>
      {error && <div style={{color:'red', marginBottom:'12px'}}>{error}</div>}
      <div style={{display:'grid', gridTemplateColumns:'repeat(auto-fill, minmax(300px, 1fr))', gap:'20px'}}>
        {courses.map(course => (
          <div
            key={course.id}
            style={{
              background:'#fff',
              borderRadius:'22px',
              boxShadow:'0 8px 32px rgba(30, 60, 90, 0.13)',
              padding:'0',
              marginBottom:'32px',
              maxWidth:'420px',
              fontFamily:'Segoe UI, Arial, sans-serif',
              border:'none',
              position:'relative',
              overflow:'hidden',
              transition:'box-shadow 0.3s, transform 0.2s',
              cursor:'pointer',
            }}
            onMouseEnter={e => e.currentTarget.style.boxShadow = '0 16px 48px rgba(30, 60, 90, 0.18)'}
            onMouseLeave={e => e.currentTarget.style.boxShadow = '0 8px 32px rgba(30, 60, 90, 0.13)'}
          >
            <div style={{height:'8px', width:'100%', background:'linear-gradient(90deg, #1976d2 0%, #27ae60 100%)'}} />
            {course.image_url && (
              <div style={{display:'flex', justifyContent:'center', alignItems:'center', gap:'12px', margin:'24px 0 18px 0'}}>
                <img src={`${BASE_API}${course.image_url}`} alt="Course Cover" style={{height:'180px', width:'320px', objectFit:'cover', borderRadius:'18px', boxShadow:'0 2px 12px rgba(0,0,0,0.10)', border:'3px solid #f7f8fa'}} />
              </div>
            )}
            <div style={{padding:'0 28px 24px 28px'}}>
              <h3 style={{fontSize:'1.5rem', fontWeight:'800', color:'#1976d2', marginBottom:'12px', letterSpacing:'0.5px', textShadow:'0 2px 8px rgba(25,118,210,0.08)'}}>{course.course_name}</h3>
              <div style={{marginBottom:'10px'}}>
                <span style={{fontWeight:'600', color:'#222'}}>Type:</span> <span style={{color:'#444'}}>{course.type}</span>
              </div>
              <div style={{marginBottom:'10px'}}>
                <span style={{fontWeight:'600', color:'#222'}}>Language:</span> <span style={{color:'#444'}}>{course.course_language}</span>
              </div>
              <div style={{marginBottom:'10px'}}>
                <span style={{fontWeight:'600', color:'#222'}}>Description:</span> <span style={{color:'#444'}}>{course.description}</span>
              </div>
              <div style={{marginBottom:'22px'}}>
                <span style={{fontWeight:'600', color:'#222'}}>Created By:</span> <span style={{color:'#888'}}>{course.create_by}</span>
              </div>
              <button onClick={() => handleEditClick(course)} style={{
                background:'linear-gradient(90deg, #27ae60 60%, #1976d2 100%)',
                color:'#fff',
                border:'none',
                borderRadius:'10px',
                padding:'12px 32px',
                fontWeight:'bold',
                fontSize:'1.08rem',
                cursor:'pointer',
                boxShadow:'0 2px 8px rgba(39,174,96,0.10)',
                transition:'background 0.2s, box-shadow 0.2s',
                marginTop:'4px',
                marginBottom:'0',
                display:'block',
                width:'fit-content',
              }}>Edit</button>
            </div>
          </div>
        ))}
      </div>

      {showModal && (
        <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.2)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}}>
          <form onSubmit={handleAddCourse} style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'320px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}}>
            <h3>Add Course</h3>
            <label>Type:</label>
            <select name="type" value={form.type} onChange={handleInputChange} required style={{width:'100%', marginBottom:'10px'}}>
              <option value="">Select type</option>
              {schoolTypes.map(s => (
                <option key={s.id} value={s.type}>{s.type} ({s.name})</option>
              ))}
            </select>
            <label>Language:</label>
            <select name="course_language" value={form.course_language} onChange={handleInputChange} required style={{width:'100%', marginBottom:'10px'}}>
              <option value="">Select language</option>
              {languages.map(lang => (
                <option key={lang} value={lang}>{lang}</option>
              ))}
            </select>
            <label>Course Name:</label>
            <input name="course_name" value={form.course_name} onChange={handleInputChange} required style={{width:'100%', marginBottom:'10px'}} />
            <label>Description:</label>
            <textarea name="description" value={form.description} onChange={handleInputChange} required style={{width:'100%', marginBottom:'10px'}} />
            <label>Course Image:</label>
            <select name="image_id" value={form.image_id} onChange={handleInputChange} style={{width:'100%', marginBottom:'8px'}}>
              <option value="">Select an image</option>
              {images.map(img => (
                <option key={img.id} value={img.id}>{img.image_name}</option>
              ))}
            </select>
            {form.image_id && (
              <img src={images.find(img => img.id === Number(form.image_id))?.image_url} alt="Course" style={{maxWidth:'120px', maxHeight:'80px', borderRadius:'8px', marginTop:'8px'}} />
            )}
            {/* Created By is hidden and set automatically */}
            {error && <div style={{color:'red', marginBottom:'12px'}}>{error}</div>}
            <button type="submit" style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} disabled={loading}>{loading ? 'Adding...' : 'Add Course'}</button>
            <button type="button" onClick={() => setShowModal(false)} style={{marginLeft:'12px', background:'#ccc', color:'#222', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Cancel</button>
          </form>
        </div>
      )}

      {editModal && editCourse && (
        <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.2)', zIndex:999, display:'flex', alignItems:'center', justifyContent:'center'}}>
          <form onSubmit={handleEditCourse} style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:'320px', boxShadow:'0 2px 12px rgba(0,0,0,0.12)'}}>
            <h3>Edit Course</h3>
            <label>Type:</label>
            <select name="type" value={editCourse.type} onChange={handleEditInputChange} required style={{width:'100%', marginBottom:'10px'}}>
              <option value="">Select type</option>
              {schoolTypes.map(s => (
                <option key={s.id} value={s.type}>{s.type} ({s.name})</option>
              ))}
            </select>
            <label>Language:</label>
            <select name="course_language" value={editCourse.course_language} onChange={handleEditInputChange} required style={{width:'100%', marginBottom:'10px'}}>
              <option value="">Select language</option>
              {languages.map(lang => (
                <option key={lang} value={lang}>{lang}</option>
              ))}
            </select>
            <label>Course Name:</label>
            <input name="course_name" value={editCourse.course_name} onChange={handleEditInputChange} required style={{width:'100%', marginBottom:'10px'}} />
            <label>Description:</label>
            <textarea name="description" value={editCourse.description} onChange={handleEditInputChange} required style={{width:'100%', marginBottom:'10px'}} />
            <label>Course Image:</label>
            <select name="image_id" value={editCourse.image_id} onChange={handleEditInputChange} style={{width:'100%', marginBottom:'8px'}}>
              <option value="">Select an image</option>
              {images.map(img => (
                <option key={img.id} value={img.id}>{img.image_name}</option>
              ))}
            </select>
            {editCourse.image_id && (
              <img src={images.find(img => img.id === Number(editCourse.image_id))?.image_url} alt="Course" style={{maxWidth:'120px', maxHeight:'80px', borderRadius:'8px', marginTop:'8px'}} />
            )}
            {/* Created By is hidden and set automatically */}
            {error && <div style={{color:'red', marginBottom:'12px'}}>{error}</div>}
            <button type="submit" style={{background:'#27ae60', color:'#fff', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}} disabled={loading}>{loading ? 'Saving...' : 'Save Changes'}</button>
            <button type="button" onClick={() => { setEditModal(false); setEditCourse(null); }} style={{marginLeft:'12px', background:'#ccc', color:'#222', border:'none', borderRadius:'6px', padding:'10px 24px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Cancel</button>
          </form>
        </div>
      )}
    </div>
  );
};

export default CourseManager;

