import React, { useState, useEffect } from 'react';
import ChapterQuiz from './ChapterQuiz';
import axios from 'axios';

const CourseCard = ({ course, onSelect, enrolledCount }) => (
  <div
    className="course-card"
    style={{
      background: '#fff',
      borderRadius: 12,
      boxShadow: '0 2px 12px rgba(0,0,0,0.08)',
      padding: 20,
      margin: '12px 0',
      display: 'flex',
      alignItems: 'center',
      cursor: 'pointer',
      transition: 'box-shadow 0.2s',
      minWidth: 260,
      maxWidth: 400,
      width: '100%',
    }}
    onClick={() => onSelect(course)}
  >
    <img
      src={course.image_url || (process.env.PUBLIC_URL + '/images/default_course.png')}
      alt={course.course_name}
      style={{ width: 80, height: 80, objectFit: 'cover', borderRadius: 8, marginRight: 18, background: '#eee' }}
    />
    <div style={{ flex: 1 }}>
      <h3 style={{ margin: 0, color: '#1976d2', fontSize: 20 }}>{course.course_name}</h3>
      <p style={{ margin: '8px 0', color: '#555', fontSize: 15 }}>{course.description}</p>
      <span style={{ fontSize: 13, color: '#888' }}>Language: {course.course_language}</span>
      <div style={{ fontSize: 13, color: '#1976d2', marginTop: 6 }}>
        <strong>Enrolled:</strong> {enrolledCount !== undefined ? enrolledCount : '...'}
      </div>
    </div>
  </div>
);


const CourseReader = ({ course, onBack, userId }) => {
  // All useState declarations at the top
  const [showQuizSummary, setShowQuizSummary] = useState(false);
  const [quizSummaryData, setQuizSummaryData] = useState([]);
  const [modules, setModules] = useState([]);
  const [selectedModuleId, setSelectedModuleId] = useState('');
  const [chapters, setChapters] = useState([]);
  const [selectedChapterIdx, setSelectedChapterIdx] = useState(null);
  const [completedChapters, setCompletedChapters] = useState([]);
  const [quizResult, setQuizResult] = useState(null);
  const [showQuizPopup, setShowQuizPopup] = useState(false);
  const [quizStarted, setQuizStarted] = useState(false);
  const [quizQuestions, setQuizQuestions] = useState([]);
  const [quizId, setQuizId] = useState(null);
  const [quizSettings, setQuizSettings] = useState(null);
  // Progress bar calculation
  const progress = chapters.length > 0 ? ((completedChapters.length / chapters.length) * 100).toFixed(0) : 0;

  // Fetch all quiz results for the course when summary modal is opened
  useEffect(() => {
    if (showQuizSummary && selectedModuleId && chapters.length > 0) {
      const promises = chapters.map(ch =>
        ch.quiz_id ? axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${ch.quiz_id}`) : Promise.resolve({ data: {} })
      );
      Promise.all(promises).then(results => {
        setQuizSummaryData(results.map((res, idx) => ({
          chapter: chapters[idx]?.name || `Chapter ${idx + 1}`,
          score: res.data?.score ?? null,
          attempts: res.data?.attempts ?? null,
          is_passed: res.data?.is_passed ?? 0
        })));
      });
    }
  }, [showQuizSummary, selectedModuleId, chapters, userId]);

  // Poll quiz_result for all chapters to update completedChapters and progress
  useEffect(() => {
    let pollInterval;
    const pollQuizResults = async () => {
      if (selectedModuleId && chapters.length > 0) {
        // Fetch quiz results for all chapters in this module
        const promises = chapters.map(ch =>
          ch.quiz_id ? axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${ch.quiz_id}`) : Promise.resolve({ data: {} })
        );
        const results = await Promise.all(promises);
        const completed = results
          .map((res, idx) => res.data && res.data.is_passed === 1 ? idx : null)
          .filter(idx => idx !== null);
        setCompletedChapters(completed);
      } else {
        setCompletedChapters([]);
      }
    };
    pollQuizResults();
    pollInterval = setInterval(pollQuizResults, 2000);
    return () => clearInterval(pollInterval);
  }, [selectedModuleId, chapters, userId]);

  // Fetch quiz result for current user/chapter and poll for updates
  useEffect(() => {
    let pollInterval;
    const fetchQuizResult = () => {
      if (selectedChapterIdx !== null && chapters[selectedChapterIdx] && chapters[selectedChapterIdx].quiz_id) {
        const currentQuizId = chapters[selectedChapterIdx].quiz_id;
        setQuizId(currentQuizId);
        axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${currentQuizId}`)
          .then(res => setQuizResult(res.data));
      } else {
        setQuizResult(null);
        setQuizId(null);
      }
    };
    fetchQuizResult();
    pollInterval = setInterval(fetchQuizResult, 2000); // Poll every 2 seconds
    return () => clearInterval(pollInterval);
  }, [selectedChapterIdx, chapters, userId]);

  // Save quiz progress on page unload
  useEffect(() => {
    const handleUnload = () => {
      if (quizResult && selectedChapterIdx !== null && chapters[selectedChapterIdx]) {
        axios.post('/api/quiz_result', {
          user_id: userId,
          quiz_id: chapters[selectedChapterIdx].quiz_id,
          score: quizResult.score,
          attempts: quizResult.attempts
        });
      }
    };
    window.addEventListener('beforeunload', handleUnload);
    return () => window.removeEventListener('beforeunload', handleUnload);
  }, [quizResult, selectedChapterIdx, chapters, userId]);

  // Mark chapter as completed when quiz is passed
  const handleQuizComplete = (score, total) => {
    if (score >= 80 && selectedChapterIdx !== null) {
      if (!completedChapters.includes(selectedChapterIdx)) {
        setCompletedChapters([...completedChapters, selectedChapterIdx]);
      }
      // Save result as passed
      if (chapters[selectedChapterIdx]) {
        axios.post('/api/quiz_result', {
          user_id: userId,
          quiz_id: chapters[selectedChapterIdx].quiz_id,
          score,
          attempts: quizResult ? quizResult.attempts + 1 : 1
        }).then(() => {
          axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${chapters[selectedChapterIdx].quiz_id}`)
            .then(res => setQuizResult(res.data));
        });
      }
    }
  };

  // Bookmark progress in localStorage
  useEffect(() => {
    const bookmarkKey = `course_${course.course_id}_bookmark`;
    const saved = localStorage.getItem(bookmarkKey);
    if (saved) {
      try {
        const { moduleId, chapterIdx } = JSON.parse(saved);
        setSelectedModuleId(moduleId);
        setSelectedChapterIdx(chapterIdx);
      } catch {}
    }
  }, [course.course_id]);

  useEffect(() => {
    axios.get(`/api/modules/${course.course_id}`).then(res => {
      setModules(res.data.modules || []);
    });
  }, [course.course_id]);

  useEffect(() => {
    if (selectedModuleId) {
      axios.get(`/api/chapters/${selectedModuleId}`).then(res => {
        setChapters(res.data.chapters || []);
      });
    } else {
      setChapters([]);
      setSelectedChapterIdx(null);
    }
  }, [selectedModuleId]);

  // Save bookmark on chapter change
  useEffect(() => {
    if (selectedModuleId && selectedChapterIdx !== null) {
      const bookmarkKey = `course_${course.course_id}_bookmark`;
      localStorage.setItem(bookmarkKey, JSON.stringify({ moduleId: selectedModuleId, chapterIdx: selectedChapterIdx }));
    }
  }, [selectedModuleId, selectedChapterIdx, course.course_id]);

  // Navigation handlers
  const goToNextChapter = () => {
    if (
      selectedChapterIdx !== null &&
      selectedChapterIdx < chapters.length - 1 &&
      quizResult && quizResult.is_passed === 1
    ) {
      setSelectedChapterIdx(selectedChapterIdx + 1);
    }
  };
  const goToPrevChapter = () => {
    if (selectedChapterIdx !== null && selectedChapterIdx > 0) {
      setSelectedChapterIdx(selectedChapterIdx - 1);
    }
  };

  // Load quiz questions for the current chapter
  useEffect(() => {
    if (showQuizPopup && chapters[selectedChapterIdx] && quizId) {
      const chapterId = chapters[selectedChapterIdx].id;
      // Fetch quiz settings
      axios.get(`/api/quizzes/${quizId}`)
        .then(res => {
          setQuizSettings(res.data.quiz || null);
          // Fetch questions after settings
          axios.get(`/api/quiz_questions?quiz_id=${quizId}&chapter_id=${chapterId}`)
            .then(res2 => {
              let questions = (res2.data.questions || []).map(q => {
                if (typeof q.options === 'string') {
                  q.options = q.options.split(',').map(opt => opt.trim());
                }
                return q;
              });
              // Only show the number of questions specified in quiz settings
              if (res.data.quiz && res.data.quiz.num_questions) {
                questions = questions.slice(0, res.data.quiz.num_questions);
              }
              setQuizQuestions(questions);
            });
        });
    }
  }, [showQuizPopup, chapters, selectedChapterIdx]);

  return (
    <div className="course-reader" style={{ padding: 24, maxWidth: 900, margin: '0 auto' }}>
      {/* Top Section: Back, Title, Image, Description, Language */}
      <div style={{ display: 'flex', alignItems: 'flex-start', gap: 32, marginBottom: 32 }}>
        <div style={{ flex: '0 0 auto' }}>
          <button onClick={onBack} style={{ marginBottom: 18, padding: '8px 18px', background: '#1976d2', color: '#fff', border: 'none', borderRadius: 6, fontWeight: 'bold', cursor: 'pointer' }}>Back to Courses</button>
          <h2 style={{ color: '#1976d2', marginBottom: 12 }}>{course.course_name}</h2>
          <p style={{ fontSize: 16, color: '#444', marginBottom: 18 }}>{course.description}</p>
          <div style={{ fontSize: 15, color: '#888', marginBottom: 18 }}>Course Language: {course.course_language}</div>
        </div>
        <img
          src={course.image_url || (process.env.PUBLIC_URL + '/images/default_course.png')}
          alt={course.course_name}
          style={{ width: 180, height: 180, objectFit: 'cover', borderRadius: 12, background: '#eee', boxShadow: '0 2px 12px rgba(0,0,0,0.08)' }}
        />
      </div>
      {/* Modules List */}
      <div style={{ marginBottom: 18 }}>
        <h3 style={{ color: '#1565c0', marginBottom: 8 }}>Modules</h3>
        <div style={{ display: 'flex', flexWrap: 'wrap', gap: 12 }}>
          {modules.length === 0 ? (
            <div>No modules available.</div>
          ) : (
            modules.map((m, idx) => {
              // Only enable the first module or if all chapters are completed
              const isEnabled = idx === 0 || completedChapters.length === chapters.length;
              return (
                <button
                  key={m.id}
                  onClick={() => {
                    if (isEnabled) {
                      setSelectedModuleId(m.id);
                      setSelectedChapterIdx(null);
                    }
                  }}
                  disabled={!isEnabled}
                  style={{
                    padding: '10px 18px',
                    background: selectedModuleId === m.id ? '#1976d2' : '#e3f2fd',
                    color: selectedModuleId === m.id ? '#fff' : '#1976d2',
                    border: 'none',
                    borderRadius: 8,
                    fontWeight: 'bold',
                    fontSize: 15,
                    cursor: isEnabled ? 'pointer' : 'not-allowed',
                    marginBottom: 6,
                    minWidth: 120,
                    transition: 'background 0.2s',
                    opacity: isEnabled ? 1 : 0.5,
                  }}
                >
                  {m.name}
                </button>
              );
            })
          )}
        </div>
      </div>
      {/* Progress Bar & Chapters Progress - Horizontal */}
      {selectedModuleId && (
        <div style={{ marginBottom: 18 }}>
          <h3 style={{ color: '#1565c0', marginBottom: 8 }}>Chapters</h3>
          {/* Quiz Summary Chart (clickable) and Progress Bar neatly arranged */}
          <div style={{ display: 'flex', alignItems: 'center', gap: 32, marginBottom: 18 }}>
            <div
              style={{
                width: 120,
                height: 120,
                borderRadius: '50%',
                background: 'linear-gradient(90deg, #43cea2 0%, #1976d2 100%)',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                cursor: 'pointer',
                boxShadow: '0 2px 12px rgba(67,206,162,0.12)'
              }}
              title="View Quiz Summary"
              onClick={() => setShowQuizSummary(true)}
            >
              <div style={{ color: '#fff', fontWeight: 'bold', fontSize: 22, textAlign: 'center' }}>
                {completedChapters.length} / {chapters.length}<br />Completed
              </div>
            </div>
            <div style={{ flex: 1, minWidth: 200 }}>
              <div style={{ width: '100%', background: '#e0e0e0', borderRadius: 8, height: 16, marginBottom: 8 }}>
                <div style={{ width: `${progress}%`, background: '#1976d2', height: '100%', borderRadius: 8, transition: 'width 0.3s' }} />
              </div>
              <div className="chapter-progress" style={{ fontSize: 15, color: '#1976d2', marginBottom: 0 }}>
                {completedChapters.length} of {chapters.length} chapters completed
              </div>
            </div>
          </div>
      {/* Quiz Summary Modal */}
      {showQuizSummary && (
        <div style={{
          position: 'fixed',
          top: 0,
          left: 0,
          width: '100vw',
          height: '100vh',
          background: 'rgba(0,0,0,0.5)',
          zIndex: 9999,
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
        }}>
          <div style={{
            background: '#fff',
            borderRadius: 16,
            padding: 32,
            minWidth: 320,
            maxWidth: 480,
            boxShadow: '0 4px 24px rgba(25,118,210,0.12)',
            position: 'relative',
          }}>
            <h3 style={{ color: '#1976d2', marginBottom: 18 }}>Quiz Summary</h3>
            <table style={{ width: '100%', borderCollapse: 'collapse', marginBottom: 18 }}>
              <thead>
                <tr style={{ background: '#e3f2fd', color: '#1976d2' }}>
                  <th style={{ padding: '8px 6px', borderRadius: 4 }}>Chapter</th>
                  <th style={{ padding: '8px 6px', borderRadius: 4 }}>Score</th>
                  <th style={{ padding: '8px 6px', borderRadius: 4 }}>Attempts</th>
                  <th style={{ padding: '8px 6px', borderRadius: 4 }}>Status</th>
                </tr>
              </thead>
              <tbody>
                {quizSummaryData.map((row, idx) => (
                  <tr key={idx} style={{ background: idx % 2 === 0 ? '#f9f9f9' : '#fff' }}>
                    <td style={{ padding: '8px 6px' }}>{row.chapter}</td>
                    <td style={{ padding: '8px 6px' }}>{row.score !== null ? row.score : '-'}</td>
                    <td style={{ padding: '8px 6px' }}>{row.attempts !== null ? row.attempts : '-'}</td>
                    <td style={{ padding: '8px 6px', color: row.is_passed === 1 ? '#27ae60' : '#d32f2f', fontWeight: 'bold' }}>
                      {row.is_passed === 1 ? 'Passed' : 'Incomplete'}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
            <button
              onClick={() => setShowQuizSummary(false)}
              style={{
                padding: '10px 28px',
                background: '#1976d2',
                color: '#fff',
                border: 'none',
                borderRadius: 8,
                fontWeight: 'bold',
                fontSize: 16,
                cursor: 'pointer',
                position: 'absolute',
                right: 24,
                bottom: 24,
              }}
            >
              Close
            </button>
          </div>
        </div>
      )}
          <div style={{ display: 'flex', flexDirection: 'row', gap: 12, overflowX: 'auto', paddingBottom: 8 }}>
            {chapters.length === 0 ? (
              <div>No chapters available.</div>
            ) : (
                chapters.map((c, idx) => (
                  <button
                    key={c.id}
                    onClick={() => {
                      // Always allow the first chapter to be selected, or if completed/current
                      if (idx === 0 || selectedChapterIdx === idx || completedChapters.includes(idx)) {
                        setSelectedChapterIdx(idx);
                      }
                    }}
                    disabled={!(idx === 0 || selectedChapterIdx === idx || completedChapters.includes(idx))}
                    style={{
                      minWidth: 80,
                      height: 54,
                      borderRadius: 28,
                      background: selectedChapterIdx === idx
                        ? 'linear-gradient(90deg, #43cea2 0%, #185a9d 100%)'
                        : completedChapters.includes(idx)
                          ? 'linear-gradient(90deg, #27ae60 0%, #a259e6 100%)'
                          : 'rgba(255,255,255,0.45)',
                      color: selectedChapterIdx === idx || completedChapters.includes(idx) ? '#fff' : '#1976d2',
                      border: selectedChapterIdx === idx ? '2px solid #43cea2' : '1.5px solid #e0e0e0',
                      fontWeight: 'bold',
                      fontSize: 22,
                      cursor: (selectedChapterIdx === idx || completedChapters.includes(idx) || idx === 0) ? 'pointer' : 'not-allowed',
                      boxShadow: '0 4px 24px rgba(67,206,162,0.10)',
                      opacity: (selectedChapterIdx === idx || completedChapters.includes(idx) || idx === 0) ? 1 : 0.5,
                    }}
                  >
                    {idx + 1}
                  </button>
                ))
            )}
          </div>
          {/* Always show content for selected chapter, and allow navigation to previous/completed chapters */}
          {selectedChapterIdx !== null && chapters[selectedChapterIdx] && (
            <div style={{ background: '#f9f9f9', borderRadius: 8, padding: 16, boxShadow: '0 1px 6px rgba(0,0,0,0.04)', marginBottom: 18, position: 'relative', transition: 'background 0.3s' }}>
              <h4 style={{ margin: '0 0 8px 0', color: '#1976d2' }}>{chapters[selectedChapterIdx].name}</h4>
              <div style={{ fontSize: 15, color: '#444', marginBottom: 8 }} dangerouslySetInnerHTML={{ __html: chapters[selectedChapterIdx].content }} />
              {chapters[selectedChapterIdx].insight && <div style={{ fontSize: 14, color: '#888', marginBottom: 8 }}><strong>Insight:</strong> <span dangerouslySetInnerHTML={{ __html: chapters[selectedChapterIdx].insight }} /></div>}
              <div style={{ fontSize: 13, color: '#888' }}>Created: {chapters[selectedChapterIdx].created_at}</div>
              {/* Navigation Buttons and Take Quiz */}
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginTop: 18 }}>
                <button
                  onClick={goToPrevChapter}
                  disabled={selectedChapterIdx === 0}
                  style={{
                    padding: '8px 18px',
                    background: selectedChapterIdx === 0 ? '#e0e0e0' : '#1976d2',
                    color: '#fff',
                    border: 'none',
                    borderRadius: 6,
                    fontWeight: 'bold',
                    cursor: selectedChapterIdx === 0 ? 'not-allowed' : 'pointer',
                    minWidth: 120,
                  }}
                >
                  Previous
                </button>
                <button
                  onClick={() => { setShowQuizPopup(true); setQuizStarted(false); }}
                  style={{
                    padding: '10px 28px',
                    background: 'linear-gradient(90deg, #43cea2 0%, #1976d2 100%)',
                    color: '#fff',
                    border: 'none',
                    borderRadius: 8,
                    fontWeight: 'bold',
                    fontSize: 17,
                    boxShadow: '0 2px 12px rgba(67,206,162,0.12)',
                    cursor: 'pointer',
                    minWidth: 140,
                    transition: 'background 0.2s',
                  }}
                >
                  <span role="img" aria-label="quiz" style={{marginRight:8}}>📝</span> Take Quiz
                </button>
                <button
                  onClick={goToNextChapter}
                  disabled={selectedChapterIdx === chapters.length - 1 || (quizResult && quizResult.is_passed !== 1)}
                  style={{
                    padding: '8px 18px',
                    background: selectedChapterIdx === chapters.length - 1 || (quizResult && quizResult.is_passed !== 1) ? '#e0e0e0' : '#1976d2',
                    color: '#fff',
                    border: 'none',
                    borderRadius: 6,
                    fontWeight: 'bold',
                    cursor: selectedChapterIdx === chapters.length - 1 || (quizResult && quizResult.is_passed !== 1) ? 'not-allowed' : 'pointer',
                    minWidth: 120,
                  }}
                >
                  Next
                </button>
              </div>
            </div>
          )}
        </div>
      )}
      {/* Responsive Quiz Popup */}
      {showQuizPopup && (
        <>
          {/* Overlay: completely black, and blocks all outside content */}
          <div style={{
            position: 'fixed',
            inset: 0,
            width: '100vw',
            height: '100vh',
            background: 'rgba(0,0,0,1)', // fully opaque
            zIndex: 9998,
            pointerEvents: 'none',
          }} />
          {/* Quiz Popup - always perfectly centered, visible, and above overlay */}
          <div style={{
            position: 'fixed',
            inset: 0,
            zIndex: 9999,
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            pointerEvents: 'auto',
          }}>
            <div style={{
              background: '#fff',
              borderRadius: 18,
              boxShadow: '0 8px 32px rgba(67,206,162,0.22)',
              border: '1.5px solid #e0e0e0',
              padding: '40px 28px',
              width: '100%',
              maxWidth: 480,
              maxHeight: '92vh',
              overflowY: 'auto',
              position: 'relative',
              animation: 'slideUp 0.3s',
              margin: '0 12px',
              display: 'flex',
              flexDirection: 'column',
              alignItems: 'stretch',
              justifyContent: 'flex-start',
            }}>
              {/* Start Assessment Button */}
              {!quizStarted ? (
                <>
                  {/* Quiz Settings Summary */}
                  {quizSettings && (
                    <div style={{
                      background: '#e3f2fd',
                      borderRadius: 10,
                      padding: '16px 18px',
                      marginBottom: 18,
                      fontSize: 15,
                      color: '#1976d2',
                      boxShadow: '0 1px 6px rgba(25,118,210,0.08)'
                    }}>
                      <div><strong>Duration:</strong> {quizSettings.duration_minutes} minutes</div>
                      <div><strong>Questions:</strong> {quizSettings.num_questions}</div>
                      <div><strong>Allowed Attempts:</strong> {quizSettings.allowed_attempts}</div>
                      <div><strong>Retry Wait:</strong> {quizSettings.retry_wait_hours} hours</div>
                      <div><strong>Passing Score:</strong> 80%</div>
                    </div>
                  )}
                  <button
                    onClick={() => setQuizStarted(true)}
                    style={{
                      padding: '12px 32px',
                      background: 'linear-gradient(90deg, #1976d2 0%, #43cea2 100%)',
                      color: '#fff',
                      border: 'none',
                      borderRadius: 8,
                      fontWeight: 'bold',
                      fontSize: 18,
                      boxShadow: '0 2px 12px rgba(67,206,162,0.12)',
                      cursor: 'pointer',
                      minWidth: 160,
                      marginBottom: 24,
                    }}
                  >
                    <span role="img" aria-label="start">🚀</span> Start Assessment
                  </button>
                </>
              ) : (
                <form style={{ width: '100%', maxWidth: 420 }}>
                  <div style={{ width: '100%' }}>
                    {quizQuestions && quizQuestions.length > 0 && quizSettings ? (
                      <ChapterQuiz
                        quiz={quizQuestions}
                        quizSettings={quizSettings}
                        userId={userId}
                        onComplete={score => {
                          handleQuizComplete(score, 100);
                          setQuizStarted(false);
                          setShowQuizPopup(true); // keep popup open for finish message
                        }}
                        onProgress={(score, answerData) => {
                          // Save progress and answer for every question
                          if (chapters[selectedChapterIdx]) {
                            axios.post('/api/quiz_result', {
                              user_id: userId,
                              quiz_id: chapters[selectedChapterIdx].quiz_id,
                              score,
                              attempts: quizResult ? quizResult.attempts + 1 : 1,
                              answer: answerData // expects answerData to be question/answer mapping
                            }).then(() => {
                              axios.get(`/api/quiz_result?user_id=${userId}&quiz_id=${chapters[selectedChapterIdx].quiz_id}`)
                                .then(res => setQuizResult(res.data));
                            });
                          }
                        }}
                        editable={true}
                        questionStyle={{
                          fontSize: 18,
                          fontWeight: 'bold',
                          marginBottom: 16,
                          color: '#1976d2',
                        }}
                        optionStyle={{
                          fontSize: 16,
                          marginBottom: 10,
                        }}
                        submitButtonStyle={{
                          padding: '12px 32px',
                          background: '#1976d2',
                          color: '#fff',
                          border: 'none',
                          borderRadius: 8,
                          fontWeight: 'bold',
                          fontSize: 17,
                          marginTop: 18,
                          cursor: 'pointer',
                        }}
                      />
                    ) : (
                      <>
                        <div style={{ color: '#d32f2f', fontWeight: 'bold', fontSize: 18, textAlign: 'center', marginTop: 24 }}>
                          No quiz questions available for this chapter.
                        </div>
                        <button
                          onClick={() => { setShowQuizPopup(false); setQuizStarted(false); }}
                          style={{
                            margin: '32px auto 0',
                            display: 'block',
                            padding: '10px 28px',
                            background: '#1976d2',
                            color: '#fff',
                            border: 'none',
                            borderRadius: 8,
                            fontWeight: 'bold',
                            fontSize: 16,
                            cursor: 'pointer',
                          }}
                        >
                          Close
                        </button>
                      </>
                    )}
                  </div>
                </form>
              )}
              {/* Add a top-level close button for the assessment popup/modal */}
              <button
                onClick={() => { setShowQuizPopup(false); setQuizStarted(false); }}
                style={{
                  position: 'absolute',
                  top: 18,
                  right: 18,
                  padding: '8px 18px',
                  background: '#888',
                  color: '#fff',
                  border: 'none',
                  borderRadius: 8,
                  fontWeight: 'bold',
                  fontSize: 15,
                  cursor: 'pointer',
                  zIndex: 10001
                }}
              >
                Close
              </button>
              <style>{`
                @keyframes slideUp { from { transform: translateY(40px); opacity: 0; } to { transform: translateY(0); opacity: 1; } }
                @media (max-width: 600px) {
                  .quiz-popup {
                    max-width: 98vw !important;
                    padding: 18px 4px !important;
                  }
                }
              `}</style>
            </div>
          </div>
        </>
      )}
    </div>
  );
};

const CourseDashboard = ({ userId, isAdmin }) => {
  const [courses, setCourses] = useState([]);
  const [selectedCourse, setSelectedCourse] = useState(null);
  const [loading, setLoading] = useState(true);
  const [enrolledCounts, setEnrolledCounts] = useState({});

  useEffect(() => {
    // All users (including admins) get only their assigned courses
    const apiUrl = `/api/assigned_courses/${userId}`;
    axios.get(apiUrl).then(res => {
      setCourses(res.data.courses || []);
      setLoading(false);
    });
  }, [userId]);

  useEffect(() => {
    if (courses.length > 0) {
      axios.get('/api/enrolled_course_count').then(res => {
        // API returns: { enrolled_counts: [ { course_id, num_users }, ... ] }
        const counts = {};
        (res.data.enrolled_counts || []).forEach(item => {
          counts[item.course_id] = item.num_users;
        });
        setEnrolledCounts(counts);
      }).catch(() => {
        setEnrolledCounts({});
      });
    }
  }, [courses]);

  if (loading) return <div style={{ padding: 32 }}>Loading courses...</div>;

  if (selectedCourse) {
    return <CourseReader course={selectedCourse} onBack={() => setSelectedCourse(null)} userId={userId} />;
  }

  return (
    <div style={{ padding: 24, maxWidth: 1200, margin: '0 auto' }}>
      <h2 style={{ color: '#1976d2', marginBottom: 24 }}>Courses</h2>
      <div style={{ display: 'flex', flexWrap: 'wrap', gap: 24 }}>
        {courses.length === 0 ? (
          <div>No courses available.</div>
        ) : (
          courses.map(course => (
            <CourseCard
              key={course.course_id}
              course={course}
              onSelect={setSelectedCourse}
              enrolledCount={enrolledCounts[course.course_id]}
            />
          ))
        )}
      </div>
      <style>{`
        @media (max-width: 900px) {
          .course-card {
            min-width: 0 !important;
            max-width: 100vw !important;
            width: 100vw !important;
            font-size: 1rem !important;
            padding: 12px !important;
          }
          .course-reader {
            padding: 8px !important;
            max-width: 100vw !important;
            margin: 0 !important;
          }
          .course-reader h2, .course-reader h3, .course-reader h4 {
            font-size: 1.2rem !important;
            margin-bottom: 8px !important;
          }
          .course-reader img {
            width: 100% !important;
            max-width: 100vw !important;
            height: auto !important;
            margin-bottom: 12px !important;
          }
          .course-reader button {
            font-size: 1rem !important;
            padding: 10px 12px !important;
            min-width: 80px !important;
            margin-bottom: 8px !important;
          }
          .chapter-progress {
            font-size: 0.95rem !important;
            margin-bottom: 8px !important;
          }
          .chapter-btn {
            min-width: 36px !important;
            height: 36px !important;
            font-size: 1rem !important;
          }
        }
      `}</style>
    </div>
  );
};

export default CourseDashboard;

