import React, { useEffect, useState } from 'react';
import axios from 'axios';

const CourseAssignment = () => {
  const [courses, setCourses] = useState([]);
  const [users, setUsers] = useState([]);
  const [courseAssignments, setCourseAssignments] = useState({});
  const [loading, setLoading] = useState(false);
  const [feedback, setFeedback] = useState('');
  // For per-course user search input
  const [userSearches, setUserSearches] = useState({});
  const [openCourse, setOpenCourse] = useState(null);
  const [search, setSearch] = useState('');


  useEffect(() => {
    fetchCourses();
    fetchUsers();
  }, []);

  useEffect(() => {
    if (openCourse) {
      fetchAssignedUsers(openCourse);
    }
  }, [openCourse]);

  // Fetch assigned users for each course using the correct API
  const fetchAssignedUsers = async (courseId) => {
    try {
      const res = await fetch(`/api/assign_course/${courseId}`);
      if (!res.ok) return [];
      const data = await res.json();
      setCourseAssignments(prev => ({ ...prev, [courseId]: data.assigned_users || [] }));
    } catch (e) {
      // ignore
    }
  };

  const fetchCourses = async () => {
    try {
      const res = await axios.get('/api/courses');
      setCourses(res.data.courses || []);
    } catch (error) {
      console.error('Error fetching courses:', error);
    }
  };

  const fetchUsers = async () => {
    try {
      const res = await axios.get('/api/users');
      setUsers(res.data || []);
    } catch (error) {
      console.error('Error fetching users:', error);
    }
  };

  const handleAssign = async (userId, courseId) => {
    setLoading(true);
    try {
      await axios.post('/api/assign_course', { course_id: courseId, user_ids: [userId] });
      await fetchCourses();
      setFeedback('User assigned to course.');
    } catch (error) {
      console.error('Error assigning user:', error);
      setFeedback('Error assigning user.');
    }
    setLoading(false);
  };

            const handleRemove = async (userId, courseId) => {
              setLoading(true);
              await axios.delete(`/api/assign_course/${courseId}/${userId}`);
              await fetchCourses();
              setFeedback('User removed from course.');
              setLoading(false);
            };

            return (
              <div className="container py-4">
                <h2>Course Assignment</h2>
                {feedback && <div className="alert alert-success">{feedback}</div>}
                <div className="mb-4">
                  <div style={{marginBottom: 18, display: 'flex', alignItems: 'center', gap: 12}}>
                    <input
                      type="text"
                      className="form-control"
                      style={{maxWidth: 320, display: 'inline-block'}}
                      placeholder="Search courses..."
                      value={search}
                      onChange={e => setSearch(e.target.value)}
                    />
                  </div>
                  <table className="table table-bordered">
                    <thead>
                      <tr>
                        <th>Course</th>
                        <th style={{width: 60}}></th>
                      </tr>
                    </thead>
                    <tbody>
                      {courses.length === 0 ? (
                        <tr><td colSpan="2">No courses available.</td></tr>
                      ) : courses
                          .filter(course => {
                            const name = (course.course_name || course.title || course.name || '').toLowerCase();
                            return name.includes(search.toLowerCase());
                          })
                          .map(course => {
                        const assigned = courseAssignments[course.course_id] || [];
                        const assignedIds = assigned.map(u => u.id_user || u.id);
                        const isOpen = openCourse === course.course_id;
                        return (
                          <React.Fragment key={course.course_id}>
                            <tr>
                              <td style={{verticalAlign:'middle', fontWeight:'bold', cursor:'pointer'}} onClick={() => setOpenCourse(isOpen ? null : course.course_id)}>
                                <span style={{marginRight:8}}>{isOpen ? '▼' : '▶'}</span>
                                {course.course_name || course.title || course.name || `Course #${course.course_id}`}
                              </td>
                              <td style={{verticalAlign:'middle'}}>
                                <button className="btn btn-link btn-sm" onClick={() => setOpenCourse(isOpen ? null : course.course_id)}>
                                  {isOpen ? 'Hide' : 'Show'}
                                </button>
                              </td>
                            </tr>
                            {isOpen && (
                              <tr>
                                <td colSpan={2} style={{background:'#f9f9f9'}}>
                                  <div style={{display:'flex', gap:32, flexWrap:'wrap'}}>
                                    <div style={{minWidth:220, flex:1}}>
                                      <h5>Assigned Users</h5>
                                      {assigned.length === 0 ? <span className="text-muted">No users assigned.</span> : (
                                        <ul style={{paddingLeft:18}}>
                                          {assigned.map(u => (
                                            <li key={u.id_user || u.id} style={{marginBottom:4}}>
                                              <b>{u.id_user || u.id}</b> {u.prenom ? u.prenom : ''} {u.nom ? u.nom : ''}
                                              <button className="btn btn-danger btn-sm ms-2" style={{marginLeft:8}} onClick={() => handleRemove(u.id_user || u.id, course.course_id)} disabled={loading}>Remove</button>
                                            </li>
                                          ))}
                                        </ul>
                                      )}
                                    </div>
                                    <div style={{minWidth:320, flex:2}}>
                                      <h5>Assign New User</h5>
                                      {/* User search input per course */}
                                      <input
                                        type="text"
                                        className="form-control mb-2"
                                        style={{maxWidth: 320}}
                                        placeholder="Search users..."
                                        value={userSearches[course.course_id] || ''}
                                        onChange={e => {
                                          const val = e.target.value;
                                          setUserSearches(prev => ({ ...prev, [course.course_id]: val }));
                                        }}
                                      />
                                      <table className="table table-sm mb-0">
                                        <thead>
                                          <tr>
                                            <th>ID</th>
                                            <th>Name</th>
                                            <th>Email</th>
                                            <th>Action</th>
                                          </tr>
                                        </thead>
                                        <tbody>
                                          {users
                                            .filter(u => !assignedIds.includes(u.id_user))
                                            .filter(u => {
                                              const searchVal = userSearches[course.course_id] || '';
                                              const name = (`${u.prenom || ''} ${u.nom || ''}`.trim() + ' ' + (u.email || '') + ' ' + (u.id_user || '')).toLowerCase();
                                              return name.includes(searchVal.toLowerCase());
                                            })
                                            .map(u => (
                                              <tr key={u.id_user}>
                                                <td>{u.id_user}</td>
                                                <td>{`${u.prenom || ''} ${u.nom || ''}`.trim() || u.email || 'Unknown'}</td>
                                                <td>{u.email}</td>
                                                <td>
                                                  <button className="btn btn-primary btn-sm" onClick={() => handleAssign(u.id_user, course.course_id)} disabled={loading}>Assign</button>
                                                </td>
                                              </tr>
                                            ))}
                                        </tbody>
                                      </table>
                                    </div>
                                  </div>
                                </td>
                              </tr>
                            )}
                          </React.Fragment>
                        );
                      })}
                    </tbody>
                  </table>
                </div>
              </div>
            );
};

export default CourseAssignment;
