import React, { useEffect, useState } from 'react';
import { Box, Button, Tab, Tabs, Typography, TextField, Dialog, DialogTitle, DialogContent, DialogActions, IconButton, Chip, CircularProgress, Radio, RadioGroup, FormControlLabel, FormControl, FormLabel, Select, MenuItem } from '@mui/material';
import { Edit, Delete, ContentCopy, Add } from '@mui/icons-material';
import axios from 'axios';
import RichTextEditor from './RichTextEditor';

const API_URL = '/api/announcements';

function AnnouncementManager({ currentUser }) {
  const [tab, setTab] = useState(0);
  const [announcements, setAnnouncements] = useState([]);
  const [loading, setLoading] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [editAnnouncement, setEditAnnouncement] = useState(null);
  const [form, setForm] = useState({
    title: '',
    content: '',
    target_roles: '',
    target_users: '',
    announcement_type: '',
    priority: '',
    expiration_date: ''
  });
  const [targetType, setTargetType] = useState('public');
  const [roles, setRoles] = useState([]);
  const [traces, setTraces] = useState([]);

  useEffect(() => {
    fetchAnnouncements();
    fetchTraces();
    fetchRoles();
  }, []);

  const fetchRoles = async () => {
    try {
  const res = await axios.get('/api/roles/list');
      setRoles(res.data.roles || []);
    } catch (err) {}
  };
  const fetchTraces = async () => {
    try {
      const res = await axios.get('/api/trace');
      setTraces(res.data.traces || []);
    } catch (err) {}
  };

  const fetchAnnouncements = async () => {
    setLoading(true);
    try {
      const res = await axios.get(API_URL);
      setAnnouncements(res.data.announcements || []);
    } catch (err) {
      // handle error
    }
    setLoading(false);
  };

  const handleOpenDialog = (announcement = null) => {
    setEditAnnouncement(announcement);
    setForm(announcement ? {
      title: announcement.title,
      content: announcement.content,
      target_roles: announcement.target_roles || '',
      target_users: announcement.target_users || '',
      announcement_type: announcement.announcement_type || '',
      priority: announcement.priority || '',
      expiration_date: announcement.expiration_date || ''
    } : {
      title: '',
      content: '',
      target_roles: '',
      target_users: '',
      announcement_type: '',
      priority: '',
      expiration_date: ''
    });
    if (announcement) {
      if (announcement.target_roles && announcement.target_roles !== '') {
        setTargetType('role');
      } else if (announcement.target_users && announcement.target_users !== '') {
        setTargetType('user');
      } else {
        setTargetType('public');
      }
    } else {
      setTargetType('public');
    }
    setDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setDialogOpen(false);
    setEditAnnouncement(null);
    setForm({
      title: '',
      content: '',
      target_roles: '',
      target_users: '',
      announcement_type: '',
      priority: '',
      expiration_date: ''
    });
  };

  const handleChange = (e) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleEditorChange = (content) => {
    setForm({ ...form, content });
  };

  const handleSave = async () => {
    setLoading(true);
    let payload = { ...form, target_type: targetType };
    if (targetType === 'public') {
      payload.target_roles = '';
      payload.target_users = '';
    } else if (targetType === 'role') {
      payload.target_users = '';
    } else if (targetType === 'user') {
      payload.target_roles = '';
    }
    try {
      if (editAnnouncement) {
        await axios.put(`${API_URL}/${editAnnouncement.id}`, {
          ...payload,
          updated_by: currentUser?.id,
          actor_email: currentUser?.email
        });
      } else {
        await axios.post(API_URL, {
          ...payload,
          created_by: currentUser?.id,
          actor_email: currentUser?.email
        });
      }
      fetchAnnouncements();
      handleCloseDialog();
    } catch (err) {
      // handle error
    }
    setLoading(false);
  };

  const handleDelete = async (id) => {
    setLoading(true);
    try {
      await axios.delete(`${API_URL}/${id}`, {
        data: { deleted_by: currentUser?.id, actor_email: currentUser?.email }
      });
      fetchAnnouncements();
    } catch (err) {
      // handle error
    }
    setLoading(false);
  };

  const handleClone = async (id) => {
    setLoading(true);
    try {
      await axios.post(`${API_URL}/${id}/clone`, {
        cloned_by: currentUser?.id,
        actor_email: currentUser?.email
      });
      fetchAnnouncements();
    } catch (err) {
      // handle error
    }
    setLoading(false);
  };

  // Announcement type color mapping
  const announcementTypeColors = {
    Info: '#1976d2',
    Warning: '#ffa726',
    Alert: '#e53935',
    Update: '#43cea2',
    'Payment Delay': '#fbc02d',
    'User Supension': '#8e24aa',
    Payroll: '#27ae60',
    Account: '#a259e6',
    User: '#90caf9',
  };

  return (
    <Box sx={{ p: 2 }}>
      <Typography variant="h5" gutterBottom>Announcement Management</Typography>
      <Button variant="contained" startIcon={<Add />} onClick={() => handleOpenDialog()} sx={{ mb: 2 }}>Add Announcement</Button>
      {loading ? <CircularProgress /> : (
        <Box>
          {(() => {
            // Filter announcements for Administrator
            const adminId = currentUser?.id;
            const adminEmail = currentUser?.email;
            const adminRole = currentUser?.role || 'Administrator';
            const filtered = announcements.filter(a => {
              // Public: both target_roles and target_users empty
              if (!a.target_roles && !a.target_users) return true;
              // Role: target_roles includes adminRole
              if (a.target_roles && a.target_roles.split(',').includes(adminRole)) return true;
              // User: target_users includes adminId or adminEmail
              if (a.target_users && (a.target_users.split(',').includes(String(adminId)) || a.target_users.split(',').includes(adminEmail))) return true;
              return false;
            });
            return filtered.length === 0 ? <Typography>No announcements found.</Typography> : (
              filtered.map(a => (
                <Box key={a.id} sx={{ mb: 2, p: 2, border: '1px solid #eee', borderRadius: 2, background: '#fff' }}>
                  <div style={{marginBottom:'8px'}}>
                    <Typography variant="h6" sx={{ display: 'inline-block', fontWeight: 'bold', fontSize: '1.25rem' }}>{a.title}</Typography>
                    {a.announcement_type && (
                      <span style={{
                        background: announcementTypeColors[a.announcement_type] || '#1976d2',
                        color: '#fff',
                        borderRadius: '6px',
                        padding: '2px 10px',
                        marginLeft: '12px',
                        fontSize: '0.95rem',
                        fontWeight: 'bold',
                        verticalAlign: 'middle',
                        display: 'inline-block',
                      }}>{a.announcement_type}</span>
                    )}
                  </div>
                  <div dangerouslySetInnerHTML={{ __html: a.content }} style={{ marginBottom: 8 }} />
                  <Box sx={{ mb: 1 }}>
                    {a.expiration_date && (() => {
                      // Expiration severity logic
                      const today = new Date();
                      const expDate = new Date(a.expiration_date);
                      today.setHours(0,0,0,0);
                      expDate.setHours(0,0,0,0);
                      const diffDays = Math.ceil((expDate - today) / (1000 * 60 * 60 * 24));
                      let chipColor = 'default';
                      if (diffDays < 0 || diffDays === 0 || diffDays === 1 || diffDays === 2 || (diffDays >= 3 && diffDays <= 5)) {
                        chipColor = 'error'; // red
                      }
                      return <Chip label={`Expires: ${a.expiration_date}`} size="small" color={chipColor} sx={{ mr: 1 }} />;
                    })()}
                    {a.target_roles && a.target_roles.split(',').map(role => (
                      <Chip key={role} label={`Role: ${role}`} size="small" sx={{ mr: 1, background:'#e3f2fd', color:'#1976d2' }} />
                    ))}
                    {a.target_users && a.target_users.split(',').map(user => (
                      <Chip key={user} label={`User: ${user}`} size="small" color="info" sx={{ mr: 1 }} />
                    ))}
                    {a.created_by && (
                      <Chip label={`Created by: ${a.created_by}`} size="small" sx={{ mr: 1, background:'#f5f5f5', color:'#333' }} />
                    )}
                    {a.updated_by && (
                      <Chip label={`Edited by: ${a.updated_by}`} size="small" sx={{ mr: 1, background:'#f5f5f5', color:'#333' }} />
                    )}
                  </Box>
                  <Box>
                    <IconButton onClick={() => handleOpenDialog(a)}><Edit /></IconButton>
                    <IconButton onClick={() => handleDelete(a.id)}><Delete /></IconButton>
                    <IconButton onClick={() => handleClone(a.id)}><ContentCopy /></IconButton>
                  </Box>
                </Box>
              ))
            );
          })()}
        </Box>
      )}
      <Dialog open={dialogOpen} onClose={handleCloseDialog} maxWidth="sm" fullWidth>
        <DialogTitle>{editAnnouncement ? 'Edit Announcement' : 'Add Announcement'}</DialogTitle>
        <DialogContent>
          <TextField label="Title" name="title" value={form.title} onChange={handleChange} fullWidth sx={{ mb: 2 }} />
          <FormControl fullWidth sx={{ mb: 2 }}>
            <FormLabel>Announcement Type</FormLabel>
            <Select
              name="announcement_type"
              value={form.announcement_type}
              onChange={handleChange}
            >
              <MenuItem value="">Select Type</MenuItem>
              <MenuItem value="Info">Info</MenuItem>
              <MenuItem value="Warning">Warning</MenuItem>
              <MenuItem value="Alert">Alert</MenuItem>
              <MenuItem value="Update">Update</MenuItem>
              <MenuItem value="Payment Delay">Payment Delay</MenuItem>
              <MenuItem value="User Supension">User Supension</MenuItem>
              <MenuItem value="Payroll">Payroll</MenuItem>
              <MenuItem value="Account">Account</MenuItem>
              <MenuItem value="User">User</MenuItem>
            </Select>
          </FormControl>
          <FormControl fullWidth sx={{ mb: 2 }}>
            <FormLabel>Priority</FormLabel>
            <Select
              name="priority"
              value={form.priority}
              onChange={handleChange}
            >
              <MenuItem value="">Select Priority</MenuItem>
              <MenuItem value="Low">Low</MenuItem>
              <MenuItem value="Medium">Medium</MenuItem>
              <MenuItem value="High">High</MenuItem>
              <MenuItem value="Critical">Critical</MenuItem>
            </Select>
          </FormControl>
          <TextField
            label="End Date (Optional)"
            name="expiration_date"
            type="date"
            value={form.expiration_date}
            onChange={handleChange}
            fullWidth
            sx={{ mb: 2 }}
            InputLabelProps={{ shrink: true }}
          />
          <RichTextEditor value={form.content} onChange={handleEditorChange} />
          <FormControl sx={{ mt: 2, mb: 2 }} component="fieldset">
            <FormLabel component="legend">Target Audience</FormLabel>
            <RadioGroup row value={targetType} onChange={e => setTargetType(e.target.value)}>
              <FormControlLabel value="public" control={<Radio />} label="All Users (Public)" />
              <FormControlLabel value="role" control={<Radio />} label="Specific Role" />
              <FormControlLabel value="user" control={<Radio />} label="Specific User" />
            </RadioGroup>
            {targetType === 'role' && (
              <Select
                label="Select Role"
                value={form.target_roles}
                onChange={e => setForm({ ...form, target_roles: e.target.value })}
                fullWidth
                sx={{ mt: 2 }}
              >
                {roles.map(role => (
                  <MenuItem key={role} value={role}>{role}</MenuItem>
                ))}
              </Select>
            )}
            {targetType === 'user' && (
              <TextField
                label="Target User (ID or Email)"
                name="target_users"
                value={form.target_users}
                onChange={handleChange}
                fullWidth
                sx={{ mt: 2 }}
              />
            )}
          </FormControl>
          {/* Live Preview */}
          <Box sx={{ mt: 2, p: 2, border: '1px solid #eee', borderRadius: 2, background: '#f9f9fc' }}>
            <Typography variant="subtitle1" sx={{ mb: 1 }}>Live Preview</Typography>
            <Typography variant="h6">{form.title || 'Announcement Title'}</Typography>
            <div dangerouslySetInnerHTML={{ __html: form.content || '<i>Your announcement content will appear here.</i>' }} style={{ marginBottom: 8 }} />
            <Typography variant="body2" sx={{ color: '#1976d2', mb: 1 }}>
              {targetType === 'public' && 'Visible to: All Users'}
              {targetType === 'role' && `Visible to: ${form.target_roles || 'Select a role'}`}
              {targetType === 'user' && `Visible to: ${form.target_users || 'Enter user ID or email'}`}
            </Typography>
          </Box>
        </DialogContent>
        <DialogActions>
          <Button onClick={handleCloseDialog}>Cancel</Button>
          <Button onClick={handleSave} variant="contained">Save</Button>
        </DialogActions>
      </Dialog>
      <Box sx={{ mt: 4 }}>
        <Typography variant="h6">Recent Announcement Actions</Typography>
        {traces.length === 0 ? <Typography>No recent actions.</Typography> : (
          <Box>
            {traces.filter(t => t.action_type && t.action_type.match(/announcement|create|update|delete|clone/)).slice(0, 10).map((t, idx) => (
              <Box key={idx} sx={{ mb: 1, p: 1, border: '1px solid #eee', borderRadius: 1 }}>
                <Typography variant="body2"><b>{t.action_type}</b> by {t.actor_email || t.actor_id} on {t.timestamp}</Typography>
                <Typography variant="caption">{t.action_details}</Typography>
              </Box>
            ))}
          </Box>
        )}
      </Box>
    </Box>
  );
}

export default AnnouncementManager;
