import React, { useEffect, useState } from 'react';

const AddTeacherModal = ({ open, onClose, onSave, initialData }) => {
  const [users, setUsers] = useState([]);
  const [schools, setSchools] = useState([]);
  const [userId, setUserId] = useState(initialData ? initialData.user_id : '');
  const [schoolType, setSchoolType] = useState(initialData ? initialData.school_type : 'both');
  const [subject, setSubject] = useState(initialData ? initialData.subject : '');
  const [bio, setBio] = useState(initialData ? initialData.bio : '');
  const [visible, setVisible] = useState(initialData && typeof initialData.visible !== 'undefined' ? initialData.visible : 1);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  useEffect(() => {
    if (open) {
      setLoading(true);
      Promise.all([
        fetch('/api/users').then(res => res.json()),
        fetch('/api/schools').then(res => res.json())
      ])
        .then(([usersData, schoolsData]) => {
          setUsers(usersData.filter(u => u.id_user));
          setSchools(schoolsData.schools || []);
          setLoading(false);
        })
        .catch(() => {
          setError('Failed to fetch users or schools');
          setLoading(false);
        });
      // Reset fields for add/edit
      setUserId(initialData ? initialData.user_id : '');
      setSchoolType(initialData ? initialData.school_type : 'both');
      setSubject(initialData ? initialData.subject : '');
      setBio(initialData ? initialData.bio : '');
      setVisible(initialData && typeof initialData.visible !== 'undefined' ? initialData.visible : 1);
    }
  }, [open, initialData]);

  const handleSubmit = (e) => {
    e.preventDefault();
    if (!userId) {
      setError('Please select a user');
      return;
    }
    if (!schoolType) {
      setError('Please select a school type');
      return;
    }
    setError(null);
    // For edit, pass id and email if present
    if (initialData) {
      onSave({
        id: initialData.id,
        user_id: userId,
        subject,
        bio,
        school_type: schoolType,
        visible: Number(visible),
        email: initialData.email
      });
    } else {
      onSave({ user_id: userId, subject, bio, school_type: schoolType, visible: Number(visible) });
    }
  };

  if (!open) return null;

  return (
    <div style={{position:'fixed', top:0, left:0, width:'100vw', height:'100vh', background:'rgba(0,0,0,0.18)', zIndex:9999, display:'flex', alignItems:'center', justifyContent:'center'}}>
      <div style={{background:'#fff', borderRadius:'16px', padding:'32px', minWidth:340, boxShadow:'0 4px 24px rgba(0,0,0,0.18)'}}>
        <h3 style={{marginBottom:'18px'}}>{initialData ? 'Edit Teacher' : 'Add Teacher'}</h3>
        {error && <div style={{color:'red', marginBottom:8}}>{error}</div>}
        <form onSubmit={handleSubmit}>
          <div style={{marginBottom:'14px'}}>
            <label style={{fontWeight:'bold'}}>Select User</label>
            <select value={userId} onChange={e=>setUserId(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} required disabled={!!initialData}>
              <option value=''>Select user</option>
              {users.map(u => (
                <option key={u.id_user} value={u.id_user}>{u.prenom} {u.nom} ({u.email})</option>
              ))}
            </select>
          </div>
          <div style={{marginBottom:'14px'}}>
            <label style={{fontWeight:'bold'}}>School Type</label>
            <select value={schoolType} onChange={e=>setSchoolType(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} required>
              <option value='both'>Both</option>
              {schools.map(s => (
                <option key={s.type} value={s.type}>{s.name} ({s.type})</option>
              ))}
            </select>
          </div>
          <div style={{marginBottom:'14px'}}>
            <label style={{fontWeight:'bold'}}>Subject</label>
            <input value={subject} onChange={e=>setSubject(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} placeholder='Subject' required />
          </div>
          <div style={{marginBottom:'18px'}}>
            <label style={{fontWeight:'bold'}}>Bio</label>
            <textarea value={bio} onChange={e=>setBio(e.target.value)} style={{width:'100%', padding:'8px', borderRadius:'6px', border:'1px solid #ccc'}} placeholder='Short bio' rows={3} />
          </div>
          <div style={{marginBottom:'18px', display:'flex', alignItems:'center', gap:8}}>
            <input id="visible-checkbox" type="checkbox" checked={!!visible} onChange={e=>setVisible(e.target.checked ? 1 : 0)} />
            <label htmlFor="visible-checkbox" style={{fontWeight:'bold', margin:0}}>Visible</label>
          </div>
          <div style={{display:'flex', justifyContent:'flex-end', gap:'12px'}}>
            <button type='button' onClick={onClose} style={{background:'#eaf6ff', color:'#1565c0', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Cancel</button>
            <button type='submit' style={{background:'#1976d2', color:'#fff', border:'none', borderRadius:'6px', padding:'8px 18px', fontWeight:'bold', fontSize:'1rem', cursor:'pointer'}}>Save</button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default AddTeacherModal;
